import QtQuick 1.0
import "components"
import "scripts/videolistscripts.js" as Scripts
import "scripts/createobject.js" as ObjectCreator
import "scripts/youtube.js" as YT

Page {
    id: root

    property variant playlist: []
    property string videoFeed

    function setPlaylist(playlistItem) {
        playlist = playlistItem;
        videoFeed = "http://gdata.youtube.com/feeds/api/playlists/" + playlist.playlistId + "?v=2&max-results=30&alt=json&safeSearch=none";
        YT.getYouTubeVideos(videoFeed);
    }

    function showWatchLaterPlaylist() {
        playlist = { "name": qsTr("Watch Later"), "description": qsTr("Your YouTube 'Watch Later' playlist"), "author": YouTube.username };
        videoFeed = _WATCH_LATER_FEED;
        YT.getYouTubeVideos(videoFeed);
    }

    function setPlaylistVideos(playlistItem) {
        /* This function is used to pass the videos of
          a fully loaded ListModel */

        playlist = playlistItem.info;
        for (var i = 0; i < playlistItem.videos.length; i++) {
            videoListModel.append(playlistItem.videos[i]);
        }
        videoListModel.loading = false;
        videoList.positionViewAtIndex(0, ListView.Beginning);
    }

    function playPlaylist() {
        var list = [];
        for (var i = 0; i < videoListModel.count; i++) {
            list.push(ObjectCreator.cloneObject(videoListModel.get(i)));
        }
        playVideos(list);
    }

    function goToVideoInfo(video) {
        pageStack.push(Qt.resolvedUrl("YTInfoPage.qml"));
        pageStack.currentPage.setVideo(video);
    }

    function goToUserProfile() {
        pageStack.push(Qt.resolvedUrl("YTUserPage.qml"));
        pageStack.currentPage.getUserProfile(playlist.author);
    }

    function selectAll() {
        Scripts.selectAll()
    }

    function selectNone() {
        Scripts.selectNone();
    }

    function addVideosToFavourites() {
        YT.addVideosToFavourites();
    }

    function showPlaylistDialog() {
        YT.showPlaylistDialog();
    }

    function deleteVideosFromPlaylist() {
        toggleBusy(true);
        var list = [];
        for (var i = 0; i < videoList.checkList.length; i++) {
            list.push(videoListModel.get(videoList.checkList[i]).id);
        }
        if (videoFeed == _WATCH_LATER_FEED) {
            YouTube.deleteFromWatchLaterPlaylist(list, playlist.playlistId);
        }
        else {
            YouTube.deleteFromPlaylist(list, playlist.playlistId);
        }
        videoList.checkList = [];
    }

    function addVideosToWatchLaterPlaylist() {
        YT.addVideosToWatchLaterPlaylist();
    }

    function addVideosToPlaybackQueue() {
        YT.addVideosToPlaybackQueue();
    }

    function addVideosToDownloadQueue(convertToAudio) {
        YT.addVideosToDownloadQueue(convertToAudio);
    }

    Connections {
        target: YouTube

        onAddedToPlaylist: if (id == playlist.playlistId) playlistVideosTimer.restart();
        onDeletedFromPlaylist: {
            if (id == playlist.playlistId) {
                messages.displayMessage(qsTr("Video(s) deleted from playlist"));
                playlistVideosTimer.restart();
            }
        }
        onAddedToWatchLaterPlaylist: if (videoFeed == _WATCH_LATER_FEED) playlistVideosTimer.restart();
        onDeletedFromWatchLaterPlaylist: {
            if (videoFeed == _WATCH_LATER_FEED) {
                messages.displayMessage(qsTr("Video(s) deleted from playlist"));
                playlistVideosTimer.restart();
            }
        }
    }

    Connections {
        target: videoListModel
        onLoadingChanged: {
            if (!((videoListModel.loading) || (videoListModel.error)) &&
                    (videoListModel.totalResults > 30) &&
                    (videoListModel.totalResults > videoListModel.count)) {
                YT.getYouTubeVideos(videoFeed);
            }
        }
    }

    Timer {
        id: playlistVideosTimer

        interval: 3000
        onTriggered: videoListModel.reset()
    }

    title: qsTr("Playlist")
    tools: Menu {

        menuItems: [

            MenuItem {
                text: videoList.checkList.length === 0 ? qsTr("Select all") : qsTr("Select none")
                enabled: videoList.count > 0
                onClicked: videoList.checkList.length === 0 ? selectAll() : selectNone()
                closeMenuWhenClicked: false
            },

            MenuItem {
                text: qsTr("Add to favourites")
                onClicked: addVideosToFavourites()
                visible: YouTube.userSignedIn
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Add to another playlist")
                onClicked: showPlaylistDialog()
                visible: YouTube.userSignedIn
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Delete from playlist")
                onClicked: deleteVideosFromPlaylist()
                visible: (YouTube.userSignedIn) && (playlist.author == YouTube.username) ? true : false
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Watch later")
                onClicked: addVideosToWatchLaterPlaylist()
                visible: (YouTube.userSignedIn) && (videoFeed != _WATCH_LATER_FEED)
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Play")
                visible: Settings.mediaPlayer == "cutetubePlayer"
                onClicked: addVideosToPlaybackQueue()
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Download")
                onClicked: addVideosToDownloadQueue(false)
                enabled: videoList.checkList.length > 0
            },

            MenuItem {
                text: qsTr("Download (audio)")
                onClicked: addVideosToDownloadQueue(true)
                enabled: videoList.checkList.length > 0
            }
        ]
    }

    Flickable {
        id: flicker

        anchors { fill: parent; topMargin: titleArea.height }
        contentWidth: parent.width
        contentHeight: 350 + descriptionText.height + col2.height
        flickableDirection: Flickable.VerticalFlick
        clip: true

        Label {
            id: titleText

            anchors { top: parent.top; left: parent.left; right: parent.right; margins: 10 }
            
            font.bold: true
            
            wrapMode: Text.WordWrap
            text: !playlist.name ? "" : playlist.name
        }

        Image {
            id: cover

            anchors { left: titleText.left; top: titleText.bottom; topMargin: 10 }
            width: 160
            height: 120
            source: (videoListModel.count > 0) ? videoListModel.get(0).largeThumbnail : !theme.inverted ? "images/video.png" : "images/video-white.png"
            sourceSize.width: width
            sourceSize.height: height
            smooth: true
            onStatusChanged: if (status == Image.Error) source = !theme.inverted ? "images/video.png" : "images/video-white.png";
        }

        Button {
            id: playButton

            anchors { right: parent.right; top: cover.bottom; margins: 10 }
            text: qsTr("Play all")
            visible: Settings.mediaPlayer == "cutetubePlayer"
            enabled: (videoListModel.count > 0) && (!videoListModel.loading)
            onClicked: playPlaylist()
        }

        Column {
            id: col1

            anchors { left: cover.right; leftMargin: 10; top: cover.top; right: parent.right; rightMargin: 10 }

            Row {
                width: parent.width
                visible: playlist.author ? true : false

                Label {
                    id: authorLabel

                    
                    
                    text: qsTr("Created by")
                }

                Label {
                    width: parent.width - authorLabel.width - 10
                    
                    color: (!playlist.author) || (playlist.author == YouTube.username) ? _TEXT_COLOR : Settings.activeColor
                    elide: Text.ElideRight
                    text: !playlist.author ? "" : " " + playlist.author
                    opacity: authorMouseArea.pressed ? 0.5 : 1

                    MouseArea {
                        id: authorMouseArea

                        anchors.fill: parent
                        onClicked: goToUserProfile()
                        enabled: (!playlist.author) || (playlist.author == YouTube.username) ? false : true
                    }
                }
            }


            Label {
                width: parent.width
                
                
                elide: Text.ElideRight
                text: !playlist.createdDate ?  "" : qsTr("Created on") + " " + playlist.createdDate.split("T")[0]
                visible: !(text == "")
            }

            Label {
                width: parent.width
                
                
                elide: Text.ElideRight
                text: !playlist.updatedDate ?  "" : qsTr("Updated on") + " " + playlist.updatedDate.split("T")[0]
                visible: !(text == "")
            }
        }

        SeparatorLabel {
            id: separator

            anchors { left: parent.left; right: parent.right; top: playButton.visible ? playButton.bottom : cover.bottom; margins: 10 }
            text: qsTr("Description")
        }

        Label {
            id: descriptionText

            anchors { left: parent.left; right: parent.right; top: separator.bottom; margins: 10  }
            
            
            wrapMode: Text.WordWrap
            text: (!playlist.description) || (playlist.description === "") ? qsTr("No description") : playlist.description
            visible: !(text == "")
        }

        SeparatorLabel {
            id: separator2

            anchors { left: parent.left; right: parent.right; top: descriptionText.bottom; margins: 10  }
            text: videoListModel.loading ? qsTr("Videos") : videoListModel.count + " " + qsTr("Videos")
        }

        Column {
            id: col2

            anchors { left: parent.left; right: parent.right; top: separator2.bottom; margins: 10  }

            Repeater {
                id: videoList

                property variant checkList : []

                model: ListModel {
                    id: videoListModel

                    property bool loading : true
                    property bool error: false
                    property int totalResults
                    property int page : 0

                    function reset() {
                        videoListModel.clear();
                        videoListModel.loading = false;
                        videoListModel.error = false;
                        videoListModel.totalResults = 0;
                        videoListModel.page = 0;
                        YT.getYouTubeVideos(videoFeed);
                    }
                }

                YTListDelegate {
                    id: delegate

                    x: -10
                    width: col2.width + 20

                    function addOrRemoveFromCheckList() {
                        var cl = videoList.checkList;
                        if (!delegate.checked) {
                            cl.push(index);
                        }
                        else {
                            for (var i = 0; i < cl.length; i++) {
                                if (cl[i] == index) {
                                    cl.splice(i, 1);
                                }
                            }
                        }
                        videoList.checkList = cl;
                    }

                    checked: Scripts.indexInCheckList(index)
                    onClicked: {
                        if (videoList.checkList.length === 0) {
                            goToVideoInfo(ObjectCreator.cloneObject(videoListModel.get(index)));
                        }
                        else {
                            addOrRemoveFromCheckList();
                        }
                    }
                    onPressAndHold: addOrRemoveFromCheckList()
                    onPlayClicked: {
                        if (videoList.checkList.length === 0) {
                            playVideos([ObjectCreator.cloneObject(videoListModel.get(index))]);
                        }
                    }
                }
            }

            Item {
                width: parent.width
                height: 100
                visible: videoListModel.loading

                BusyDialog {
                    anchors.centerIn: parent
                    visible: parent.visible
                }
            }

            Label {
                id: noResultsText

                width: parent.width
                font.pixelSize: _LARGE_FONT_SIZE
                font.bold: true
                color: "#4d4d4d"
                horizontalAlignment: Text.AlignHCenter
                verticalAlignment: Text.AlignVCenter
                text: qsTr("No videos found")
                visible: (!videoListModel.loading) && (videoListModel.count == 0)
            }
        }
    }
}
