/*******************************************************************************
**
** vksettingssource.cpp - implement VkSettingsSource, gui configurator
**                        of settings for source function of Vkontakte service.
**                        Inherited by QMailMessageServiceEditor.
** This file is part of the QMF-Vkontakte plug-in.
**
** Copyright (C) 2010 PetrSU. Pavel Shiryaev <shiryaev AT cs.karelia.ru>
** Contact: QMF-vkontakte team (qmf-vkontakte AT cs.karelia.ru)
**
** QMF-Vkontakte plug-in is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** QMF-Vkontakte plug-in is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with QMF-Vkontakte plug-in; if not, write to the Free Software
** Foundation, Inc., 51 Franklin St, Fifth Floor,
** Boston, MA  02110-1301  USA
**
*******************************************************************************/

#include "vksettingssource.h"
#include "vkconfiguration.h"
#include <QLineEdit>
#include <QMessageBox>
#include <qmailaccount.h>
#include <qmailaccountconfiguration.h>
#include <qmailtransport.h>


namespace {

const QString serviceKey("vk");

class PortValidator : public QValidator
{
public:
    PortValidator(QWidget *parent = 0, const char *name = 0);

    QValidator::State validate(QString &str, int &) const;
};

PortValidator::PortValidator(QWidget *parent, const char *name)
    : QValidator(parent) 
{
    setObjectName(name);
}

QValidator::State PortValidator::validate(QString &str, int &) const
{
    // allow empty strings, as it's a bit awkward to edit otherwise
    if ( str.isEmpty() )
        return QValidator::Acceptable;

    bool ok = false;
    int i = str.toInt(&ok);
    if ( !ok )
        return QValidator::Invalid;

    if ( i <= 0 || i >= 65536 )
        return QValidator::Invalid;

    return QValidator::Acceptable;
}

}


VkSettingsSource::VkSettingsSource()
    : QMailMessageServiceEditor(),
      warningEmitted(false)
{
    setupUi(this);
    setLayoutDirection(qApp->layoutDirection());
    connect(intervalCheckBox, SIGNAL(stateChanged(int)),
            this, SLOT(intervalCheckChanged(int)));
    connect(manualProxyButton, SIGNAL(toggled(bool)),
            this, SLOT(proxyStateChanged(bool)));

    const QString uncapitalised("email noautocapitalization");

    // These fields should not be autocapitalised

    proxyPortInput->setValidator(new PortValidator(this));
    mailPasswInput->setEchoMode(QLineEdit::Password);
}

void VkSettingsSource::intervalCheckChanged(int enabled)
{
    intervalPeriod->setEnabled(enabled);
}

void VkSettingsSource::proxyStateChanged(bool enabled)
{
    proxyPortInput->setEnabled(enabled);
    proxyServerInput->setEnabled(enabled);
}

void VkSettingsSource::displayConfiguration(const QMailAccount &account,
                                      const QMailAccountConfiguration &config)
{
    proxyStateChanged(false);
    deleteCheckBox->setEnabled(false);
    if (!config.services().contains(serviceKey)) {
        // New account
        mailUserInput->setText("");
        mailPasswInput->setText("");
        proxyServerInput->setText("");
        proxyPortInput->setText("3128");
        intervalCheckBox->setChecked(false);
        systemProxyButton->setChecked(true);
        intervalPeriod->setEnabled(false);
    } else {
        VkConfiguration vkConfig(config);
        switch (vkConfig.proxyState()) {
        case VkConfiguration::SystemProxy:
            systemProxyButton->setChecked(true);
            break;
        case VkConfiguration::ManualProxy:
            manualProxyButton->setChecked(true);
            break;
        case VkConfiguration::NoProxy:
        default:
            noProxyButton->setChecked(true);
            break;
        }

        mailUserInput->setText(vkConfig.vkUserName());
        mailPasswInput->setText(vkConfig.vkPassword());
        proxyServerInput->setText(vkConfig.proxyServer());
        proxyPortInput->setText(QString::number(vkConfig.proxyPort()));
        deleteCheckBox->setChecked(vkConfig.canDeleteMail());
        intervalCheckBox->setChecked(vkConfig.checkInterval() > 0);
        intervalPeriod->setValue(qAbs(vkConfig.checkInterval() ));
        messagesPeriod->setValue(qAbs(vkConfig.messagesInterval() ));

        if (QMailStore::instance()->
                    countFolders(QMailFolderKey::parentAccountId(account.id())) > 0)
        {
            inboxFolderId = account.standardFolder(QMailFolder::InboxFolder);
            sentFolderId = account.standardFolder(QMailFolder::SentFolder);
            wallFolderId = account.standardFolder(QMailFolder::JunkFolder);
        }

        inboxFolderInput->setText(inboxFolderId.isValid() ? QMailFolder(inboxFolderId).path() : "");
        inboxFolderInput->setEnabled(inboxFolderId.isValid());

        sentFolderInput->setText(sentFolderId.isValid() ? QMailFolder(sentFolderId).path() : "");
        sentFolderInput->setEnabled(sentFolderId.isValid());

        wallFolderInput->setText(wallFolderId.isValid() ? QMailFolder(wallFolderId).path() : "");
        wallFolderInput->setEnabled(wallFolderId.isValid());
    }
}

bool VkSettingsSource::updateAccount(QMailAccount *account,
                               QMailAccountConfiguration *config)
{
    bool result;
    int port = proxyPortInput->text().toInt(&result);
    if ( (!result) ) {
        // should only happen when the string is empty, since we use a validator
        port = -1;
    }

    if (!config->services().contains(serviceKey))
        config->addServiceConfiguration(serviceKey);

    VkConfigurationEditor vkConfig(config);

    vkConfig.setVersion(101);

    vkConfig.setMailUserName(mailUserInput->text());
    vkConfig.setMailPassword(mailPasswInput->text());
    vkConfig.setProxyServer(proxyServerInput->text());
    vkConfig.setProxyPort(port == -1 ? 3128 : port);
    vkConfig.setDeleteMail(deleteCheckBox->isChecked());
    vkConfig.setAutoDownload(false );
    vkConfig.setCheckInterval(
            intervalPeriod->value() * (intervalCheckBox->isChecked() ? 1 : -1));
    vkConfig.setMessagesInterval(messagesPeriod->value());
    if (manualProxyButton->isChecked())
        vkConfig.setProxyState(VkConfiguration::ManualProxy);
    else if (systemProxyButton->isChecked())
        vkConfig.setProxyState(VkConfiguration::SystemProxy);
    else
        vkConfig.setProxyState(VkConfiguration::NoProxy);

    if (inboxFolderId.isValid()) {
        QMailFolder folder(inboxFolderId);
        folder.setDisplayName(inboxFolderInput->text());
        QMailStore::instance()->updateFolder(&folder);
    }

    if (sentFolderId.isValid()) {
        QMailFolder folder(sentFolderId);
        folder.setDisplayName(sentFolderInput->text());
        QMailStore::instance()->updateFolder(&folder);
    }

    if (wallFolderId.isValid()) {
        QMailFolder folder(wallFolderId);
        folder.setDisplayName(wallFolderInput->text());
        QMailStore::instance()->updateFolder(&folder);
    }

    // We cannot create new folders. Only Inbox, Sent and Wall
    account->setStatus(QMailAccount::CanCreateFolders, false);

    // Do we have a configuration we can use?
    if (!vkConfig.vkUserName().isEmpty())
        account->setStatus(QMailAccount::CanRetrieve, true);

    return true;
}
