/*
 * QCPUFreq - a simple cpufreq GUI
 * Copyright (C) 2010 Daniel Klaffenbach <daniel.klaffenbach@cs.tu-chemnitz.de>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QFile>
#include <QMessageBox>
#include <QTextStream>
#include <QProcess>
#include <QDesktopWidget>
#if defined(Q_WS_MAEMO_5)
    #include <QMaemo5InformationBox>
#endif


#define APPNAME "QCPUFreq"
#define APPVERSION "0.2"

MainWindow::MainWindow(QWidget *parent) :
    QMainWindow(parent),
    ui(new Ui::MainWindow)
{
    ui->setupUi(this);

    refresh();

    // enable auto rotation
    setAutoRotaion();

    //create a QGraphicsScene for the little chip icon
    scene = new QGraphicsScene();
    orientationChanged();

    //connect signals and slots
    connect( ui->actionAbout, SIGNAL(triggered()), this, SLOT(about()) );
    connect( ui->freq_adjust, SIGNAL(valueChanged(int)), this, SLOT(adjustFreq()) );
    connect(QApplication::desktop(), SIGNAL(resized(int)), this, SLOT(orientationChanged()));
    connect( ui->sr_btn, SIGNAL(clicked()), this, SLOT(setSmartReflex()) );

}

MainWindow::~MainWindow()
{
    delete ui;
    delete scene;
}


/**
  * SLOT: Displays an about box
  */
void MainWindow::about()
{
    QMessageBox::about(this, APPNAME " " APPVERSION, "<p style=\"align:center;\">&copy; 2010 Daniel Klaffenbach</p>" );
    refresh();
}


/**
  * SLOT: Adjusts the maximum CPU frequency according to the scaler
  */
void MainWindow::adjustFreq()
{
    int newmax = getScalingFreq( ui->freq_adjust->sliderPosition() );
    QString max;
    max.setNum( newmax );
    QStringList arguments;
#if defined(Q_WS_MAEMO_5)
    //on Maemo5 the set_scalingmaxfreq-Script is not in $PATH
    arguments.append( "/opt/usr/bin/set_scalingmaxfreq" );
#else
    arguments.append( "set_scalingmaxfreq" );
#endif
    arguments.append( max );

    //execute the scaling script
    QProcess script;
    script.execute( "sudo", arguments );

    refresh();
}


/**
  * Returns the current CPU temperature
  */
QString MainWindow::getCPUTemp()
{
#if defined(Q_WS_MAEMO_5)
    return readSysFile( "devices/platform/omap34xx_temp/temp1_input_raw" );
#endif
    return tr( "Unknown" );
}


/**
  * Returns the maximum CPU frequency
  */
int MainWindow::getMaxFreq()
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_max_freq" );
    return tmp.toInt();
}


/**
  * Returns the minimum CPU frequency
  */
int MainWindow::getMinFreq()
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_min_freq" );
    return tmp.toInt();
}


/**
  * Returns the CPU frequency for the specified scaling step
  */
int MainWindow::getScalingFreq(int step)
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies" );
    QStringList freqs = tmp.split( " " );
    step = step - 1;
    if ( step < 0 )
         step = 0;
    if ( step > getScalingSteps() )
        step = getScalingSteps();

    tmp = freqs[ step ];
    return tmp.toInt();
}


/**
  * Returns the name of the current CPU frequency scaling governor
  *
  * \return	name of governor
  */
QString MainWindow::getScalingGovernor()
{
    return readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_governor" );
}


/**
  * Returns the amount of available scaling steps.
  */
int MainWindow::getScalingSteps()
{
    QString tmp = readSysFile( "devices/system/cpu/cpu0/cpufreq/scaling_available_frequencies" );
    QStringList freqs = tmp.split( " " );
    return (freqs.size() - 1);
}


/**
  * Returns the scaling step for the specified frequency.
  */
int MainWindow::getScalingStep( int freq )
{
    for( int i = 1; i <= getScalingSteps(); ++i ) {
           if ( getScalingFreq(i) == freq )
                return i;
    }

    return 1;
}


/**
  * Returns the SmartReflex(tm) state
  *
  * \return	0|1
  */
int MainWindow::getSmartReflexState()
{
//SmartReflex is only supprted on Maemo5
#if defined(Q_WS_MAEMO_5)
    QString tmp = readSysFile( "power/sr_vdd1_autocomp" );

    if ( tmp == "1" )
	return 1;
    else
	return 0;
#else
    return 0;
#endif
}


/**
  * Reads any file in /sys/
  *
  * \param	sys_file : full path to sys file - omit "/sys/"
  * \return	content of sys file
  */
QString MainWindow::readSysFile(QString sys_file)
{
    QFile file( "/sys/"+sys_file );

    //open the file
    if ( !file.exists() || !file.open( QIODevice::ReadOnly ) ) {
        QMessageBox::critical(this, tr("QCPUFreq"), tr("Could not get information from /sys!"));
        return "";
    }

    //read the file
    QTextStream in( &file );
    QString txt = in.readLine();

    return txt;
}


/**
  * Refreshes all of the values to display
  */
void MainWindow::refresh()
{
    //get the current frequency and calculate the MHz value
    int freq = ( getMinFreq() / 1000 );
    QString display;
    display.setNum( freq );
    display.append( " MHz" );
    ui->freq_min->setText( display );

    //do the same thing for the maximum frequency
    freq = ( getMaxFreq() / 1000 );
    display.setNum( freq );
    display.append( " MHz" );
    ui->freq_max->setText( display );

    //display the current governor
    ui->freq_governor->setText( getScalingGovernor() );

    //display current temperature
    ui->cpu_temp->setText( getCPUTemp() );

    //smart reflex button
    if ( getSmartReflexState() == 1 ) {
	ui->sr_btn->setDown( true );
	ui->sr_btn->setText( tr( "Enabled" ) );
    } else {
	ui->sr_btn->setDown( false );
	ui->sr_btn->setText( tr( "Disabled" ) );
    }


    //display frequency slider
    ui->freq_adjust->setMinimum( 1 );
    ui->freq_adjust->setMaximum( getScalingSteps() );
    ui->freq_adjust->setInvertedAppearance( true );
    ui->freq_adjust->setSliderPosition( getScalingStep(getMaxFreq()) );

    //ui->retranslateUi(this);
}


/**
  * Repaints part of the GUI after the device was rotated
  */
void MainWindow::orientationChanged()
{
    QPixmap image;

    //check whether we are using portrait or landscape mode
    if ( usePortrait() ) {
	//in portrait mode we want to display the large image
	image.load( ":/img/chip256" );
	this->scene->clear();
	this->scene->addPixmap(  image  );

	ui->graphicsPortrait->setScene( this->scene );
	ui->graphicsPortrait->setMaximumSize( 256, 256 );
	ui->graphicsLandscape->setMaximumSize( 0, 0 );
    } else {
	image.load( ":/img/chip128" );
	this->scene->clear();
	this->scene->addPixmap(  image  );

	ui->graphicsLandscape->setScene( this->scene );
	ui->graphicsLandscape->setMaximumSize( 128, 128 );
	ui->graphicsPortrait->setMaximumSize( 0, 0 );
    }
}


/**
  * Enables the auto-rotation feature of Maemo5 devices
  */
void MainWindow::setAutoRotaion()
{
#if defined(Q_WS_MAEMO_5)
    setAttribute(Qt::WA_Maemo5AutoOrientation, true);
#endif
}


/**
  * SLOT: Enables or disables Smart Reflex(tm) after pressing sr_btn
  */
void MainWindow::setSmartReflex()
{
//SmartReflex is only supprted on Maemo5
#if defined(Q_WS_MAEMO_5)
    QStringList arguments;
    arguments.append( "/opt/usr/bin/set_sr" );

    if ( getSmartReflexState() == 1 )
	arguments.append( "off" );
    else {
	QMaemo5InformationBox::information(this, tr( "SmartReflex support is known to be unstable on some devices and may cause random reboots." ), QMaemo5InformationBox::DefaultTimeout);
	arguments.append( "on" );
    }

    //execute the sr script script
    QProcess script;
    script.execute( "sudo", arguments );

#endif
    //refresh the UI
    refresh();
}


/**
  * Returns true when the device is in portrait mode
  */
bool MainWindow::usePortrait()
{
    QRect screenGeometry = QApplication::desktop()->screenGeometry();
    if (screenGeometry.width() > screenGeometry.height())
	return false;
    else
	return true;
}
