/*
 * This file is part of PySide: Python for Qt
 *
 * Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
 *
 * Contact: PySide team <contact@pyside.org>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * version 2.1 as published by the Free Software Foundation.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */


//workaround to access protected functions
#define protected public

// default includes
#include <shiboken.h>
#include <typeresolver.h>
#include <typeinfo>
#include "pyside_qtscript_python.h"

#include "qscriptengineagent_wrapper.h"

// Extra includes
#include <qscriptengine.h>
#include <qscriptengineagent.h>
#include <qscriptvalue.h>
#include <qstring.h>
#include <qvariant.h>

using namespace Shiboken;

// Native ---------------------------------------------------------

QScriptEngineAgentWrapper::QScriptEngineAgentWrapper(QScriptEngine * engine) : QScriptEngineAgent(engine) {
    // ... middle
}

void QScriptEngineAgentWrapper::contextPop()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contextPop"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::contextPop();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::contextPush()
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "contextPush"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::contextPush();
    }

    Shiboken::AutoDecRef pyargs(PyTuple_New(0));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::exceptionCatch(qint64 scriptId, const QScriptValue & exception)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "exceptionCatch"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::exceptionCatch(scriptId, exception);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(LS)",
        Shiboken::Converter<long long >::toPython(scriptId),
        Shiboken::Converter<QScriptValue & >::toPython(exception)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::exceptionThrow(qint64 scriptId, const QScriptValue & exception, bool hasHandler)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "exceptionThrow"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::exceptionThrow(scriptId, exception, hasHandler);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(LSS)",
        Shiboken::Converter<long long >::toPython(scriptId),
        Shiboken::Converter<QScriptValue & >::toPython(exception),
        Shiboken::Converter<bool >::toPython(hasHandler)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

QVariant QScriptEngineAgentWrapper::extension(QScriptEngineAgent::Extension extension, const QVariant & argument)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "extension"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::extension(extension, argument);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(SS)",
        Shiboken::Converter<QScriptEngineAgent::Extension >::toPython(extension),
        Shiboken::Converter<QVariant & >::toPython(argument)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return QVariant();
    }
    // Check return type
    bool typeIsValid = Shiboken::Converter<QVariant >::isConvertible(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QScriptEngineAgent.extension", SbkType<QVariant >()->tp_name, py_result->ob_type->tp_name);
        return QVariant();
    }
    QVariant cpp_result(Shiboken::Converter<QVariant >::toCpp(py_result));
    return cpp_result;
}

void QScriptEngineAgentWrapper::functionEntry(qint64 scriptId)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "functionEntry"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::functionEntry(scriptId);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(L)",
        Shiboken::Converter<long long >::toPython(scriptId)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::functionExit(qint64 scriptId, const QScriptValue & returnValue)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "functionExit"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::functionExit(scriptId, returnValue);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(LS)",
        Shiboken::Converter<long long >::toPython(scriptId),
        Shiboken::Converter<QScriptValue & >::toPython(returnValue)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::positionChange(qint64 scriptId, int lineNumber, int columnNumber)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "positionChange"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::positionChange(scriptId, lineNumber, columnNumber);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(Lii)",
        Shiboken::Converter<long long >::toPython(scriptId),
        lineNumber,
        columnNumber
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::scriptLoad(qint64 id, const QString & program, const QString & fileName, int baseLineNumber)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "scriptLoad"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::scriptLoad(id, program, fileName, baseLineNumber);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(LSSi)",
        Shiboken::Converter<long long >::toPython(id),
        Shiboken::Converter<QString & >::toPython(program),
        Shiboken::Converter<QString & >::toPython(fileName),
        baseLineNumber
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

void QScriptEngineAgentWrapper::scriptUnload(qint64 id)
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "scriptUnload"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::scriptUnload(id);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(L)",
        Shiboken::Converter<long long >::toPython(id)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
}

bool QScriptEngineAgentWrapper::supportsExtension(QScriptEngineAgent::Extension extension) const
{
    Shiboken::GilState gil;
    Shiboken::AutoDecRef py_override(BindingManager::instance().getOverride(this, "supportsExtension"));
    if (py_override.isNull()) {
        return this->QScriptEngineAgent::supportsExtension(extension);
    }

    Shiboken::AutoDecRef pyargs(Py_BuildValue("(S)",
        Shiboken::Converter<QScriptEngineAgent::Extension >::toPython(extension)
    ));

    Shiboken::AutoDecRef py_result(PyObject_Call(py_override, pyargs, NULL));
    // An error happened in python code!
    if (py_result.isNull()) {
        PyErr_Print();
        return bool(0);
    }
    // Check return type
    bool typeIsValid = PyBool_Check(py_result);
    if (!typeIsValid) {
        PyErr_Format(PyExc_TypeError, "Invalid return value in function %s, expected %s, got %s.", "QScriptEngineAgent.supportsExtension", SbkType<bool >()->tp_name, py_result->ob_type->tp_name);
        return bool(0);
    }
    bool cpp_result(Shiboken::Converter<bool >::toCpp(py_result));
    return cpp_result;
}

QScriptEngineAgentWrapper::~QScriptEngineAgentWrapper()
{
    BindingManager::instance().invalidateWrapper(this);
}

// Target ---------------------------------------------------------

static int
SbkQScriptEngineAgent_Init(PyObject* self, PyObject* args, PyObject* kwds)
{
    QScriptEngineAgentWrapper* cptr = 0;
    SbkBaseWrapper* sbkSelf = reinterpret_cast<SbkBaseWrapper*>(self);

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0};

    if (!PyArg_UnpackTuple(args, "QScriptEngineAgent", 1, 1, &(pyargs[0])))
        return -1;

    if (numArgs == 1 && Shiboken::Converter<QScriptEngine * >::isConvertible(pyargs[0])) {
        if (Shiboken::cppObjectIsInvalid(pyargs[0]))
            return -1;
        QScriptEngine* cpp_arg0 = Shiboken::Converter<QScriptEngine* >::toCpp(pyargs[0]);
        // QScriptEngineAgent(QScriptEngine*)
        cptr = new QScriptEngineAgentWrapper(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgent_Init_TypeError;

    if (PyErr_Occurred() || !Shiboken::setCppPointer(sbkSelf, Shiboken::SbkType<QScriptEngineAgent >(), cptr)) {
        delete cptr;
        return -1;
    }
    if (!cptr) goto SbkQScriptEngineAgent_Init_TypeError;

    sbkSelf->validCppObject = 1;
    sbkSelf->containsCppWrapper = 1;
    BindingManager::instance().registerWrapper(sbkSelf, cptr);

    return 1;

    SbkQScriptEngineAgent_Init_TypeError:
        const char* overloads[] = {"PySide.QtScript.QScriptEngine", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent", overloads);
        return -1;
}

static PyObject* SbkQScriptEngineAgentFunc_contextPop(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // contextPop()
    QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
    cppSelf->QScriptEngineAgent::contextPop();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQScriptEngineAgentFunc_contextPush(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    // contextPush()
    QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
    cppSelf->QScriptEngineAgent::contextPush();
    //CppGenerator::writeParentChildManagement

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;
}

static PyObject* SbkQScriptEngineAgentFunc_engine(PyObject* self)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    // engine()const
    QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
    QScriptEngine * cpp_result = cppSelf->QScriptEngineAgent::engine();
    py_result = Shiboken::Converter<QScriptEngine * >::toPython(cpp_result);
    //CppGenerator::writeParentChildManagement
    Shiboken::setParent(self, py_result);

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;
}

static PyObject* SbkQScriptEngineAgentFunc_exceptionCatch(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "exceptionCatch", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && Shiboken::Converter<QScriptValue & >::isConvertible(pyargs[1])) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QScriptValue > cpp_arg1_auto_ptr;
        QScriptValue* cpp_arg1 = Shiboken::Converter<QScriptValue* >::toCpp(pyargs[1]);
        if (!SbkQScriptValue_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QScriptValue >(cpp_arg1);
        // exceptionCatch(qint64,QScriptValue)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::exceptionCatch(cpp_arg0, *cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_exceptionCatch_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_exceptionCatch_TypeError:
        const char* overloads[] = {"long long, PySide.QtScript.QScriptValue", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.exceptionCatch", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_exceptionThrow(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "exceptionThrow", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && PyNumber_Check(pyargs[0]) && Shiboken::Converter<QScriptValue & >::isConvertible(pyargs[1]) && PyNumber_Check(pyargs[2])) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QScriptValue > cpp_arg1_auto_ptr;
        QScriptValue* cpp_arg1 = Shiboken::Converter<QScriptValue* >::toCpp(pyargs[1]);
        if (!SbkQScriptValue_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QScriptValue >(cpp_arg1);
        bool cpp_arg2 = Shiboken::Converter<bool >::toCpp(pyargs[2]);
        // exceptionThrow(qint64,QScriptValue,bool)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::exceptionThrow(cpp_arg0, *cpp_arg1, cpp_arg2);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_exceptionThrow_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_exceptionThrow_TypeError:
        const char* overloads[] = {"long long, PySide.QtScript.QScriptValue, bool", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.exceptionThrow", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_extension(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "extension", 1, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (Shiboken::Converter<QScriptEngineAgent::Extension >::isConvertible(pyargs[0])) {
        QScriptEngineAgent::Extension cpp_arg0 = Shiboken::Converter<QScriptEngineAgent::Extension >::toCpp(pyargs[0]);
        if (numArgs == 1) {
            // extension(QScriptEngineAgent::Extension,QVariant)
            QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
            QVariant cpp_result = cppSelf->QScriptEngineAgent::extension(cpp_arg0);
            py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else if (Shiboken::Converter<QVariant & >::isConvertible(pyargs[1])) {
            if (Shiboken::cppObjectIsInvalid(pyargs[1]))
                return 0;
            std::auto_ptr<QVariant > cpp_arg1_auto_ptr;
            QVariant* cpp_arg1 = Shiboken::Converter<QVariant* >::toCpp(pyargs[1]);
            if (!SbkQVariant_Check(pyargs[1]))
                cpp_arg1_auto_ptr = std::auto_ptr<QVariant >(cpp_arg1);
            // extension(QScriptEngineAgent::Extension,QVariant)
            QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
            QVariant cpp_result = cppSelf->QScriptEngineAgent::extension(cpp_arg0, *cpp_arg1);
            py_result = Shiboken::Converter<QVariant >::toPython(cpp_result);
            //CppGenerator::writeParentChildManagement
        } else goto SbkQScriptEngineAgentFunc_extension_TypeError;
    } else goto SbkQScriptEngineAgentFunc_extension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQScriptEngineAgentFunc_extension_TypeError:
        const char* overloads[] = {"PySide.QtScript.QScriptEngineAgent.Extension, PySide.QtCore.QVariant = QVariant()", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.extension", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_functionEntry(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(arg);
        // functionEntry(qint64)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::functionEntry(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_functionEntry_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_functionEntry_TypeError:
        const char* overloads[] = {"long long", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtScript.QScriptEngineAgent.functionEntry", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_functionExit(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0};

    if (!PyArg_UnpackTuple(args, "functionExit", 2, 2, &(pyargs[0]), &(pyargs[1])))
        return 0;

    if (numArgs == 2 && PyNumber_Check(pyargs[0]) && Shiboken::Converter<QScriptValue & >::isConvertible(pyargs[1])) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QScriptValue > cpp_arg1_auto_ptr;
        QScriptValue* cpp_arg1 = Shiboken::Converter<QScriptValue* >::toCpp(pyargs[1]);
        if (!SbkQScriptValue_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QScriptValue >(cpp_arg1);
        // functionExit(qint64,QScriptValue)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::functionExit(cpp_arg0, *cpp_arg1);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_functionExit_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_functionExit_TypeError:
        const char* overloads[] = {"long long, PySide.QtScript.QScriptValue", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.functionExit", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_positionChange(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0};

    if (!PyArg_UnpackTuple(args, "positionChange", 3, 3, &(pyargs[0]), &(pyargs[1]), &(pyargs[2])))
        return 0;

    if (numArgs == 3 && PyNumber_Check(pyargs[0]) && PyNumber_Check(pyargs[1]) && PyNumber_Check(pyargs[2])) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(pyargs[0]);
        int cpp_arg1 = Shiboken::Converter<int >::toCpp(pyargs[1]);
        int cpp_arg2 = Shiboken::Converter<int >::toCpp(pyargs[2]);
        // positionChange(qint64,int,int)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::positionChange(cpp_arg0, cpp_arg1, cpp_arg2);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_positionChange_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_positionChange_TypeError:
        const char* overloads[] = {"long long, int, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.positionChange", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_scriptLoad(PyObject* self, PyObject* args)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    int numArgs = PyTuple_GET_SIZE(args);
    PyObject* pyargs[] = {0, 0, 0, 0};

    if (!PyArg_UnpackTuple(args, "scriptLoad", 4, 4, &(pyargs[0]), &(pyargs[1]), &(pyargs[2]), &(pyargs[3])))
        return 0;

    if (numArgs == 4 && PyNumber_Check(pyargs[0]) && Shiboken::Converter<QString & >::isConvertible(pyargs[1]) && Shiboken::Converter<QString & >::isConvertible(pyargs[2]) && PyNumber_Check(pyargs[3])) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(pyargs[0]);
        if (Shiboken::cppObjectIsInvalid(pyargs[1]))
            return 0;
        std::auto_ptr<QString > cpp_arg1_auto_ptr;
        QString* cpp_arg1 = Shiboken::Converter<QString* >::toCpp(pyargs[1]);
        if (!SbkQString_Check(pyargs[1]))
            cpp_arg1_auto_ptr = std::auto_ptr<QString >(cpp_arg1);
        if (Shiboken::cppObjectIsInvalid(pyargs[2]))
            return 0;
        std::auto_ptr<QString > cpp_arg2_auto_ptr;
        QString* cpp_arg2 = Shiboken::Converter<QString* >::toCpp(pyargs[2]);
        if (!SbkQString_Check(pyargs[2]))
            cpp_arg2_auto_ptr = std::auto_ptr<QString >(cpp_arg2);
        int cpp_arg3 = Shiboken::Converter<int >::toCpp(pyargs[3]);
        // scriptLoad(qint64,QString,QString,int)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::scriptLoad(cpp_arg0, *cpp_arg1, *cpp_arg2, cpp_arg3);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_scriptLoad_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_scriptLoad_TypeError:
        const char* overloads[] = {"long long, PySide.QtCore.QString, PySide.QtCore.QString, int", 0};
        Shiboken::setErrorAboutWrongArguments(args, "PySide.QtScript.QScriptEngineAgent.scriptLoad", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_scriptUnload(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;


    if (PyNumber_Check(arg)) {
        qint64 cpp_arg0 = Shiboken::Converter<qint64 >::toCpp(arg);
        // scriptUnload(qint64)
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        cppSelf->QScriptEngineAgent::scriptUnload(cpp_arg0);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_scriptUnload_TypeError;

    if (PyErr_Occurred()) {
        return 0;
    }
    Py_RETURN_NONE;

    SbkQScriptEngineAgentFunc_scriptUnload_TypeError:
        const char* overloads[] = {"long long", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtScript.QScriptEngineAgent.scriptUnload", overloads);
        return 0;
}

static PyObject* SbkQScriptEngineAgentFunc_supportsExtension(PyObject* self, PyObject* arg)
{
    if (Shiboken::cppObjectIsInvalid(self))
        return 0;

    PyObject* py_result = 0;

    if (Shiboken::Converter<QScriptEngineAgent::Extension >::isConvertible(arg)) {
        QScriptEngineAgent::Extension cpp_arg0 = Shiboken::Converter<QScriptEngineAgent::Extension >::toCpp(arg);
        // supportsExtension(QScriptEngineAgent::Extension)const
        QScriptEngineAgent* cppSelf = Shiboken::Converter<QScriptEngineAgent* >::toCpp((PyObject*)self);
        bool cpp_result = cppSelf->QScriptEngineAgent::supportsExtension(cpp_arg0);
        py_result = Shiboken::Converter<bool >::toPython(cpp_result);
        //CppGenerator::writeParentChildManagement
    } else goto SbkQScriptEngineAgentFunc_supportsExtension_TypeError;

    if (PyErr_Occurred() || !py_result) {
        Py_XDECREF(py_result);
        return 0;
    }
    return py_result;

    SbkQScriptEngineAgentFunc_supportsExtension_TypeError:
        const char* overloads[] = {"PySide.QtScript.QScriptEngineAgent.Extension", 0};
        Shiboken::setErrorAboutWrongArguments(arg, "PySide.QtScript.QScriptEngineAgent.supportsExtension", overloads);
        return 0;
}

static PyMethodDef SbkQScriptEngineAgent_methods[] = {
    {"contextPop", (PyCFunction)SbkQScriptEngineAgentFunc_contextPop, METH_NOARGS},
    {"contextPush", (PyCFunction)SbkQScriptEngineAgentFunc_contextPush, METH_NOARGS},
    {"engine", (PyCFunction)SbkQScriptEngineAgentFunc_engine, METH_NOARGS},
    {"exceptionCatch", (PyCFunction)SbkQScriptEngineAgentFunc_exceptionCatch, METH_VARARGS},
    {"exceptionThrow", (PyCFunction)SbkQScriptEngineAgentFunc_exceptionThrow, METH_VARARGS},
    {"extension", (PyCFunction)SbkQScriptEngineAgentFunc_extension, METH_VARARGS},
    {"functionEntry", (PyCFunction)SbkQScriptEngineAgentFunc_functionEntry, METH_O},
    {"functionExit", (PyCFunction)SbkQScriptEngineAgentFunc_functionExit, METH_VARARGS},
    {"positionChange", (PyCFunction)SbkQScriptEngineAgentFunc_positionChange, METH_VARARGS},
    {"scriptLoad", (PyCFunction)SbkQScriptEngineAgentFunc_scriptLoad, METH_VARARGS},
    {"scriptUnload", (PyCFunction)SbkQScriptEngineAgentFunc_scriptUnload, METH_O},
    {"supportsExtension", (PyCFunction)SbkQScriptEngineAgentFunc_supportsExtension, METH_O},
    {0} // Sentinel
};

extern "C"
{

// Class Definition -----------------------------------------------
static SbkBaseWrapperType SbkQScriptEngineAgent_Type = { { {
    PyObject_HEAD_INIT(&Shiboken::SbkBaseWrapperType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "PySide.QtScript.QScriptEngineAgent",
    /*tp_basicsize*/        sizeof(Shiboken::SbkBaseWrapper),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          &Shiboken::deallocWrapper,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             0,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              0,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT|Py_TPFLAGS_BASETYPE|Py_TPFLAGS_CHECKTYPES,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          SbkQScriptEngineAgent_methods,
    /*tp_members*/          0,
    /*tp_getset*/           0,
    /*tp_base*/             reinterpret_cast<PyTypeObject*>(&Shiboken::SbkBaseWrapper_Type),
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             SbkQScriptEngineAgent_Init,
    /*tp_alloc*/            0,
    /*tp_new*/              Shiboken::SbkBaseWrapper_TpNew,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
}, },
    /*mi_offsets*/          0,
    /*mi_init*/             0,
    /*mi_specialcast*/      0,
    /*type_discovery*/      0,
    /*obj_copier*/          0,
    /*ext_isconvertible*/   0,
    /*ext_tocpp*/           0,
    /*cpp_dtor*/            &Shiboken::callCppDestructor<QScriptEngineAgent >,
    /*is_multicpp*/         0,
    /*is_user_type*/        0
};

static SbkBaseWrapperType* SbkQScriptEngineAgent_typeDiscovery(void* cptr, SbkBaseWrapperType* instanceType)
{
    if (instanceType->mi_specialcast)
        cptr = instanceType->mi_specialcast(cptr, &SbkQScriptEngineAgent_Type);
    TypeResolver* typeResolver = TypeResolver::get(typeid(*reinterpret_cast<QScriptEngineAgent*>(cptr)).name());
    if (typeResolver)
        return reinterpret_cast<SbkBaseWrapperType*>(typeResolver->pythonType());
    return 0;
}

static PyGetSetDef SbkPySide_QtScript_QScriptEngineAgent_Extension_getsetlist[] = {
    {const_cast<char*>("name"), (getter)Shiboken::SbkEnumObject_name},
    {0}  // Sentinel
};

// forward declaration of new function
static PyObject* SbkPySide_QtScript_QScriptEngineAgent_Extension_New(PyTypeObject*, PyObject*, PyObject*);

static PyTypeObject SbkPySide_QtScript_QScriptEngineAgent_Extension_Type = {
    PyObject_HEAD_INIT(&Shiboken::SbkEnumType_Type)
    /*ob_size*/             0,
    /*tp_name*/             "Extension",
    /*tp_basicsize*/        sizeof(Shiboken::SbkEnumObject),
    /*tp_itemsize*/         0,
    /*tp_dealloc*/          0,
    /*tp_print*/            0,
    /*tp_getattr*/          0,
    /*tp_setattr*/          0,
    /*tp_compare*/          0,
    /*tp_repr*/             Shiboken::SbkEnumObject_repr,
    /*tp_as_number*/        0,
    /*tp_as_sequence*/      0,
    /*tp_as_mapping*/       0,
    /*tp_hash*/             0,
    /*tp_call*/             0,
    /*tp_str*/              Shiboken::SbkEnumObject_repr,
    /*tp_getattro*/         0,
    /*tp_setattro*/         0,
    /*tp_as_buffer*/        0,
    /*tp_flags*/            Py_TPFLAGS_DEFAULT,
    /*tp_doc*/              0,
    /*tp_traverse*/         0,
    /*tp_clear*/            0,
    /*tp_richcompare*/      0,
    /*tp_weaklistoffset*/   0,
    /*tp_iter*/             0,
    /*tp_iternext*/         0,
    /*tp_methods*/          0,
    /*tp_members*/          0,
    /*tp_getset*/           SbkPySide_QtScript_QScriptEngineAgent_Extension_getsetlist,
    /*tp_base*/             &PyInt_Type,
    /*tp_dict*/             0,
    /*tp_descr_get*/        0,
    /*tp_descr_set*/        0,
    /*tp_dictoffset*/       0,
    /*tp_init*/             0,
    /*tp_alloc*/            0,
    /*tp_new*/              SbkPySide_QtScript_QScriptEngineAgent_Extension_New,
    /*tp_free*/             0,
    /*tp_is_gc*/            0,
    /*tp_bases*/            0,
    /*tp_mro*/              0,
    /*tp_cache*/            0,
    /*tp_subclasses*/       0,
    /*tp_weaklist*/         0
};

static PyObject* SbkPySide_QtScript_QScriptEngineAgent_Extension_New(PyTypeObject* type, PyObject* args, PyObject* kwds)
{
    int item_value = 0;
    if (!PyArg_ParseTuple(args, "|i:__new__", &item_value))
        return 0;
    PyObject* self = Shiboken::SbkEnumObject_New(type, item_value);

    if (!self)
        return 0;
    return self;
}


PyAPI_FUNC(void) init_QScriptEngineAgent(PyObject* module)
{
    SbkPySide_QtScriptTypes[SBK_QSCRIPTENGINEAGENT_IDX] = reinterpret_cast<PyTypeObject*>(&SbkQScriptEngineAgent_Type);

    // Fill type discovery information
    SbkQScriptEngineAgent_Type.type_discovery = new Shiboken::TypeDiscovery;
    SbkQScriptEngineAgent_Type.type_discovery->addTypeDiscoveryFunction(&SbkQScriptEngineAgent_typeDiscovery);

    if (PyType_Ready((PyTypeObject*)&SbkQScriptEngineAgent_Type) < 0)
        return;

    Py_INCREF(reinterpret_cast<PyObject*>(&SbkQScriptEngineAgent_Type));
    PyModule_AddObject(module, "QScriptEngineAgent",
        ((PyObject*)&SbkQScriptEngineAgent_Type));

    // Initialize enums
    PyObject* enum_item;

    // init enum class: Extension
    SbkPySide_QtScriptTypes[SBK_QSCRIPTENGINEAGENT_EXTENSION_IDX] = &SbkPySide_QtScript_QScriptEngineAgent_Extension_Type;
    if (PyType_Ready((PyTypeObject*)&SbkPySide_QtScript_QScriptEngineAgent_Extension_Type) < 0)
        return;
    Py_INCREF(&SbkPySide_QtScript_QScriptEngineAgent_Extension_Type);
    PyDict_SetItemString(SbkQScriptEngineAgent_Type.super.ht_type.tp_dict,
            "Extension",((PyObject*)&SbkPySide_QtScript_QScriptEngineAgent_Extension_Type));

    enum_item = Shiboken::SbkEnumObject_New(&SbkPySide_QtScript_QScriptEngineAgent_Extension_Type,
        (long) QScriptEngineAgent::DebuggerInvocationRequest, "DebuggerInvocationRequest");
    PyDict_SetItemString(SbkQScriptEngineAgent_Type.super.ht_type.tp_dict,
        "DebuggerInvocationRequest", enum_item);
    PyDict_SetItemString(SbkPySide_QtScript_QScriptEngineAgent_Extension_Type.tp_dict,
        "DebuggerInvocationRequest", enum_item);
    Shiboken::TypeResolver::createValueTypeResolver<int>("QScriptEngineAgent::Extension");

    Shiboken::TypeResolver::createObjectTypeResolver<QScriptEngineAgent >("QScriptEngineAgent*");
    Shiboken::TypeResolver::createObjectTypeResolver<QScriptEngineAgent >(typeid(QScriptEngineAgent).name());
}


} // extern "C"

