/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
                     
#include <src/callbacks.h>
#include <src/interface.h>
#include <src/crypto.h>
#include <gtk/gtk.h>
#include <libintl.h>
/* strlen needed from string.h */
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dbus/dbus-glib.h>

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-app.h>
#endif
#include <gdk/gdkkeysyms.h>

/* Privates: */
void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview);
void maybe_save_file(MainView *mainview);
void save_mod_file(MainView *mainview);
char *get_confirmed_pass(MainView *mainview);

/* activate grid */
void callback_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, GtkTreeViewColumn *column, MainView *mainview) {
	callback_hildon_activate_treeView(treeView, path, mainview);
}

/* activate hildon grid */
void callback_hildon_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, MainView *mainview) {
	GtkTreeIter iter;
	rec *onrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	if ( !gtk_tree_model_get_iter(GTK_TREE_MODEL(mainview->store), &iter, path) ) {
		return;
	}
	gtk_tree_model_get(GTK_TREE_MODEL(mainview->store), &iter,
	    DATA_COLUMN, &onrec,
	    -1);
	/*
	onrec=g_object_get_data (G_OBJECT(&iter), "r");
	*/
	if (onrec!=NULL) {
		show_rec(mainview, onrec);
	} else if (gtk_tree_view_collapse_row(treeView, path)) {
		gtk_tree_view_expand_row(treeView, path, FALSE);
	} else {
		gtk_tree_view_collapse_row(treeView, path);
	}
}

gboolean callback_key_press(GtkWidget* widget, GdkEventKey* event,  MainView *mainview) {
	switch (event->keyval) {
		case GDK_F6: /* fullscreen */
			interface_fullscreen(mainview);
			return TRUE;
		case GDK_F7: /* button + */
			return TRUE;
		case GDK_F8: /* button - */
			return TRUE;
	}
	return FALSE;
}

void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview) {
/* scrub and delete oldstr, malloc and duplicate newstr, return pointer and tag file as changed... */
/* but only if the strings are different (and be careful of null old string pointers) */
	if ((*oldstr)==NULL) {
		if (newstr[0]!='\0') {
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	} else {
		if (strcmp((*oldstr),newstr)!=0) { /* check they differ before copying stuff around */
			trashMemory((*oldstr), strlen((*oldstr)));
			g_free((*oldstr));
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	}
}

const gchar *entryGetText(GtkWidget *widget) {
#if MAEMO_VERSION_MAJOR < 5
	return gtk_entry_get_text(GTK_ENTRY(widget));
#else
	return hildon_entry_get_text(HILDON_ENTRY(widget));
#endif
}

/* record detail events */
void callback_rec_response(GtkDialog *dialog, gint arg1, MainView *mainview) {
	RecWindow *recwindow = NULL;
	rec *rec=NULL;
	GtkTextBuffer *buffer=NULL;
	GtkTextIter start;
	GtkTextIter end;

	g_assert(mainview != NULL && mainview->data != NULL );
	
	recwindow = mainview->rec_win;
	rec=recwindow->rec;
	switch (arg1) {
		case 1: /* save any changes */
			safeReplaceString(&rec->title, entryGetText(recwindow->title), mainview);
			safeReplaceString(&rec->user, entryGetText(recwindow->username), mainview);
			safeReplaceString(&rec->password, entryGetText(recwindow->password), mainview);
#if MAEMO_VERSION_MAJOR < 5
			safeReplaceString(&rec->group, gtk_combo_box_get_active_text(GTK_COMBO_BOX(recwindow->group)), mainview);
			buffer=gtk_text_view_get_buffer (GTK_TEXT_VIEW(recwindow->notes));
#else
			safeReplaceString(&rec->group, hildon_button_get_value(HILDON_BUTTON(recwindow->group)), mainview);
			buffer=hildon_text_view_get_buffer(HILDON_TEXT_VIEW(recwindow->notes));
#endif
			if (gtk_text_buffer_get_modified(buffer)) { /* update notes if buffer has changed */
				gtk_text_buffer_get_start_iter (buffer, &start);
				gtk_text_buffer_get_end_iter (buffer, &end);
				safeReplaceString(&rec->notes, gtk_text_buffer_get_text (buffer, &start, &end, FALSE), mainview);
			}
			resort_rec(rec);
			unshow_rec(mainview);  /* close rec window */
			populate_treeView(mainview); /* redraw grid */
			break;
		case 2:
			gtk_clipboard_set_text(mainview->clipboard, rec->user, -1);
			break;
		case 3:
			gtk_clipboard_set_text(mainview->clipboard, rec->password, -1);
			break;
		case 4:
			if ((recwindow->editable) && (CONFRESP_YES==interface_yes_no_note(mainview, "Confirm Delete?"))) {
				unshow_rec(mainview);
				delete_rec(rec);
				populate_treeView(mainview); /* redraw grid */
			}
			break;
		case 5:
			interface_set_rec_editable(recwindow, TRUE);
			break;
		case -4:
			//Cancel fro Fremantle (out of dialog click)
			unshow_rec(mainview);  /* close rec window */
			populate_treeView(mainview); /* redraw grid */
			break;
	}
}

void setUrlTag(GtkTextBuffer *buffer, gint posStart, gint posEnd) {
	GtkTextIter start, end;
	gtk_text_buffer_get_iter_at_offset(buffer, &start, posStart);
	gtk_text_buffer_get_iter_at_offset(buffer, &end, posEnd);
	gtk_text_buffer_apply_tag_by_name(buffer, "url", &start, &end);

}

#if MAEMO_VERSION_MAJOR < 5
void enhanceBufferWithString(GtkTextBuffer *buffer, const gchar *search) {
	GtkTextIter start, end;
	gchar *text, *bout, *oneChar;
	gunichar uniChar;
	gint posStart, posEnd, index, listIdx;
	gchar **list, **searchList;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	text = gtk_text_buffer_get_slice(buffer, &start, &end, TRUE);
	if (text == NULL || text[0] == '\0') {
		return;
	}
	list = g_strsplit_set(text, " \t\r\n", -1);
	searchList = g_strsplit(search, " ", -1);
	index = 0;
	posStart = 0;
	while (list[index]) {
		posEnd = posStart + g_utf8_strlen(list[index], -1);
		if (posEnd > posStart) {
			bout = g_ascii_strdown(list[index], -1);
			listIdx = 0;
			while (searchList[listIdx]) {
				if (g_str_has_prefix(bout, searchList[listIdx])) {
					setUrlTag(buffer, posStart, posEnd);
					break;
				}
				listIdx ++;
			}
			g_free(bout);
		}
		posStart =  posEnd + 1;
		index++;
	}
	g_strfreev(list);
	g_strfreev(searchList);
}
#else
void enhanceBufferWithRegex(GtkTextBuffer *buffer, GRegex *regex) {
	GtkTextIter start, end;
	gchar *text;
	GMatchInfo *matchInfo;
	GError *error = NULL;
	gint posStart, posEnd;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	text = gtk_text_buffer_get_slice(buffer, &start, &end, TRUE);
	
	g_regex_match_full(regex, text, -1, 0, 0, &matchInfo, &error);
	while (g_match_info_matches(matchInfo)) {
		if (g_match_info_fetch_pos(matchInfo, 0, &posStart, &posEnd)) {
			setUrlTag(buffer, posStart, posEnd);
		}
		g_match_info_next(matchInfo, &error);
	}
	g_match_info_free(matchInfo);
	if (error != NULL) {
		g_printerr("Error while mathing: %s\n", error->message);
		g_error_free(error);
	}
}
#endif

void callback_buffer_changed(GtkTextBuffer *buffer, MainView *mainview) {
	GtkTextIter start, end;
	
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_remove_all_tags(buffer, &start, &end);

#if MAEMO_VERSION_MAJOR < 5
	enhanceBufferWithString(buffer, "http:// https:// www.");
#else
	enhanceBufferWithRegex(buffer, mainview->regexHttp);
	enhanceBufferWithRegex(buffer, mainview->regexWww);
#endif
}

static void launch_browser (gchar *url, MainView *mainView) {
  DBusGConnection *connection;
  GError *error = NULL;

  DBusGProxy *proxy;

  connection = dbus_g_bus_get (DBUS_BUS_SESSION, &error);
  if (connection == NULL) {
      msgbox(mainView, error->message);
      g_error_free (error);
      return;
    }

  proxy = dbus_g_proxy_new_for_name (connection,
				     "com.nokia.osso_browser",
				     "/com/nokia/osso_browser/request",
				     "com.nokia.osso_browser");

  error = NULL;
  if (!dbus_g_proxy_call (proxy, "load_url", &error,
			  G_TYPE_STRING, url,
			  G_TYPE_INVALID,
			  G_TYPE_INVALID)) {
      msgbox(mainView, error->message);
      g_error_free (error);
    }
}


gboolean callback_tag_event(GtkTextTag *tag, GObject *object, GdkEvent *event, GtkTextIter *iter, MainView *mainView) {
	GtkTextIter start, end;
	gchar *text;
	if (event->type == GDK_BUTTON_PRESS && !gtk_text_view_get_editable(GTK_TEXT_VIEW(object))) {
		start = *iter; 
		while (!gtk_text_iter_begins_tag(&start, tag)) {
			if (!gtk_text_iter_backward_char(&start)) {
				break;
			}
		}
		end = *iter; 
		while (!gtk_text_iter_ends_tag(&end, tag)) {
			if (!gtk_text_iter_forward_char(&end)) {
				break;
			}
		}
		text = gtk_text_iter_get_text(&start, &end);
		g_print("text associ au tag %s\n", text);
		launch_browser(text, mainView);
	}
	return FALSE;
}

/* Callback for hardware D-BUS events */
void hw_event_handler(osso_hw_state_t *state, MainView *mainview) {
 	g_assert(mainview != NULL && mainview->data != NULL );

	if ((state->shutdown_ind)	/* system shutdown */
			|| (state->system_inactivity_ind)) {	/* system sleeping */
        save_mod_file(mainview);
		close_file(mainview);
        return;
	}
	if (state->memory_low_ind) {	/* memory low... */
		unshow_rec(mainview);	/* close the rec window... changes not saved... */
	}
    save_mod_file(mainview);

    /*
	if (state->save_unsaved_data_ind && mainview->file_edited) {	/ * save unsaved data... * /
		write_pwsafe_file(mainview);
/ *		if (CONFRESP_YES==interface_save_changes_note(mainview)) {
		} * /
	} 
    */
    
}

/* application forced exit callback  */
//void exit_event_handler (gboolean die_now, MainView *mainview) {
gboolean exit_event_handler( GtkWidget *widget, GdkEvent  *event, MainView *mainview) {
    if (mainview && mainview->data) {
        save_mod_file(mainview);
        close_file(mainview);
    }
	gtk_main_quit();
	return FALSE;
}


void maybe_save_file(MainView *mainview) {
/* save changes if file changed, with requester*/
    if (mainview->pass && mainview->file_edited) {
        if (CONFRESP_YES==interface_yes_no_note(mainview, "Save Changes?")) { 
            if (!mainview->file_name) {  /* should never happen because file name set on open */
                mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
            }
            write_pwsafe_file( mainview );
        } 
    }
}

void save_mod_file(MainView *mainview) {
/* save changes if file changed, don't ask */
    if (mainview->pass && mainview->file_edited) {
        if (!mainview->file_name) {  /* should never happen because file name set on open */
            mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
        }
        write_pwsafe_file( mainview );
    }
}

gchar * getCurrentGroup(MainView *mainview ) {
	//TODO get the current Group from the selected one
	gchar *group = g_strdup("");
	return group;
}

/* add entry */
void callback_add_entry ( GtkAction * action, MainView *mainview ) {
	rec* newrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	newrec=add_new_record("new", getCurrentGroup(mainview));
	populate_treeView(mainview);
	show_rec(mainview, newrec);
    interface_set_rec_editable(mainview->rec_win, TRUE);
}

/* change password */
void callback_change_password(GtkAction * action, MainView *mainview) {
	char *oldpass=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	oldpass=get_password(mainview, "Old Password");
	if (oldpass && mainview->pass && (strcmp(oldpass, mainview->pass)==0)) {
		char *newpass=get_confirmed_pass(mainview);
		if (newpass) mainview->pass=newpass;
	}
}

char *get_confirmed_pass(MainView *mainview) {
	char *result=NULL;
	char *confirm=NULL;
	result=get_password(mainview, "New Password");
	if (result==NULL) return NULL;
	confirm=get_password(mainview, "Confirm Password");
	if (confirm==NULL) {
		trashFreeMemory(result, -1);
		return NULL;
	}
	if (strcmp(result, confirm)==0) {
		trashFreeMemory(confirm, -1);
		return result;
	}
	trashFreeMemory(confirm, -1);
	trashFreeMemory(result, -1);
	msgbox(mainview, "Password Mismatch");
	return NULL;
}

/* new */
void callback_file_new(GtkAction * action, MainView *mainview) {
	char *newpass;
	g_assert(mainview != NULL && mainview->data != NULL );
	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	close_file(mainview);
	newpass=get_confirmed_pass(mainview);
	if (newpass) {
		mainview->pass=newpass;
		add_new_record("new", getCurrentGroup(mainview));
		populate_treeView(mainview);
        /* maybe_save_file(mainview); */
        write_pwsafe_file(mainview);  /* force set file name */
	}
}

/* open */
void callback_file_open(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_OPEN );
	read_pwsafe_file(mainview);
}

void callback_file_open_last ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = mainview->last_file;
	read_pwsafe_file(mainview);
}

/* close file */
void callback_file_close ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
	maybe_save_file(mainview);
	close_file(mainview);
}

/* save */
void callback_file_save(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	write_pwsafe_file(mainview);
}


/* save as... */
void callback_file_saveas(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	mainview->file_name=NULL;
	write_pwsafe_file(mainview);
}


/* fullscreen */
void callback_fullscreen( GtkAction * action, gpointer data ) {
	interface_fullscreen((MainView *) data);
}


/* filter */

void callback_filter(GtkWidget *widget, MainView *mainview ) {
    const gchar *ftxt = NULL;
    GList *onrec=NULL;

#if MAEMO_VERSION_MAJOR < 5
	ftxt = gtk_entry_get_text(GTK_ENTRY(widget));
#else
	ftxt = hildon_entry_get_text(HILDON_ENTRY(widget));
#endif
//    g_object_get(G_OBJECT(action), "text", &ftxt, NULL);
    
    onrec=g_list_first(recs);
    if (ftxt[0] == '\0') { /* no filter, skip= none */
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            data->skip=0;
            onrec=g_list_next(onrec);
        }
    } else {
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            
            data->skip= ( strcasestr(data->group,ftxt) ? 0 : 1 );
            if (data->skip != 0) {
                data->skip= ( strcasestr(data->title,ftxt) ? 0 : 1);
                if (data->skip != 0) {
                    data->skip= ( strcasestr(data->user,ftxt) ? 0 : 1);
                    if (data->skip != 0) {
                        data->skip= ( strcasestr(data->notes,ftxt) ? 0 : 1);
                        if (data->skip != 0) {
                            data->skip= ( strcasestr(data->password,ftxt) ? 0 : 1);
                        }
                    }
                }
            }
            onrec=g_list_next(onrec);
        }
    }
    populate_treeView(mainview);
}

/* export */
void callback_export ( GtkAction * action, MainView *mainview ) {
    char *fname;
    
    fname = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
    if (fname != NULL) {
        FILE *outf;
        outf = fopen(fname,"w");
        if (outf != (FILE *) NULL) {
            GList *onrec=NULL;
     
            onrec=g_list_first(recs);
            while (onrec!=NULL) {
                rec *data=(rec *) onrec->data;
                   
                fprintf(outf,"Group: %s\n",data->group);
                fprintf(outf,"Title: %s\n",data->title);
                fprintf(outf,"User: %s\n",data->user);
                fprintf(outf,"Password: %s\n",data->password);
                fprintf(outf,"Notes: %s\n",data->notes);
                fprintf(outf,"\n--\n");
                       
                onrec=g_list_next(onrec);
            }
        } else {
            interface_error(PWSAFE_ERROR_SAVE_FAILED, mainview);
        }
    }
}
