/*
 * This file is part of pwsafe
 *
 * Copyright (C) 2005 Nokia Corporation.
 * Copyright (C) 2005 HolisTech Limited.
 *
 *
 * This software is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
                     
#include <src/callbacks.h>
#include <src/interface.h>
#include <src/crypto.h>
#include <gtk/gtk.h>
#include <libintl.h>
/* strlen needed from string.h */
#include <string.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#if HILDON == 1
#include <hildon/hildon.h>
#else
#include <hildon-widgets/hildon-app.h>
#endif
#include <gdk/gdkkeysyms.h>

/* Privates: */
void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview);
void maybe_save_file(MainView *mainview);
void save_mod_file(MainView *mainview);
char *get_confirmed_pass(MainView *mainview);

/* activate grid */
void callback_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, GtkTreeViewColumn *column, MainView *mainview) {
	callback_hildon_activate_treeView(treeView, path, mainview);
}

/* activate hildon grid */
void callback_hildon_activate_treeView(GtkTreeView *treeView, GtkTreePath *path, MainView *mainview) {
	GtkTreeIter iter;
	rec *onrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	if ( !gtk_tree_model_get_iter(GTK_TREE_MODEL(mainview->store), &iter, path) ) {
		return;
	}
	gtk_tree_model_get(GTK_TREE_MODEL(mainview->store), &iter,
	    DATA_COLUMN, &onrec,
	    -1);
	/*
	onrec=g_object_get_data (G_OBJECT(&iter), "r");
	*/
	if (onrec!=NULL) {
		show_rec(mainview, onrec);
	} else if (gtk_tree_view_collapse_row(treeView, path)) {
		gtk_tree_view_expand_row(treeView, path, FALSE);
	} else {
		gtk_tree_view_collapse_row(treeView, path);
	}
}

gboolean callback_key_press(GtkWidget* widget, GdkEventKey* event,  MainView *mainview) {
	switch (event->keyval) {
		case GDK_F6: /* fullscreen */
			interface_fullscreen(mainview);
			return TRUE;
		case GDK_F7: /* button + */
			return TRUE;
		case GDK_F8: /* button - */
			return TRUE;
	}
	return FALSE;
}

void safeReplaceString(char** oldstr, const gchar *newstr, MainView *mainview) {
/* scrub and delete oldstr, malloc and duplicate newstr, return pointer and tag file as changed... */
/* but only if the strings are different (and be careful of null old string pointers) */
	if ((*oldstr)==NULL) {
		if (newstr[0]!='\0') {
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	} else {
		if (strcmp((*oldstr),newstr)!=0) { /* check they differ before copying stuff around */
			trashMemory((*oldstr), strlen((*oldstr)));
			g_free((*oldstr));
			(*oldstr)=g_strdup(newstr);
			mainview->file_edited=TRUE; /* tag file as changed */
		}
	}
}

const gchar *entryGetText(GtkWidget *widget) {
#if MAEMO_VERSION_MAJOR < 5
	return gtk_entry_get_text(GTK_ENTRY(widget));
#else
	return hildon_entry_get_text(HILDON_ENTRY(widget));
#endif
}

/* record detail events */
void callback_rec_response(GtkDialog *dialog, gint arg1, MainView *mainview) {
	RecWindow *recwindow = NULL;
	rec *rec=NULL;
	GtkTextBuffer *buffer=NULL;
	GtkTextIter start;
	GtkTextIter end;

	g_assert(mainview != NULL && mainview->data != NULL );
	
	recwindow = mainview->rec_win;
	rec=recwindow->rec;
	switch (arg1) {
		case 1: /* save any changes */
			safeReplaceString(&rec->title, entryGetText(recwindow->title), mainview);
			safeReplaceString(&rec->user, entryGetText(recwindow->username), mainview);
			safeReplaceString(&rec->password, entryGetText(recwindow->password), mainview);
#if MAEMO_VERSION_MAJOR < 5
			safeReplaceString(&rec->group, gtk_combo_box_get_active_text(GTK_COMBO_BOX(recwindow->group)), mainview);
			buffer=gtk_text_view_get_buffer (GTK_TEXT_VIEW(recwindow->notes));
#else
			safeReplaceString(&rec->group, hildon_button_get_value(HILDON_BUTTON(recwindow->group)), mainview);
			buffer=hildon_text_view_get_buffer(HILDON_TEXT_VIEW(recwindow->notes));
#endif
			if (gtk_text_buffer_get_modified(buffer)) { /* update notes if buffer has changed */
				gtk_text_buffer_get_start_iter (buffer, &start);
				gtk_text_buffer_get_end_iter (buffer, &end);
				safeReplaceString(&rec->notes, gtk_text_buffer_get_text (buffer, &start, &end, FALSE), mainview);
			}
			resort_rec(rec);
			unshow_rec(mainview);  /* close rec window */
			populate_treeView(mainview); /* redraw grid */
			break;
		case 2:
			gtk_clipboard_set_text(mainview->clipboard, rec->user, -1);
			break;
		case 3:
			gtk_clipboard_set_text(mainview->clipboard, rec->password, -1);
			break;
		case 4:
			if ((recwindow->editable) && (CONFRESP_YES==interface_yes_no_note(mainview, "Confirm Delete?"))) {
				unshow_rec(mainview);
				delete_rec(rec);
				populate_treeView(mainview); /* redraw grid */
			}
			break;
		case 5:
			interface_set_rec_editable(recwindow, TRUE);
			break;
		case -4:
			//Cancel fro Fremantle (out of dialog click)
			unshow_rec(mainview);  /* close rec window */
			populate_treeView(mainview); /* redraw grid */
			break;
	}
}

static void on_picker_value_changed (HildonPickerButton * button, gpointer data) {
  g_print ("Newly selected value: %s\n",
           hildon_button_get_value (HILDON_BUTTON (button)));
}


/* Callback for hardware D-BUS events */
void hw_event_handler(osso_hw_state_t *state, MainView *mainview) {
 	g_assert(mainview != NULL && mainview->data != NULL );

	if ((state->shutdown_ind)	/* system shutdown */
			|| (state->system_inactivity_ind)) {	/* system sleeping */
        save_mod_file(mainview);
		close_file(mainview);
        return;
	}
	if (state->memory_low_ind) {	/* memory low... */
		unshow_rec(mainview);	/* close the rec window... changes not saved... */
	}
    save_mod_file(mainview);

    /*
	if (state->save_unsaved_data_ind && mainview->file_edited) {	/ * save unsaved data... * /
		write_pwsafe_file(mainview);
/ *		if (CONFRESP_YES==interface_save_changes_note(mainview)) {
		} * /
	} 
    */
    
}

/* application forced exit callback  */
//void exit_event_handler (gboolean die_now, MainView *mainview) {
gboolean exit_event_handler( GtkWidget *widget, GdkEvent  *event, MainView *mainview) {
    if (mainview && mainview->data) {
        save_mod_file(mainview);
        close_file(mainview);
    }
	gtk_main_quit();
	return FALSE;
}


void maybe_save_file(MainView *mainview) {
/* save changes if file changed, with requester*/
    if (mainview->pass && mainview->file_edited) {
        if (CONFRESP_YES==interface_yes_no_note(mainview, "Save Changes?")) { 
            if (!mainview->file_name) {  /* should never happen because file name set on open */
                mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
            }
            write_pwsafe_file( mainview );
        } 
    }
}

void save_mod_file(MainView *mainview) {
/* save changes if file changed, don't ask */
    if (mainview->pass && mainview->file_edited) {
        if (!mainview->file_name) {  /* should never happen because file name set on open */
            mainview->file_name = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
        }
        write_pwsafe_file( mainview );
    }
}

gchar * getCurrentGroup(MainView *mainview ) {
	//TODO get the current Group from the selected one
	gchar *group = g_strdup("");
	return group;
}

/* add entry */
void callback_add_entry ( GtkAction * action, MainView *mainview ) {
	rec* newrec=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	newrec=add_new_record("new", getCurrentGroup(mainview));
	populate_treeView(mainview);
	show_rec(mainview, newrec);
    interface_set_rec_editable(mainview->rec_win, TRUE);
}

/* change password */
void callback_change_password(GtkAction * action, MainView *mainview) {
	char *oldpass=NULL;
	g_assert(mainview != NULL && mainview->data != NULL );
	oldpass=get_password(mainview, "Old Password");
	if (oldpass && mainview->pass && (strcmp(oldpass, mainview->pass)==0)) {
		char *newpass=get_confirmed_pass(mainview);
		if (newpass) mainview->pass=newpass;
	}
}

char *get_confirmed_pass(MainView *mainview) {
	char *result=NULL;
	char *confirm=NULL;
	result=get_password(mainview, "New Password");
	if (result==NULL) return NULL;
	confirm=get_password(mainview, "Confirm Password");
	if (confirm==NULL) {
		trashFreeMemory(result, -1);
		return NULL;
	}
	if (strcmp(result, confirm)==0) {
		trashFreeMemory(confirm, -1);
		return result;
	}
	trashFreeMemory(confirm, -1);
	trashFreeMemory(result, -1);
	msgbox(mainview, "Password Mismatch");
	return NULL;
}

/* new */
void callback_file_new(GtkAction * action, MainView *mainview) {
	char *newpass;
	g_assert(mainview != NULL && mainview->data != NULL );
	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	close_file(mainview);
	newpass=get_confirmed_pass(mainview);
	if (newpass) {
		mainview->pass=newpass;
		add_new_record("new", getCurrentGroup(mainview));
		populate_treeView(mainview);
        /* maybe_save_file(mainview); */
        write_pwsafe_file(mainview);  /* force set file name */
	}
}

/* open */
void callback_file_open(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = interface_file_chooser ( mainview, GTK_FILE_CHOOSER_ACTION_OPEN );
	read_pwsafe_file(mainview);
}

void callback_file_open_last ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
 	/* maybe_save_file(mainview); */
    save_mod_file(mainview);
	mainview->file_name = mainview->last_file;
	read_pwsafe_file(mainview);
}

/* close file */
void callback_file_close ( GtkAction * action, MainView *mainview ) {
	g_assert(mainview != NULL && mainview->data != NULL );
	maybe_save_file(mainview);
	close_file(mainview);
}

/* save */
void callback_file_save(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	write_pwsafe_file(mainview);
}


/* save as... */
void callback_file_saveas(GtkAction * action, MainView *mainview) {
	g_assert(mainview != NULL && mainview->data != NULL );
	mainview->file_name=NULL;
	write_pwsafe_file(mainview);
}


/* fullscreen */
void callback_fullscreen( GtkAction * action, gpointer data ) {
	interface_fullscreen((MainView *) data);
}


/* filter */

void callback_filter( GtkAction * action, MainView *mainview ) {
    gchar *ftxt = NULL;
    GList *onrec=NULL;
    
    g_object_get(G_OBJECT(action), "text", &ftxt, NULL);
    
    onrec=g_list_first(recs);
    if (ftxt[0] == '\0') { /* no filter, skip= none */
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            data->skip=0;
            onrec=g_list_next(onrec);
        }
    } else {
        while (onrec!=NULL) {
            rec *data=(rec *) onrec->data;
            
            data->skip= ( strcasestr(data->group,ftxt) ? 0 : 1 );
            if (data->skip != 0) {
                data->skip= ( strcasestr(data->title,ftxt) ? 0 : 1);
                if (data->skip != 0) {
                    data->skip= ( strcasestr(data->user,ftxt) ? 0 : 1);
                    if (data->skip != 0) {
                        data->skip= ( strcasestr(data->notes,ftxt) ? 0 : 1);
                        if (data->skip != 0) {
                            data->skip= ( strcasestr(data->password,ftxt) ? 0 : 1);
                        }
                    }
                }
            }
            onrec=g_list_next(onrec);
        }
    }
    populate_treeView(mainview);
}

/* export */
void callback_export ( GtkAction * action, MainView *mainview ) {
    char *fname;
    
    fname = interface_file_chooser(mainview, GTK_FILE_CHOOSER_ACTION_SAVE);
    if (fname != NULL) {
        FILE *outf;
        outf = fopen(fname,"w");
        if (outf != (FILE *) NULL) {
            GList *onrec=NULL;
     
            onrec=g_list_first(recs);
            while (onrec!=NULL) {
                rec *data=(rec *) onrec->data;
                   
                fprintf(outf,"Group: %s\n",data->group);
                fprintf(outf,"Title: %s\n",data->title);
                fprintf(outf,"User: %s\n",data->user);
                fprintf(outf,"Password: %s\n",data->password);
                fprintf(outf,"Notes: %s\n",data->notes);
                fprintf(outf,"\n--\n");
                       
                onrec=g_list_next(onrec);
            }
        } else {
            interface_error(PWSAFE_ERROR_SAVE_FAILED, mainview);
        }
    }
}
