#include <QDebug>
#include "DraggableViewArea.h"

DraggableViewArea *DraggableViewArea::dragging = NULL;

DraggableViewArea::DraggableViewArea(Qt::Orientation orientation, QWidget *parent) :
    QWidget(parent), m_Orientation(orientation)
{
    this->m_Content         = new QWidget(this);
    this->m_CurrentIndex    = 0;

    this->m_Transition = new QPropertyAnimation(this->m_Content, "geometry", this);

    QObject::connect(this->m_Transition, SIGNAL(finished()),
                     this, SLOT(animationFinished()));

    this->installEventFilter(this);
}

void DraggableViewArea::addWidget(QWidget *widget)
{
    widget->setParent(this->m_Content);

    this->m_ChildWidgets.push_back(widget);

    if(this->m_Orientation == Qt::Horizontal)
    {
        widget->setGeometry(0,
                            (this->m_ChildWidgets.size() - 1) * this->height(),
                            this->width(),
                            this->height());
        this->m_Content->setGeometry(0,
                                     -this->m_CurrentIndex * this->height(),
                                     this->width(),
                                     this->m_ChildWidgets.size() * this->height());
    }
    else
    {
        widget->setGeometry((this->m_ChildWidgets.size() - 1) * this->width(),
                            0,
                            this->width(),
                            this->height());
        this->m_Content->setGeometry(-this->m_CurrentIndex * this->width(),
                                     0,
                                     this->m_ChildWidgets.size() * this->width(),
                                     this->height());
    }

    widget->show();
}

bool DraggableViewArea::eventFilter(QObject *object, QEvent *event)
{
    if(object == this)
    {
        switch(event->type())
        {
        case QEvent::Resize:
            for(unsigned int i = 0; i < m_ChildWidgets.size(); i++)
            {
                QWidget *widget = this->m_ChildWidgets[i];

                widget->setFixedSize(this->width(), this->height());

                if(this->m_Orientation == Qt::Horizontal)
                {
                    widget->setGeometry(this->width() * i, 0, this->width(), this->height());
                }
                else
                {
                    widget->setGeometry(0, this->height() * i, this->width(), this->height());
                }
            }

            if(this->m_Orientation == Qt::Horizontal)
            {
                this->m_Content->setGeometry(-this->m_CurrentIndex*width(),
                                             0,
                                             this->width() * this->m_ChildWidgets.size(),
                                             this->height());
            }
            else
            {
                this->m_Content->setGeometry(0,
                                             -this->m_CurrentIndex * this->height(),
                                             this->width(),
                                             this->height() * this->m_ChildWidgets.size());
            }
            break;

        default:
            break;
        }
    }

    return QWidget::eventFilter(object, event);
}

void DraggableViewArea::jumpTo(int index)
{
    if (dragging == this) return;

    if (index < 0)
    {
        this->jumpTo(0);
        return;
    }

    if (index >= this->m_ChildWidgets.size())
    {
        this->jumpTo(this->m_ChildWidgets.size() - 1);
        return;
    }

    QWidget *widget = this->m_ChildWidgets[index];

    this->m_Content->setGeometry(-widget->x(),
                                 -widget->y(),
                                 this->m_Content->width(),
                                 this->m_Content->height());
    this->m_CurrentIndex = index;
}

void DraggableViewArea::jumpTo(QWidget *widget)
{
    for(unsigned int i = 0; i < this->m_ChildWidgets.size(); i++)
    {
        if(this->m_ChildWidgets[i] == widget)
        {
            this->jumpTo(i);
            return;
        }
    }
}

void DraggableViewArea::animationFinished()
{
    emit slidTo(this->m_CurrentIndex);
    emit slidTo(this->m_ChildWidgets[this->m_CurrentIndex]);
}

void DraggableViewArea::slideTo(int index)
{
    if (dragging == this) return;

    if (index < 0)
    {
        this->slideTo(0);
        return;
    }
    if (index >= this->m_ChildWidgets.size())
    {
        this->slideTo(this->m_ChildWidgets.size() - 1);
        return;
    }

    QWidget *widget = this->m_ChildWidgets[index];

    this->m_CurrentIndex = index;

    this->m_Transition->setDuration(250);
    this->m_Transition->setEasingCurve(QEasingCurve::InOutQuad);
    this->m_Transition->setStartValue(this->m_Content->geometry());
    this->m_Transition->setEndValue(QRect(-widget->x(),
                                          -widget->y(),
                                          this-> m_Content->width(),
                                          this->m_Content->height()));
    this->m_Transition->start();
}

void DraggableViewArea::slideTo(QWidget *widget)
{
    for (unsigned int i = 0; i < this->m_ChildWidgets.size(); i++)
    {
        if (this->m_ChildWidgets[i] == widget)
        {
            this->slideTo(i);
            return;
        }
    }
}

void DraggableViewArea::mousePressEvent(QMouseEvent *event)
{
    this->m_MouseAnchor = event->globalPos();
    event->ignore();
}

void DraggableViewArea::mouseMoveEvent(QMouseEvent *event)
{
    int dx = event->globalX() - this->m_MouseAnchor.x();
    int dy = event->globalY() - this->m_MouseAnchor.y();

    if(this->m_Orientation == Qt::Horizontal)
    {
        if (abs(dy) < abs(dx) && abs(dx) > 40 && dragging == NULL)
        {
            dragging = this;
            this->m_MouseAnchorFiltered = m_MouseAnchor;
        }
        else if (dragging != this)
        {
            event->ignore();
            return;
        }
    }
    else
    {
        if (abs(dy) > abs(dx) && abs(dy) > 40 && dragging == NULL)
        {
            dragging = this;
            this->m_MouseAnchorFiltered = m_MouseAnchor;
        }
        else if (dragging != this)
        {
            event->ignore();
            return;
        }
    }

    if (dragging == this)
    {
        // stop any animation
        this->m_Transition->stop();

        if(this->m_Orientation == Qt::Horizontal)
        {
            int newX = this->m_Content->x()+dx;

            if (newX > 0) newX = 0;
            if (newX < this->width() - this->m_Content->width()) newX = this->width() - this->m_Content->width();

            this->m_Content->setGeometry(newX,
                                         this->m_Content->y(),
                                         this->m_Content->width(),
                                         this->m_Content->height());
            this->m_MouseAnchor = event->globalPos();
            this->m_MouseAnchorFiltered += this->m_MouseAnchor;
            this->m_MouseAnchorFiltered *= 0.5;
        }
        else
        {
            int newY = this->m_Content->y()+dy;

            if (newY > 0) newY = 0;
            if (newY < this->height() - this->m_Content->height()) newY = this->height() - this->m_Content->height();

            this->m_Content->setGeometry(this->m_Content->x(),
                                         newY,
                                         this->m_Content->width(),
                                         this->m_Content->height());
            this->m_MouseAnchor = event->globalPos();
            this->m_MouseAnchorFiltered += this->m_MouseAnchor;
            this->m_MouseAnchorFiltered *= 0.5;
        }
    }
    else
    {
        event->ignore();
    }
}

void DraggableViewArea::mouseReleaseEvent(QMouseEvent *event)
{
    if (dragging == this)
    {
        qreal dx = event->globalX() - this->m_MouseAnchorFiltered.x();
        qreal dy = event->globalY() - this->m_MouseAnchorFiltered.y();

        int dv, dc, dt;

        if(this->m_Orientation == Qt::Horizontal)
        {
            dv = dx;
            dc = this->m_Content->x();
            dt = this->width();
        }
        else
        {
            dv = dy;
            dc = this->m_Content->y();
            dt = this->height();
        }

        this->m_Transition->setStartValue(this->m_Content->geometry());

        int delta = 0;

        if (dv > 3 && this->m_CurrentIndex > 0 && dc > -dt * this->m_CurrentIndex)
        {
            delta = -1;
        }
        else if (dv < -3 && this->m_CurrentIndex < this->m_ChildWidgets.size() - 1 && dc < -dt * this->m_CurrentIndex)
        {
            delta = +1;
        }
        else if (abs(dv) < 3)
        {
            int prev    = abs(dc + dt * (this->m_CurrentIndex - 1));
            int here    = abs(dc + dt * this->m_CurrentIndex);
            int next    = abs(dc + dt * (this->m_CurrentIndex + 1));

            if (prev < here && prev < next)
            {
                delta = -1;
            }
            else if (next < here)
            {
                delta = +1;
            }
        }

        this->m_CurrentIndex += delta;

        if(this->m_Orientation == Qt::Horizontal)
        {
            this->m_Transition->setEndValue(QRect(-this->m_CurrentIndex * this->width(),
                                                  this->m_Content->y(),
                                                  this->m_Content->width(),
                                                  this->m_Content->height()));

        }
        else
        {
            this->m_Transition->setEndValue(QRect(this->m_Content->x(),
                                                  -this->m_CurrentIndex * this->height(),
                                                  this->m_Content->width(),
                                                  this->m_Content->height()));
        }

        this->m_Transition->setDuration(100);
        this->m_Transition->setEasingCurve(QEasingCurve::OutQuad);
        this->m_Transition->start();

        dragging = NULL;
    } else {
        event->ignore();
    }
}
