// -*- coding: utf-8; -*-
// (c) Copyright 2010, Nick Slobodsky (Николай Слободской)
// This file is part of PlansPlant.
//
// PlansPlant is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// PlansPlant is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with PlansPlant.  If not, see <http://www.gnu.org/licenses/>.
//
#include <timeshop.hpp>
#include "plansplant/tasks.hpp"
#include <QDebug>

namespace PlansPlant
{
  QString Task::units_name( TimeUnits Units )
  {
    QString Result = QObject::tr( "unknown" );
    switch( Units )
    {
    case Seconds:   Result = QObject::tr( "Seconds" ); break;
    case Minutes:   Result = QObject::tr( "Minutes" ); break;
    case Hours:     Result = QObject::tr( "Hours" ); break;
    case WorkDays:  Result = QObject::tr( "WorkDays" ); break;
    case Days:      Result = QObject::tr( "Days" ); break;
    case WorkWeeks: Result = QObject::tr( "WorkWeeks" ); break;
    case Weeks:     Result = QObject::tr( "Weeks" ); break;
    case Months:    Result = QObject::tr( "Months" ); break;
    case Quarters:  Result = QObject::tr( "Quarters" ); break;
    case Years:     Result = QObject::tr( "Years" ); break;
    }
    return Result;
  } // units_name( TimeUnits )
  QString Task::units_short_name( TimeUnits Units )
  {
    QString Result = QObject::tr( "unknown" );
    switch( Units )
    {
    case Seconds:   Result = QObject::tr( "s" ); break;
    case Minutes:   Result = QObject::tr( "min" ); break;
    case Hours:     Result = QObject::tr( "h" ); break;
    case WorkDays:  Result = QObject::tr( "wd" ); break;
    case Days:      Result = QObject::tr( "d" ); break;
    case WorkWeeks: Result = QObject::tr( "ww" ); break;
    case Weeks:     Result = QObject::tr( "w" ); break;
    case Months:    Result = QObject::tr( "M" ); break;
    case Quarters:  Result = QObject::tr( "Q" ); break;
    case Years:     Result = QObject::tr( "Y" ); break;
    }
    return Result;
  } // units_short_name( TimeUnits )
  QDateTime Task::add_time( const QDateTime& Start, Time Amount, TimeUnits Units )
  {
    QDateTime Result = Start;
    if( Amount != 0 )
      switch( Units )
      {
      case Years:
	Result = Start.addYears( Amount );
	break;
      case Quarters:
	Result = Start.addMonths( Amount*3 );
	break;
      case Months:
	Result = Start.addMonths( Amount );
	break;
      case WorkWeeks:
	Result = Start.addDays( Amount*7 );
	break;
      case WorkDays:
	Result = Start.addDays( ( Amount / 5 ) * 7 + Amount % 5 );
	break;
      default:
	Result = Start.addSecs( Amount * Units );
	break;
      }
    return Result;
  } // add_time( const QDateTime&, Time, TimeUnits )

  Task::Task( const QString& Name0, Pointer SuperTask0, ID ID0 ) : Ident( ID0 ), SuperTask( SuperTask0 ), Name( Name0 ), Completed( 0 ), Estimation( 0 ), EstimationUnits( Seconds )
  {
    if( LastID < Ident ) LastID = Ident;
    if( SuperTask && SuperTask->SubTasks.indexOf( this ) < 0 )
      SuperTask->SubTasks.push_back( this );
  } // Task( const QString&, Pointer* )
  Task::~Task()
  {
    while( !SubTasks.empty() )
      delete SubTasks.back(); // Will be removed from this list in the subtask's destructor.
    supertask( 0 );
  } // ~Task()
  
  void Task::supertask( Task* NewTask )
  {
    if( NewTask != SuperTask )
    {
      if( SuperTask )
	SuperTask->SubTasks.removeAll( this );
      SuperTask = NewTask;
      if( SuperTask )
	SuperTask->SubTasks.push_back( this );
    }
  } // supertask( Task* )

  bool Task::check_loop( Task& NewChild ) const
  {
    return check_loop( NewChild.needs(), &NewChild );
  } // check_loop( Task& )
  bool Task::blockers_list_helper( Task::List& List, const Task::List& Source ) const
  {
    bool Result = false;
    for( Task::List::const_iterator It = Source.begin(); It != Source.end() && !Result; It++ )
      if( Task* Dep = *It )
      {
	if( Dep == this ) Result = true;
	else
	  if( !List.contains( Dep ) )
	    List.push_back( Dep );
      }
    return Result;
  } // blockers_list_helper( Task::List&, const Task::List& )
  bool Task::check_loop( const Task::List& Blockers, Task* NewChild ) const
  {
    bool Result = this == NewChild;
    if( !Result ) // Search for it in the tasks that depends on this
    { //! \todo Optimize the search
      Task::List Deps;
      int I = 0;
      if( NewChild )
      {
	Deps.push_back( NewChild );
	Result = blockers_list_helper( Deps, NewChild->subtasks() );
	I++;
      }
      Result = Result || blockers_list_helper( Deps, Blockers );
      while( I < Deps.size() && !Result )
      {
	if( Task* Dep = Deps[ I ] )
	{
	  if( Dep == this ) Result = true; // No need for this, because we check it when add data to the Deps list.
	  else Result = Result || blockers_list_helper( Deps, Dep->subtasks() ) || blockers_list_helper( Deps, Dep->needs() );
	}
	I++;
      } // for each Dep.
    }
    return Result;
  } // check_loop( const Task::List&, Task* ) const
  bool Task::move_subtask( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < SubTasks.size() && To >= 0 && To < SubTasks.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  SubTasks.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_subtask( int, int )
  bool Task::move_dependency( int From, int To )
  {
    bool Result = false;
    if( From >= 0 && From < Needs.size() && To >= 0 && To < Needs.size() )
    {
      if( From == To )
	Result = true;
      else
      {
	int Inc = ( To > From ) ? 1 : -1;
	for( int Index = From; Index != To; Index += Inc )
	{
	  qDebug() << "Swap tasks" << Index << "and" << Index+Inc;
	  Needs.swap( Index, Index+Inc );
	}
	Result = true;
      }
    }
    return Result;
  } // move_dependency( int, int )
  bool Task::blocked() const
  {
    bool Result = false;
    foreach( Task* SubTask, subtasks() )
      if( SubTask && SubTask->completed() < 1 )
	Result = true;
    if( !Result )
      foreach( Task* Dep, needs() )
	if( Dep && Dep->completed() < 1 )
	  Result = true;
    return Result;
  } // blocked() const
  void Task::add_to_map( Map& TasksMap )
  {
    if( !Ident ) Ident = new_id();
    TasksMap[ Ident ] = this;
    foreach( Task* SubTask, SubTasks )
      if( SubTask )
	SubTask->add_to_map( TasksMap );
  } // add_to_map( Map& )
  void Task::write( QXmlStreamWriter& Stream )
  {
    Stream.writeStartElement( "task" );
    if( !Ident ) Ident = new_id();
    Stream.writeAttribute( "id", QString::number( Ident ) );
    // We don't write a supertask (parent), because we save tasks in parent's subtasks list.
    if( !Name.isEmpty() ) Stream.writeTextElement( "name", Name );
    if( !Description.isEmpty() ) Stream.writeTextElement( "description", Description );
    if( !Comment.isEmpty() ) Stream.writeTextElement( "comment", Comment );
    if( PlanStart.isValid() ) Stream.writeTextElement( "plan_start", PlanStart.toUTC().toString( "yyyy-MM-dd hh:mm:ss" ) );
    if( PlanFinish.isValid() ) Stream.writeTextElement( "plan_finish", PlanFinish.toUTC().toString( "yyyy-MM-dd hh:mm:ss" ) );
    if( Completed > 0 ) Stream.writeTextElement( "completed", QString::number( Completed ) );
    if( Estimation > 0 )
    {
      Stream.writeStartElement( "estimation" );
      Stream.writeAttribute( "units", QString::number( EstimationUnits ) );
      Stream.writeCharacters( QString::number( Estimation ) );
      Stream.writeEndElement();
    }
    if( !SubTasks.isEmpty() )
    {
      Stream.writeStartElement( "subtasks" );
      foreach( Task* SubTask, SubTasks )
	if( SubTask )
	  SubTask->write( Stream );
      Stream.writeEndElement();
    }
    Stream.writeEndElement();
  } // write( QXmlStreamWriter& ) const
  void Task::write_dependencies( QXmlStreamWriter& Stream )
  {
    if( !Needs.empty() )
    {
      Stream.writeStartElement( "dependencies" );
      Stream.writeAttribute( "task_id", QString::number( Ident ) );
      foreach( Task* Need, Needs )
	if( Need )
	{
	  if( !Need->Ident ) Need->Ident = new_id();
	  Stream.writeTextElement( "blocker", QString::number( Need->id() ) );
	}
      Stream.writeEndElement();
    }
    foreach( Task* SubTask, SubTasks )
      if( SubTask )
	SubTask->write_dependencies( Stream );
  } // write_dependencies( QXmlStreamWriter& )
  QDateTime read_time( QXmlStreamReader& Stream ) // no "fromUTCString" method in QDateTime.
  {
    QDateTime Result = QDateTime::fromString( Stream.readElementText(), "yyyy-MM-dd hh:mm:ss" );
    Result.setTimeSpec( Qt::UTC );
    return Result.toLocalTime();
  } // read_time( QXmlStreamReader& Stream )
  void Task::load( QXmlStreamReader& Stream )
  {
    if( Stream.isStartElement() && Stream.name() == "task" )
    {
      Timeshop::Persistent::Loader::attribute( Stream.attributes(), "id", Ident );
      if( LastID < Ident ) LastID = Ident;
      while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
      {
	QStringRef Tag = Stream.name();
	if( Tag == "name" ) Name = Stream.readElementText();
	else if( Tag == "description" ) Description = Stream.readElementText();
	else if( Tag == "comment" ) Comment = Stream.readElementText();
	else if( Tag == "plan_start" ) PlanStart = read_time( Stream );
	else if( Tag == "plan_finish" ) PlanFinish = read_time( Stream );
	else if( Tag == "completed" ) Completed = Stream.readElementText().toDouble();
	else if( Tag == "estimation" )
	{
	  int Units = 1;
	  if( Timeshop::Persistent::Loader::attribute( Stream.attributes(), "units", Units ) )
	    EstimationUnits = TimeUnits( Units );
	  Estimation = Stream.readElementText().toInt();
	}
	else if( Tag == "subtasks" )
	{
	  while( Timeshop::Persistent::Loader::next_subelement( Stream ) )
	    if( Stream.isStartElement() && Stream.name() == "task" )
	      ( new Task( QString(), this ) )->load( Stream );
	    else
	      Timeshop::Persistent::Loader::skip( Stream );
	}
	else
	  Timeshop::Persistent::Loader::skip( Stream );
      }
    }
  } // load( QXmlStreamReader& )

  void Task::add_dependency( Task& Upon )
  {
    if( !Needs.contains( &Upon ) && !Upon.Dependents.contains( this ) )
    {
      Needs.push_back( &Upon );
      Upon.Dependents.push_back( this );
    }
  } // add_dependency( Task&, Task& )
  void Task::remove_dependency( Task& Upon )
  {
    Needs.removeAll( &Upon );
    Upon.Dependents.removeAll( this );
  } // remove_dependency( Task&, Task& ); 
  Task::ID Task::LastID = 0;
} // PlansPlant
