# -*- coding: utf-8 -*-

from __future__ import with_statement
import os
import datetime
from PyQt4.QtCore import QObject, SIGNAL
from notify import show_notification_dialog

DateTimeFormat='%Y%m%dT%H%M%S'

class Task(object):
    def __init__(self):
        self.desc = ''
        self.tags = ''
        self.projectnum = 0

        # date/time properties
        self.created = None
        self.completed = None
        self.start = None
        self.due = None

    def assign(self, rhs):
        self.desc = rhs.desc
        self.tags = rhs.tags
        self.projectnum = rhs.projectnum
        self.created = rhs.created
        self.completed = rhs.completed
        self.start = rhs.start
        self.due = rhs.due

    def toggleComplete(self):
        if self.completed is None:
            self.completed = datetime.datetime.now()
        else:
            self.completed = None

    def doToday(self):
        self.start = datetime.datetime.now()

    def overdue(self):
        return self.due is not None and \
                self.due < datetime.datetime.now()

    def __repr__(self):
        return self.desc+"["+self.tags+"]"

def getAll(t):
    return True

# return all tasks in project (or just completed ones)
# completed == True: count only completed ones (ie. ones with a completed date)
# completed == False: count all tasks
def getProject(t, p, completed):
    return t.projectnum == p and (not completed or t.completed is not None)

def getToday(t):
    today = datetime.datetime.today().date()
    return ((t.due is not None and t.due.date() <= today) \
        or (t.start is not None and t.start.date() <= today)) \
        and t.completed is None

def getNext(t):
    today = datetime.datetime.today().date()
    return ((t.start is None) or (t.start.date() > today)) \
        and t.completed is None \
        and not getTags(t, 'someday')

def getScheduled(t):
    today = datetime.datetime.today().date()
    return (t.start is not None and t.start.date() > today) \
        and t.completed is None

def getTags(t, tag):
    return tag in t.tags.split(',')

def getAnyTag(t, taglist):
    tsplit = t.tags.split(',')
    for i in taglist:
        if i in tsplit:
            return True
    return False

def getComplete(t):
    return t.completed is not None

class DataStore(QObject):
    def __init__(self):
        QObject.__init__(self)
        self.tasks = []
        self.projects = {0:'Inbox'}
        self.nextprojnum = 1
        self.currentfile = None
        self.alltags = ['someday']

    def emitUpdated(self):
        self.emit(SIGNAL('updated()'))

    def getTasks(self, filter=getAll, args=[]):
        output = []
        for t in self.tasks:
            if filter(t, *args):
                output.append(t)
        return sorted(output, key=lambda task: task.projectnum)

    def countTasks(self, filter=getAll, args=[]):
        count = 0
        for t in self.tasks:
            if filter(t, *args):
                count += 1
        return count

    def addTask(self, task, save=True):
        self.tasks.append(task)
        self.mergeTags(task.tags.split(','))
        if save:
            self.save()
            self.emitUpdated()
            show_notification_dialog("Task added")

    def updateTask(self, task, save=True):
        self.mergeTags(task.tags.split(','))
        if save:
            self.save()
        self.emitUpdated()

    def getProjects(self):
        return self.projects

    def getAllTags(self):
        return self.alltags

    def addProject(self, proj, num=None, save=True):
        if num is None:
            num = self.nextprojnum
            self.nextprojnum += 1
        else:
            self.nextprojnum = num+1
        self.projects[num] = proj
        if save:
            self.save()
            self.emitUpdated()

    def getProject(self, num):
        return self.projects[num]

    def load(self, filename):
        self.currentfile = filename
        with open(filename) as f:
            if f.next() != "pebble-do:1\n":
                print "error, wrong file version"
            else:
                for line in f:
                    if line == "BEGIN:PPROJECT\n":
                        self.parseProject(f)
                    elif line == "BEGIN:VTODO\n":
                        self.parseTodo(f)
        self.emitUpdated()

    def parseProject(self, f):
        # parse project definition
        props = self.parseProps(f, "END:PPROJECT\n")
        if "NUM" in props and "DESC" in props:
            self.addProject(props["DESC"], int(props["NUM"]), False)
        else:
            print "error, NUM and/or DESC not present in PPROJECT"

    def parseTodo(self, f):
        # parse task definition
        props = self.parseProps(f, "END:VTODO\n")
        if "SUMMARY" in props and "X-PROJECTNUM" in props:
            t = Task()
            t.desc = props["SUMMARY"]
            t.projectnum = int(props["X-PROJECTNUM"])
            if "CATEGORIES" in props:
                t.tags = props["CATEGORIES"]
            if "COMPLETED" in props:
                t.completed = datetime.datetime.strptime(props["COMPLETED"], DateTimeFormat)
            if "CREATED" in props:
                t.created = datetime.datetime.strptime(props["CREATED"], DateTimeFormat)
            if "DUE" in props:
                t.due = datetime.datetime.strptime(props["DUE"], DateTimeFormat)
            if "DTSTART" in props:
                t.start = datetime.datetime.strptime(props["DTSTART"], DateTimeFormat)
            self.addTask(t, False)
        else:
            print "error, SUMMARY and/or X-PROJECTNUM not present in VTODO"

    def parseProps(self, f, end):
        props = {}
        for line in f:
            if line == end:
                break
            else:
                if ':' in line:
                    prop = line[:line.find(':')]
                    val = line[line.find(':')+1:-1]
                    props[prop] = val
                else:
                    print "error, no : in line", line
        return props

    def save(self, filename=None):
        if filename is None:
            filename = self.currentfile
        with open(filename+"_tmp", "w") as f:
            f.write("pebble-do:1\n")
            for pn in self.projects:
                f.write("BEGIN:PPROJECT\n")
                f.write("NUM:"+str(pn)+"\n")
                f.write("DESC:"+self.projects[pn]+"\n")
                f.write("END:PPROJECT\n")
            for t in self.tasks:
                f.write("BEGIN:VTODO\n")
                f.write("SUMMARY:"+t.desc+"\n")
                f.write("X-PROJECTNUM:"+str(t.projectnum)+"\n")
                if t.tags is not '':
                    f.write("CATEGORIES:"+t.tags+"\n")
                if t.created is not None:
                    f.write("CREATED:"+t.created.strftime(DateTimeFormat)+"\n")
                if t.completed is not None:
                    f.write("COMPLETED:"+t.completed.strftime(DateTimeFormat)+"\n")
                if t.start is not None:
                    f.write("DTSTART:"+t.start.strftime(DateTimeFormat)+"\n")
                if t.due is not None:
                    f.write("DUE:"+t.due.strftime(DateTimeFormat)+"\n")
                f.write("END:VTODO\n")
        try:
            os.rename(filename, filename+"_old")
        except OSError,e:
            if e.errno != 2:
                raise
        os.rename(filename+"_tmp", filename)

    def mergeTags(self, tlist):
        for t in tlist:
            if t != '' and not t in self.alltags:
                self.alltags.append(t)

