/*
 * Copyright (c) 2009 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Walter Guerra <walter.guerra@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtk.h>
#include <glib.h>
#include <glib/gi18n.h>
#include <hildon/hildon.h>
#include <string.h>

#include "nm_file_dialog.h"
#include "nm_settings.h"
#include "nm_file_dialog.h"


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

enum
{
   PATH_COLUMN,
   N_COLUMNS
};

enum
{
   NM_FILE_DIALOG_BUTTON_OK,
   NM_FILE_DIALOG_BUTTON_NEW,
   NM_FILE_DIALOG_BUTTON_DELETE
};

typedef struct _NmFileDialogPrivate NmFileDialogPrivate;

struct _NmFileDialogPrivate
{
    GtkWidget *treeview;
    GtkWidget *ok_button;
    GtkWidget *new_button;
    GtkWidget *delete_button;
    
    gchar *group;
    gchar *env;
    gint type;
    
    gboolean delete_active;
};

#define NM_FILE_DIALOG_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_FILE_DIALOG_TYPE, NmFileDialogPrivate))

static void nm_file_dialog_class_init (NmFileDialogClass *klass);
static void nm_file_dialog_init       (NmFileDialog *self);
static void nm_file_dialog_dispose    (GObject *object);
static void nm_file_dialog_finalize   (GObject *object);

G_DEFINE_TYPE (NmFileDialog, nm_file_dialog, GTK_TYPE_DIALOG)

static GtkWidget* nm_file_dialog_create_tree_view (NmFileDialog *);
static void nm_file_dialog_add_device (NmFileDialog *, gchar *);
static void nm_file_dialog_del_env (NmFileDialog *);
static void nm_file_dialog_new_dialog(NmFileDialog *);
static void nm_file_dialog_apply_config(NmFileDialog *);
static void nm_file_dialog_row_deleted_cb(GtkTreeModel  *, GtkTreePath *, gpointer);
static void nm_file_dialog_row_inserted_cb(GtkTreeModel  *, GtkTreePath *, GtkTreeIter *, gpointer);
static gint nm_file_dialog_get_number_of_items(GtkTreeModel *);
static void nm_file_dialog_new_file_dialog(NmFileDialog *);
static void nm_file_dialog_new_text_dialog(NmFileDialog *);

static void
nm_file_dialog_class_init (NmFileDialogClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmFileDialogPrivate));

    object_class->dispose = nm_file_dialog_dispose;
    object_class->finalize = nm_file_dialog_finalize;
}

static void
nm_file_dialog_init (NmFileDialog *self)
{
    GtkWidget *scroll;
    NmFileDialogPrivate *priv;

    priv = NM_FILE_DIALOG_GET_PRIVATE(self); 

    // Create the window
/*    gtk_window_set_title (GTK_WINDOW (self), "Folders" );*/
    gtk_window_resize (GTK_WINDOW (self), 100, 300);

    priv->treeview = nm_file_dialog_create_tree_view (self);




    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           priv->treeview);

    gtk_widget_show_all(scroll);

    // Add the content area
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox),
                    scroll, TRUE, TRUE,
                    6 /* padding */);

    // Add buttons
    priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (self), _("OK"), NM_FILE_DIALOG_BUTTON_OK);
    priv->new_button = gtk_dialog_add_button (GTK_DIALOG (self), _("New"), NM_FILE_DIALOG_BUTTON_NEW);
    priv->delete_button = gtk_dialog_add_button (GTK_DIALOG (self), _("Delete"), NM_FILE_DIALOG_BUTTON_DELETE);


    g_signal_connect (G_OBJECT (gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))),
                       "row-deleted",
                       G_CALLBACK (nm_file_dialog_row_deleted_cb), self);

    g_signal_connect (G_OBJECT (gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))),
                       "row-inserted",
                       G_CALLBACK (nm_file_dialog_row_inserted_cb), self);

    // Check if there is only one item to disable the delete button
    if (nm_file_dialog_get_number_of_items(gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview))) == 1) {
        gtk_widget_set_sensitive(priv->delete_button, FALSE);
        priv->delete_active = FALSE;
    }
    else {
        priv->delete_active = TRUE;
    }
}

static void
nm_file_dialog_dispose (GObject *object)
{

}

static void
nm_file_dialog_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_file_dialog_parent_class)->finalize (object);
}


GtkWidget*
nm_file_dialog_new (gchar *env, gchar *group, gint type)
{ 
    GtkWidget *self = g_object_new (NM_FILE_DIALOG_TYPE, NULL);
    
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE(self);
    
    priv->group = g_strdup(group);
    priv->env = g_strdup(env);
    priv->type = type;

    return self;
}

void
nm_file_dialog_show_dialog(NmFileDialog *self) 
{
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE(self); 
    
    if (priv->type == NM_FILE_DIALOG_FOLDER_DIALOG) {
        gtk_window_set_title (GTK_WINDOW (self), _("Folders") );
    }
    else {
        gtk_window_set_title (GTK_WINDOW (self), _("Allowed IPs") );
    }
    
    // Add the items to the treeview
    gchar **list;
    gsize length;

    list = nm_file_dialog_get_list(self, &length);

    int i;
    for (i = 0; i < length; i++) {
            nm_file_dialog_add_device(self, list[i]);
    }
    
    while (TRUE) {
        gint ret = gtk_dialog_run( GTK_DIALOG(self) );
        
        g_debug("[%s] - Dialog ret: %d", __FUNCTION__, ret);
        
        if (ret == NM_FILE_DIALOG_BUTTON_OK) {
            nm_file_dialog_apply_config(self);
            gtk_widget_hide(GTK_WIDGET(self));
            
            break;
        }
        else if (ret == NM_FILE_DIALOG_BUTTON_DELETE) {
            // Remove the item from the tree
            nm_file_dialog_del_env(self);
        }
        else if (ret == NM_FILE_DIALOG_BUTTON_NEW) {
            // Add an item to the view
            nm_file_dialog_new_dialog(self);
        }
    }
}

static GtkWidget*
nm_file_dialog_create_tree_view (NmFileDialog *self)
{
    GtkWidget *treeview;
    GtkTreeStore *store;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *column;
    NmFileDialogPrivate *priv;

    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    store = gtk_tree_store_new (N_COLUMNS,
                                G_TYPE_STRING);

    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

    /* Creates the render */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Device",
                                                       renderer,
                                                       "text", PATH_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    /* Setup the selection handler */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    
    return treeview;
}

static void
nm_file_dialog_add_device (NmFileDialog *self, gchar *name)
{
    GtkTreeModel *model;
    NmFileDialogPrivate *priv;
    GtkTreeIter iter;

    priv = NM_FILE_DIALOG_GET_PRIVATE (self);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->treeview));

    if (model != NULL) {
        gtk_tree_store_append (GTK_TREE_STORE(model), &iter, NULL);
        gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                            PATH_COLUMN, name,
                            -1);
    

    }
}

static void 
nm_file_dialog_del_env (NmFileDialog *self) 
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    GtkTreeSelection *selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(priv->treeview));
    
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);
    }
    
    if (iter.user_data != NULL) {
        gtk_tree_selection_select_iter(selection, &iter);
    }
    else {
        gtk_tree_model_get_iter_first(model, &iter);
        gtk_tree_selection_select_iter(selection, &iter);
    }
    
}

static void
nm_file_dialog_apply_config(NmFileDialog *self)
{
    g_debug("%s", __FUNCTION__);
    
    int ret;
    // Get the items from the treeview
    NmSettings *settings = nm_settings_new();
    
    GtkTreeIter iter;
    GtkTreeModel *model;
    
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(priv->treeview));
    
    ret = gtk_tree_model_get_iter_first(model, &iter);
    
    int i = 0;
    gchar *list[30];
    while (ret == TRUE) {
        gtk_tree_model_get(model, &iter, 
                           PATH_COLUMN, &list[i],
                           -1 );
                           
        g_debug("item[%d]: %s", i, list[i]);
        i++;

        ret = gtk_tree_model_iter_next(model, &iter);
    }
    
    // Save the settings
    if (priv->type == NM_FILE_DIALOG_FOLDER_DIALOG) {
        nm_settings_set_string_list(settings, priv->env, priv->group, NM_FILE_DIALOG_PATHS, list, i);
    }
    else {
        nm_settings_set_string_list(settings, priv->env, priv->group, NM_FILE_DIALOG_IPS, list, i);
    }
    
    
    // Apply the config
    nm_settings_save(settings);
    
}

gchar** 
nm_file_dialog_get_list(NmFileDialog *self, gsize *length) 
{
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    g_debug("[%s] - env: %s", __FUNCTION__, priv->env);
    
    NmSettings *settings = nm_settings_new();
    if (!nm_settings_start (settings)) {
        return NULL;
    }
    gchar **list;
    
    if (priv->type == NM_FILE_DIALOG_FOLDER_DIALOG) {
        list = nm_settings_get_string_list(settings, priv->env, priv->group, NM_FILE_DIALOG_PATHS, length);
    }
    else {
        list = nm_settings_get_string_list(settings, priv->env, priv->group, NM_FILE_DIALOG_IPS, length);
    }
    
    if (list == NULL) {
        g_debug("list is null");

        (*length) = 0;

        return NULL;
    }
    g_debug("list is NOT null");
    return list;
}

static void 
nm_file_dialog_new_dialog(NmFileDialog *self) 
{
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    if (priv->type == NM_FILE_DIALOG_FOLDER_DIALOG) {
        nm_file_dialog_new_file_dialog(self);
    }
    else {
        nm_file_dialog_new_text_dialog(self);
    }
    
}

static void
nm_file_dialog_new_file_dialog(NmFileDialog *self) 
{
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    GtkWidget *dialog_new = gtk_file_chooser_dialog_new(_("Choose a Folder"),
                                                        GTK_WINDOW(self),
                                                        GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
                                                        GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                                        GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
                                                        NULL);
    
    gtk_window_resize (GTK_WINDOW(dialog_new), 550, 350);
    
    gint ret = gtk_dialog_run( GTK_DIALOG(dialog_new) );
    if (ret == GTK_RESPONSE_ACCEPT) {
         gchar *selected_uri = NULL;
         gchar *new_entry = NULL;
         selected_uri = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER (dialog_new));;
         if (g_str_has_prefix(selected_uri, "file://")) {
             new_entry = g_strndup(selected_uri+7, strlen(selected_uri)-7);
         }
         else {
             new_entry = g_strdup(selected_uri);
         }
         nm_file_dialog_add_device(self, new_entry);
         g_free(new_entry);
         g_free(selected_uri);
    }

    gtk_widget_destroy (dialog_new);    
}

static void
nm_file_dialog_new_text_dialog(NmFileDialog *self) 
{
    NmFileDialogPrivate *priv;
    
    priv = NM_FILE_DIALOG_GET_PRIVATE (self);
    
    GtkWidget *dialog_new = gtk_dialog_new();
    
    gtk_window_set_title (GTK_WINDOW (dialog_new), _("Enter the IP") );
    
    GtkWidget *entry = gtk_entry_new();
    gtk_entry_set_width_chars(GTK_ENTRY(entry), 10);
    
    gtk_widget_show_all(entry);
    
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog_new)->vbox),
                    entry, TRUE, TRUE,
                    6 /* padding */);
    
    gtk_dialog_add_button (GTK_DIALOG (dialog_new), _("OK"), 1);
    gtk_dialog_add_button (GTK_DIALOG (dialog_new), _("Cancel"), 2);
    
    gint ret = gtk_dialog_run( GTK_DIALOG(dialog_new) );
    if (ret == 1) {
         const gchar *new_entry = gtk_entry_get_text (GTK_ENTRY(entry));
         if (strcmp(new_entry, "")) {
            nm_file_dialog_add_device(self, new_entry);
         }
    }

    gtk_widget_destroy (dialog_new);    
}

static void 
nm_file_dialog_row_deleted_cb(GtkTreeModel *model, GtkTreePath *path, gpointer self)
{
    NmFileDialogPrivate *priv = NM_FILE_DIALOG_GET_PRIVATE(self); 
   
    gint size = nm_file_dialog_get_number_of_items(model);

    if (size == 0) {
        gtk_widget_set_sensitive (priv->delete_button, FALSE);
        priv->delete_active = FALSE;
    }

}

static void 
nm_file_dialog_row_inserted_cb(GtkTreeModel *model, GtkTreePath *path, GtkTreeIter *iter, gpointer self)
{
    NmFileDialogPrivate *priv = NM_FILE_DIALOG_GET_PRIVATE(self); 
    
    if (priv->delete_active == FALSE) {
        gtk_widget_set_sensitive (priv->delete_button, TRUE);
        priv->delete_active = TRUE;
    }
}

static gint
nm_file_dialog_get_number_of_items(GtkTreeModel *model) 
{
    GtkTreeIter iter;
    gboolean valid;
    gint size = 0;
    
    valid = gtk_tree_model_get_iter_first (model, &iter);
    while(valid) {
        size++;
        valid = gtk_tree_model_iter_next (model, &iter);
    }
    
    return size;  
}

