/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Marcos Morais   <morais@dee.ufcg.edu.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *          Walter Guerra <walter.guerra@signove.com>
 *          Andre Rodrigues <andre.rodrigues@signove.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <stdlib.h>
#include <string.h>

#include "nm_wlan_utils.h"
#include "nm_wlan_setup.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

static DBusConnection* nm_wlan_utils_create_connection ();

typedef struct _NmWlanUtilsPrivate NmWlanUtilsPrivate;

struct _NmWlanUtilsPrivate
{
    DBusConnection *connection;
};

#define NM_WLAN_UTILS_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_WLAN_UTILS_TYPE, NmWlanUtilsPrivate))

static void nm_wlan_utils_class_init (NmWlanUtilsClass *klass);
static void nm_wlan_utils_init       (NmWlanUtils *self);
static void nm_wlan_utils_dispose    (GObject *object);
static void nm_wlan_utils_finalize   (GObject *object);

G_DEFINE_TYPE (NmWlanUtils, nm_wlan_utils, G_TYPE_OBJECT)

static void
nm_wlan_utils_class_init (NmWlanUtilsClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmWlanUtilsPrivate));

    object_class->dispose = nm_wlan_utils_dispose;
    object_class->finalize = nm_wlan_utils_finalize;
}

static void
nm_wlan_utils_init (NmWlanUtils *self)
{
    NmWlanUtilsPrivate *priv;

    priv = NM_WLAN_UTILS_GET_PRIVATE (self);

    priv->connection = NULL;
}

static void
nm_wlan_utils_dispose (GObject *object)
{
    NmWlanUtilsPrivate *priv;

    priv = NM_WLAN_UTILS_GET_PRIVATE (object);

    if (priv->connection) {
        dbus_connection_unref (priv->connection);
    }

    G_OBJECT_CLASS (nm_wlan_utils_parent_class)->dispose (object);
}

static void
nm_wlan_utils_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_wlan_utils_parent_class)->finalize (object);
}

static DBusConnection*
nm_wlan_utils_create_connection ()
{
    DBusConnection *connection;

    connection = dbus_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (!connection) {
        g_warning ("D-Bus Connection not created");
    }

    return connection;
}

/************************************************************************
 Tool methods to check/activate/deactivate wlan
************************************************************************/

NmWlanUtils*
nm_wlan_utils_new (void)
{
    return g_object_new (NM_WLAN_UTILS_TYPE, NULL);
}

gboolean
nm_wlan_utils_is_active ()
{    
    DBusGProxy *p_conn = NULL;
    gchar *conn_name;
    gchar *ip_addr; 
    gchar *netmask;
    gchar *gateway;
    gchar *tmp1;
    gchar *tmp2;
    GError *error = NULL;

    DBusGConnection *connection;
    
    // Get a D-Bus connection
    // TODO Move to the initialization
    connection = dbus_g_bus_get (DBUS_BUS_SYSTEM, NULL);
    if (!connection) {
        g_warning ("D-Bus Connection not created");
        printf("error creating connection");
    }
    // TODO
    
    p_conn = dbus_g_proxy_new_for_name (connection,
                                        DBUS_SERVICE_ICD,
                                        DBUS_PATH_ICD,
                                        DBUS_INTERFACE_ICD);
        
    if (!dbus_g_proxy_call_with_timeout (p_conn, ICD_GETIPINFO, 1 * 1000, &error, G_TYPE_INVALID,
                            G_TYPE_STRING, &conn_name, 
                            G_TYPE_STRING, &ip_addr,
                            G_TYPE_STRING, &netmask,
                            G_TYPE_STRING, &gateway,
                            G_TYPE_STRING, &tmp1,
                            G_TYPE_STRING, &tmp2,
                            G_TYPE_INVALID) ) {
    
    
        if (error->domain == DBUS_GERROR &&
            error->code == DBUS_GERROR_REMOTE_EXCEPTION) {

            g_warning ("Caught remote method exception %s: %s\n\t%s: %d\n",
                        dbus_g_error_get_name (error), error->message,
                        __FILE__, __LINE__);
        } else {
            
            g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__,
                        __LINE__);
        }

        g_error_free (error);
        g_object_unref (p_conn);
    
        return FALSE;
    }
    
    
    g_free(ip_addr); 
    g_free(netmask);
    g_free(gateway);
    g_free(tmp1);
    g_free(tmp2);
    g_object_unref (p_conn);

    // Compare the wlan name of the config file with the current one
    if ( strcmp("devel_adhoc", conn_name) == 0 ) {
        g_free(conn_name);
        return TRUE;
    }
    else {
        g_free(conn_name);
        return FALSE;
    }
    
}



gboolean
nm_wlan_utils_activate (void)
{
    gint status = 1;

    status = system("dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true");
    sleep(2);
    status = system("dbus-send --type=method_call --print-reply --system --dest=com.nokia.icd /com/nokia/icd com.nokia.icd.connect string:devel_adhoc uint32:0");
    
    return status == 0;
}

gboolean
nm_wlan_utils_deactivate (void)
{
    gint status = 1;

    status = system("dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true");

    return status == 0;
}

/************************************************************************
Tool methods to start d-bus network connection.
************************************************************************/

/*
gboolean
nm_wlan_utils_activate_dbus (NmWlanUtils *self)
{
    GError *error = NULL;
    DBusGProxy *proxy = NULL;
    NmWlanUtilsPrivate *priv;

    priv = NM_WLAN_UTILS_GET_PRIVATE (self);

    if (priv->connection == NULL) {
        priv->connection = nm_wlan_utils_create_connection(self);
    }

  nm_wlan_utils_deactivate_dbus (self);
  // nm_wlan_utils_deactivate ();

    // Create a proxy object for the "bus driver" (name "com.nokia.icd")
    proxy = dbus_g_proxy_new_for_name (priv->connection,
                                        DBUS_SERVICE_ICD,
                                        DBUS_PATH_ICD,
                                        DBUS_INTERFACE_ICD);

    // Call connect method, wait for reply
    if (!dbus_g_proxy_call (proxy, ICD_CONNECT, &error,
                             G_TYPE_STRING, "devel_adhoc",
                             DBUS_TYPE_UINT32, 0,
                             G_TYPE_INVALID, G_TYPE_INVALID)) {

        g_warning ("Error: %s\n\t%s: %d\n", error->message, __FILE__, __LINE__);
        g_error_free (error);
        g_object_unref (proxy);

        return FALSE;
    }

    g_object_unref (proxy);

    return TRUE;
}
*/

gboolean
nm_wlan_utils_deactivate_dbus (NmWlanUtils *self)
{
    DBusMessage *msg;
    NmWlanUtilsPrivate *priv;

    priv = NM_WLAN_UTILS_GET_PRIVATE (self);

    if (priv->connection == NULL) {

        priv->connection = nm_wlan_utils_create_connection ();
    }

    // dbus-send --system --dest=com.nokia.icd /com/nokia/icd_ui com.nokia.icd_ui.disconnect boolean:true

    // create the signal and check for errors
    msg = dbus_message_new_signal(DBUS_PATH_ICD_UI,
                                   DBUS_INTERFACE_ICD_UI,
                                   ICD_UI_DISCONNECT);
    if (msg == NULL) {

        return FALSE;
    }

    // FIXME - crash here
    dbus_message_append_args (msg, DBUS_TYPE_BOOLEAN, TRUE, DBUS_TYPE_INVALID);

    // send the message and flush the connection
    if (!dbus_connection_send(priv->connection, msg, NULL)) {

        return FALSE;
    }

    dbus_connection_flush (priv->connection);

    // free the message
    dbus_message_unref(msg);

    return TRUE;
}

