/*
 * Copyright (c) 2008 Nokia Corporation
 * Contact: integration@maemo.org
 *
 * Authors: Hallyson Melo <hallyson.melo@indt.org.br>
 *          Kyller Gorgonio <kyllercg@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,
 * USA.
 */

#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <hildon/hildon.h>

#include "nm_bt_search_dialog.h"
#include "nm_bt_setup.h"
#include "nm_bt_utils.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

enum
{
   CONNECTION_DEVICE_COLUMN,
   CONNECTION_MAC_COLUMN,
   N_COLUMNS
};

static GtkWidget* nm_bt_search_dialog_create_tree_view (NmBtSearchDialog *);
static void nm_bt_search_dialog_selection_cb (GtkTreeSelection *selection, gpointer user_data);
static void nm_bt_search_dialog_add_device (NmBtSearchDialog *, gchar *, gchar *);
static gboolean nm_bt_search_dialog_name_is_in_tree(NmBtSearchDialog *, gchar *);

static void nm_bt_search_dialog_device_found_cb (NmBtUtils *bt_utils, gchar *name,
                                                 gchar *addr, gpointer user_data);
static void nm_bt_search_dialog_discovery_completed_cb (NmBtUtils *bt_utils, gpointer user_data);
static void nm_bt_search_dialog_disconnect_signals (NmBtSearchDialog *self);

typedef struct _NmBtSearchDialogPrivate NmBtSearchDialogPrivate;

struct _NmBtSearchDialogPrivate
{
    GtkWidget *treeview;
    GtkTreeStore *store;
    GtkWidget *ok_button;

    NmBtUtils *bt_utils;

    /* Signals handler id */
    gulong found_handler_id;
    gulong completed_handler_id;
};

#define NM_BT_SEARCH_DIALOG_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), NM_BT_SEARCH_DIALOG_TYPE, NmBtSearchDialogPrivate))

static void nm_bt_search_dialog_class_init (NmBtSearchDialogClass *klass);
static void nm_bt_search_dialog_init       (NmBtSearchDialog *self);
static void nm_bt_search_dialog_dispose    (GObject *object);
static void nm_bt_search_dialog_finalize   (GObject *object);

G_DEFINE_TYPE (NmBtSearchDialog, nm_bt_search_dialog, GTK_TYPE_DIALOG)

static void
nm_bt_search_dialog_class_init (NmBtSearchDialogClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS (klass);

    g_type_class_add_private (klass, sizeof (NmBtSearchDialogPrivate));

    object_class->dispose = nm_bt_search_dialog_dispose;
    object_class->finalize = nm_bt_search_dialog_finalize;
}

static void
nm_bt_search_dialog_init (NmBtSearchDialog *self)
{
    GtkWidget *scroll;
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    gtk_window_set_title (GTK_WINDOW (self), _("Select a device") );
    gtk_window_resize (GTK_WINDOW (self), 100, 250);

    priv->treeview = nm_bt_search_dialog_create_tree_view (self);

    scroll = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll), 
                                    GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport (GTK_SCROLLED_WINDOW (scroll),
                                           priv->treeview);

    /* Show all widgets */
    gtk_widget_show_all (scroll);

    /* Add the hbox to the vbox */
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (self)->vbox),
                        scroll, TRUE, TRUE,
                        6 /* padding */);

    /* Add the buttons */
    priv->ok_button = gtk_dialog_add_button (GTK_DIALOG (self),
                                             _("Select"), GTK_RESPONSE_OK);
    gtk_widget_set_sensitive (priv->ok_button, FALSE);

    gtk_dialog_add_button (GTK_DIALOG (self),
                           _("Cancel"), GTK_RESPONSE_CANCEL);
}

static void
nm_bt_search_dialog_dispose (GObject *object)
{
    NmBtSearchDialogPrivate *priv;
    GtkWidget *self = g_object_new (NM_BT_SEARCH_DIALOG_TYPE, NULL);

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    nm_bt_search_dialog_disconnect_signals (NM_BT_SEARCH_DIALOG (self));

    if (priv->bt_utils) {
        g_object_unref (priv->bt_utils);
        priv->bt_utils = NULL;
    }

    G_OBJECT_CLASS (nm_bt_search_dialog_parent_class)->dispose (object);
}

static void
nm_bt_search_dialog_finalize (GObject *object)
{
    G_OBJECT_CLASS (nm_bt_search_dialog_parent_class)->finalize (object);
}

GtkWidget*
nm_bt_search_dialog_new (NmBtUtils *bt_utils)
{
    NmBtSearchDialogPrivate *priv;
    GtkWidget *self = g_object_new (NM_BT_SEARCH_DIALOG_TYPE, NULL);

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    priv->bt_utils = bt_utils;
    g_object_ref (priv->bt_utils);
    
    return self;
}

static GtkWidget*
nm_bt_search_dialog_create_tree_view (NmBtSearchDialog *self)
{
    GtkWidget *treeview;
    GtkTreeStore *store;
    GtkCellRenderer *renderer;
    GtkTreeSelection *selection;
    GtkTreeViewColumn *column;
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);
    store = gtk_tree_store_new (N_COLUMNS,
                                G_TYPE_STRING,
                                G_TYPE_STRING);

    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));

    /* Creates the render */
    renderer = gtk_cell_renderer_text_new ();
    column = gtk_tree_view_column_new_with_attributes ("Device",
                                                       renderer,
                                                       "text", CONNECTION_DEVICE_COLUMN,
                                                       NULL);
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);

    /* Setup the selection handler */
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview));
    gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
    g_signal_connect (G_OBJECT (selection), "changed",
                      G_CALLBACK(nm_bt_search_dialog_selection_cb),
                      self);

    return treeview;
}

static gboolean 
nm_bt_search_dialog_name_is_in_tree(NmBtSearchDialog *self, gchar *name)
{
    GtkTreeModel *model;
    NmBtSearchDialogPrivate *priv;
    GtkTreeIter iter;
    gboolean valid;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->treeview));

    if (model != NULL) {
        valid = gtk_tree_model_get_iter_first (model, &iter);
        while (valid) {
            gchar *iter_name;
            gchar *mac;
            gtk_tree_model_get(model, &iter,
                               CONNECTION_DEVICE_COLUMN, &iter_name,
                               CONNECTION_MAC_COLUMN, &mac,
                               -1);
            if (!strcmp(iter_name, name)) {
                g_free(iter_name);
                g_free(mac);
                return TRUE;
            }
            
            g_free(iter_name);
            g_free(mac);
            
            valid = gtk_tree_model_iter_next (model, &iter);
        }
    }
    return FALSE; 
}

static void
nm_bt_search_dialog_add_device (NmBtSearchDialog *self, gchar *name, gchar *mac)
{
    GtkTreeModel *model;
    NmBtSearchDialogPrivate *priv;
    GtkTreeIter iter;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    model = gtk_tree_view_get_model (GTK_TREE_VIEW (priv->treeview));

    if (model != NULL) {
        gtk_tree_store_append (GTK_TREE_STORE(model), &iter, NULL);
        gtk_tree_store_set (GTK_TREE_STORE(model), &iter,
                            CONNECTION_DEVICE_COLUMN, name,
                            CONNECTION_MAC_COLUMN, mac,
                            -1);
    }
}

gchar*
nm_bt_search_dialog_get_selected_addr (NmBtSearchDialog *self)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;
    gchar *btaddr = NULL;
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(priv->treeview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        gtk_tree_model_get (model, &iter, CONNECTION_MAC_COLUMN, &btaddr, -1);
    }

    return btaddr;
}

gchar*
nm_bt_search_dialog_get_selected_name (NmBtSearchDialog *self)
{
    GtkTreeIter iter;
    GtkTreeModel *model;
    GtkTreeSelection *selection;
    gchar *name = NULL;
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);
    selection = gtk_tree_view_get_selection (GTK_TREE_VIEW(priv->treeview));
    if (gtk_tree_selection_get_selected (selection, &model, &iter)) {
        gtk_tree_model_get (model, &iter, CONNECTION_DEVICE_COLUMN, &name, -1);
    }

    return name;
}

gboolean
nm_bt_search_dialog_start_discovery (NmBtSearchDialog *self)
{
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    priv->found_handler_id = g_signal_connect (priv->bt_utils, "device-found",
                                               G_CALLBACK(nm_bt_search_dialog_device_found_cb), self);
    priv->completed_handler_id = g_signal_connect (priv->bt_utils, "discovery-completed",
                                                   G_CALLBACK(nm_bt_search_dialog_discovery_completed_cb), self);

    nm_bt_utils_start_device_discovery (priv->bt_utils);

    return FALSE;
}

gboolean
nm_bt_search_dialog_stop_discovery (NmBtSearchDialog *self)
{
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    g_debug ("[%s] Device discovery stopped", __FUNCTION__);

    nm_bt_search_dialog_disconnect_signals (self);

    nm_bt_utils_stop_device_discovery (priv->bt_utils);

    return TRUE;
}

static void
nm_bt_search_dialog_selection_cb (GtkTreeSelection *selection, gpointer user_data)
{
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (user_data);

    gtk_widget_set_sensitive (priv->ok_button, TRUE);
}

static void
nm_bt_search_dialog_device_found_cb (NmBtUtils *bt_utils, gchar *name, gchar *addr, gpointer user_data)
{
    if ((name != NULL) && (strlen(name) != 0) && 
        !nm_bt_search_dialog_name_is_in_tree(NM_BT_SEARCH_DIALOG(user_data), name)) {
            
        nm_bt_search_dialog_add_device (NM_BT_SEARCH_DIALOG (user_data), name, addr);
    }  
}

static void
nm_bt_search_dialog_discovery_completed_cb (NmBtUtils *bt_utils, gpointer user_data)
{
    hildon_banner_show_information (GTK_WIDGET (user_data), NULL, _("Bluetooth Search Completed"));
}

static void
nm_bt_search_dialog_disconnect_signals (NmBtSearchDialog *self)
{
    NmBtSearchDialogPrivate *priv;

    priv = NM_BT_SEARCH_DIALOG_GET_PRIVATE (self);

    /* Disconnect the signals */
    if (priv->found_handler_id) {
        g_signal_handler_disconnect (priv->bt_utils, priv->found_handler_id);
        priv->found_handler_id = 0;
    }

    if (priv->completed_handler_id) {
        g_signal_handler_disconnect (priv->bt_utils, priv->completed_handler_id);
        priv->completed_handler_id = 0;
    }
}

