#!/bin/sh
#
# Copyright (C) 2007 Nokia Corporation. All rights reserved.
#
# Contact: Jukka Rissanen <jukka.rissanen@nokia.com>
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# version 2 as published by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
# 02110-1301 USA

DEBUG=0
ENABLE=0
IAP=""
USE_CURRENT_IAP=0
MODULES_ONLY=""
INTERFACES=""
INTERFACES_SET=0
AUTOCONF=1
SET_DEFAULTS=0
ONLY=0
IAP_BASE=/system/osso/connectivity/IAP
IPV6_KEY=$IAP_BASE/ipv6_enabled
AUTOCONF_KEY=$IAP_BASE/ipv6_autoconf
MODULE_DIR=/mnt/initfs/lib/modules/`uname -r`
MODULES="
  ipv6 
  xfrm6_mode_transport
  xfrm6_mode_tunnel
  xfrm6_mode_ro
  tunnel6
  xfrm6_tunnel
  ip6_tunnel
"

IP6_TABLES="
  ip6_tables
  ip6table_filter
"


usage()
{
cat <<EOF
Usage: $0 [options]
    where options can be:
    --enable (activate IPv6 globally)
    --debug (activate debugging)
    --iap <IAP name to activate IPv6>
    --current (activate IPv6 for the current IAP)
    --modules (load modules)
    --interface <interfaces to use> (separate by spaces, example "wlan0 ppp0"), to clear the setting use empty interface list
    --noautoconf, do not use autoconfig addresses (autoconfig is enabled by default)
    --autoconf, use autoconfig addresses
    --default, set all IPv6 settings to factory defaults

without parameters the current settings are shown.
EOF
}


if [ $# -lt 1 ]; then
	echo "IPv6 modules currently loaded:"
	lsmod | awk '{ if ($1 ~ /6/) { print $0 } }'
	echo
	echo "IPv6 global status (overrides IAP specific setting):"
	VAL=`gconftool-2 -g $IPV6_KEY`
	if [ "$VAL" = 1 ]; then
	    echo "IPv6 is enabled."
	    DEF="enabled"
	    INFO=""
	else
	    echo "IPv6 is disabled."
	    DEF="disabled"
	    INFO=" when enabled"
	fi
	VAL=`gconftool-2 -g $AUTOCONF_KEY`
	if [ "$VAL" = 1 ]; then
	    echo "IPv6 autoconfiguring is enabled."
	else
	    echo "IPv6 autoconfiguring is disabled."
	fi
	echo
	echo "IPv6 status for each IAP:"
	for IAP in `gconftool-2 -R $IAP_BASE | grep $IAP_BASE | sed 's/://g'`
	do
	  VAL_IAP=`gconftool-2 -g $IAP/ipv6_enabled 2>/dev/null`
	  VAL_NAME=`gconftool-2 -g $IAP/name 2>/dev/null`
	  if [ -z "$VAL_IAP" ]; then
	      STR="status not set, using default ($DEF)"
	  else
	      if [ "$VAL_IAP" = 0 ]; then
		  STR=disabled
	      else
		  STR=enabled
	      fi
	  fi
	  echo "$IAP ($VAL_NAME): $STR"
	done
	echo
	echo "IPv6 is used for these network interfaces${INFO}:"
	IF=`gconftool-2 -g $IAP_BASE/ipv6_interfaces`
	echo $IF | sed -e 's/\[//g' -e 's/\]//g'
	echo
	exit 0
fi


while
        [ $# -gt 0 ]
do
        case $1 in
	    -e|-enable|--enable) ENABLE=1; shift;;
	    -d|-debug|--debug) DEBUG=1; shift;;
	    -i|-iap|--iap) IAP="$2"; ENABLE=1; shift 2;;
	    -c|-current|--current) IAP=current; USE_CURRENT_IAP=1; ENABLE=1; shift;;
	    -m|-modules|--modules) MODULES_ONLY=yes; shift;;
	    -o|-only|--only) ONLY=1; shift;;
	    -f|-interface|--interface|-interfaces|--interfaces) INTERFACES_SET=1; INTERFACES="$2"; ENABLE=1; shift 2;;
	    -n|-noauto|-noautoconf|--noautoconf) AUTOCONF=0; shift;;
	    -a|-auto|-autoconf|--autoconf) AUTOCONF=1; shift;;
	    -default|--default|-defaults|--defaults) SET_DEFAULTS=1; shift;;
	    
            *) usage; exit 1;;
        esac
done


if [ $SET_DEFAULTS -eq 1 ]; then
    IAP_LIST=`gconftool-2 -R $IAP_BASE | grep "^ /" | awk -F / '{ print $NF }' | sed 's/:$//'`
    for IAP in $IAP_LIST
    do
      ESCAPED=`gconf_escape "$IAP"`
      gconftool-2 -u "$IAP_BASE/$ESCAPED/ipv6_enabled"
    done
    gconftool-2 -u "$IAP_BASE/ipv6_enabled"
    gconftool-2 -u "$IAP_BASE/ipv6_autoconf"
    gconftool-2 -u "$IAP_BASE/ipv6_interfaces"
    echo "IPv6 settings cleared in gconf"
    exit
fi


check_module()
{
    MOD="$1"

    /sbin/lsmod | grep "$MOD" > /dev/null
    if [ $? != 0 ]; then
	[ $DEBUG -ne 0 ] && echo "Module $MOD not yet loaded"
	RET=0
    else
	[ $DEBUG -ne 0 ] && echo "Module $MOD already loaded"
	RET=1
    fi

    return $RET
}


check_module_file()
{
    MOD="$1"

    if [ -f $MODULE_DIR/$MOD.ko ]; then
	[ $DEBUG -ne 0 ] && echo "Module $MODULE_DIR/$MOD.ko exists"
	RET=1
    else
	[ $DEBUG -ne 0 ] && echo "Module $MODULE_DIR/$MOD.ko not found"
	RET=0
    fi

    return $RET
}


load_module()
{
    MOD="$1"
    FAILED=0

    check_module $MOD
    if [ $? -eq 0 ]; then
	check_module_file $MOD
	if [ $? -eq 1 ]; then	    
	    /sbin/insmod $MODULE_DIR/$MOD.ko
	    if [ $? -ne 0 ]; then
		FAILED=1
	    fi
	    [ $DEBUG -ne 0 ] && echo "insmod $MODULE_DIR/$MOD.ko returned $RET"
	else
	    FAILED=1
	fi
    fi

    return $FAILED
}


FAILURE=0

if [ "$MODULES_ONLY" = "yes" ]; then
  LOADED=""
  for mod in $MODULES
  do
    load_module $mod
    if [ $? -ne 0 ]; then
      echo "Cannot load $mod"
      FAILURE=1
    else
      LOADED="$LOADED$mod "
    fi
  done

  if [ ! -f /proc/net/ip6_tables_names ]; then
    for mod in $IP6_TABLES
    do
      load_module $mod
      if [ $? -ne 0 ]; then
	echo "Cannot load $mod"
	FAILURE=1
      else
	LOADED="$LOADED$mod "
      fi
    done
  fi

  if [ "$LOADED" != "" ]; then
    echo "Loaded modules: $LOADED"
  fi
  if [ $ONLY -eq 1 ]; then
    exit
  fi
fi


if [ $FAILURE -eq 0 ]; then
    gconftool-2 -t int -s "$IPV6_KEY" 1
    if [ $? -ne 0 ]; then
	echo "Cannot enable IPv6 in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $IPV6_KEY to 1"
    fi
fi


# What interfaces should use ipv6
if [ ! -z "$INTERFACES" ]; then
    IFACES="["
    for i in $INTERFACES
    do
      IFACES="${IFACES}$i,"
    done
    LEN=`expr length "$IFACES"`
    LEN=`expr $LEN - 1`
    IFACES=`echo "$IFACES" | cut -c -$LEN`
    IFACES="${IFACES}]"
    [ $DEBUG -ne 0 ] && echo "IPv6 interfaces: $INTERFACES"
    gconftool-2 -s -t list --list-type string "$IAP_BASE/ipv6_interfaces" "$IFACES"
else
    if [ $INTERFACES_SET = 1 ]; then
	[ $DEBUG -ne 0 ] && echo "Clearing IPv6 interfaces."

        # Clear the ipv6 setting
	gconftool-2 -u "$IAP_BASE/ipv6_interfaces"
    fi
fi


# Enable IPv6 for this IAP
if [ ! -z "$IAP" ]; then
    if [ "$USE_CURRENT_IAP" = 1 ]; then
	CURRENT=`gconftool-2 -g "$IAP_BASE/current"`
	if [ -z "$CURRENT" ]; then
	    echo "No current IAP found, activating globally."
	    ESCAPED=""
	else
	    ESCAPED=`gconf_escape "$CURRENT"`
	fi
    else
	ESCAPED=`gconf_escape "$IAP"`
    fi

    if [ ! -z "$ESCAPED" ]; then
	gconftool-2 -g "$IAP_BASE/$ESCAPED/type" 2>&1 | grep "No value" > /dev/null
	if [ $? -eq 0 ]; then
	    # No such escaped IAP found
	    echo "Cannot find $IAP ($ESCAPED) in gconf"
	    FAILURE=1
	else
	    IAP_IPV6_KEY="$IAP_BASE/$ESCAPED/ipv6_enabled"
	    gconftool-2 -t int -s "$IAP_IPV6_KEY" 1
	    if [ $? -ne 0 ]; then
		echo "Cannot enable IPv6 for $ESCAPED in gconf"
		FAILURE=1
	    else
		[ $DEBUG -ne 0 ] && echo "Set $IAP_IPV6_KEY to 1"
	    fi
	fi
    fi
fi


if [ "$AUTOCONF" = 1 ]; then
    # Turn on ipv6 autoconf globally
    gconftool-2 -t int -s "$AUTOCONF_KEY" 1
    if [ $? -ne 0 ]; then
	echo "Cannot enable IPv6 autoconf in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $AUTOCONF_KEY to 1"
    fi
else
    # Turn off ipv6 autoconf globally
    gconftool-2 -t int -s "$AUTOCONF_KEY" 0
    if [ $? -ne 0 ]; then
	echo "Cannot disable IPv6 autoconf in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $AUTOCONF_KEY to 0"
    fi
fi


if [ "$ENABLE" = 1 ]; then
    # Turn on ipv6 globally
    gconftool-2 -t int -s "$IPV6_KEY" 1
    if [ $? -ne 0 ]; then
	echo "Cannot enable IPv6 in gconf"
	FAILURE=1
    else
	[ $DEBUG -ne 0 ] && echo "Set $IPV6_KEY to 1"
    fi
fi

exit $FAILURE

