/*
 * This code was created by Jeff Molofee '99
 * (ported to Linux/SDL by Ti Leggett '01)
 * (ported to Maemo/SDL_gles by Till Harbaum '10)
 *
 * If you've found this code useful, please let me know.
 *
 * Visit Jeff at http://nehe.gamedev.net/
 *
 * or for port-specific comments, questions, bugreports etc.
 * email to leggett@eecs.tulane.edu or till@harbaum.org
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <SDL.h>
#include <SDL_image.h>
#include <SDL_gles.h>
#include <GLES/gl.h>
#include "glu.h"

/* screen width, height, and bit depth */
#define SCREEN_WIDTH  800
#define SCREEN_HEIGHT 480
#define SCREEN_BPP     16

/* Set up some booleans */
#define TRUE  1
#define FALSE 0

/* Number of stars */
#define NUM 50

/* This is our SDL surface */
SDL_Surface *surface;
SDL_GLES_Context *context;

/* Twinkling stars */
int twinkle = FALSE;

/* Define the star structure */
typedef struct
{
    int r, g, b;   /* Stars Color */
    GLfloat dist;  /* Stars Distance From Center */
    GLfloat angle; /* Stars Current Angle */
} star;

star stars[NUM];       /* Make an array of size 'NUM' of stars */

GLfloat zoom = -15.0f; /* Viewing Distance Away From Stars */
GLfloat tilt = 90.0f;  /* Tilt The View */

GLuint loop;           /* General Loop Variable */
GLuint texture[1];     /* Storage For One Texture */


/* function to release/destroy our resources and restoring the old desktop */
void Quit( int returnCode )
{
    SDL_GLES_DeleteContext(context);

    /* clean up the window */
    SDL_Quit( );

    /* and exit appropriately */
    exit( returnCode );
}

/* function to load in bitmap as a GL texture */
int LoadGLTextures( )
{
    /* Status indicator */
    int Status = FALSE;

    /* Create storage space for the texture */
    SDL_Surface *TextureImage[1]; 

    /* Load The Bitmap, Check For Errors, If Bitmap's Not Found Quit */
    if ( ( TextureImage[0] = IMG_Load( "/opt/nehegles/star.png" ) ) )
        {

	    /* Set the status to true */
	    Status = TRUE;

	    /* Create The Texture */
	    glGenTextures( 1, &texture[0] );

	    /* Load in texture */
	    /* Typical Texture Generation Using Data From The Bitmap */
	    glBindTexture( GL_TEXTURE_2D, texture[0] );

	    /* Linear Filtering */
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
	    glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

	    /* Generate The Texture */
	    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, TextureImage[0]->w,
			  TextureImage[0]->h, 0, GL_RGB,
			  GL_UNSIGNED_BYTE, TextureImage[0]->pixels );

        }

    /* Free up any memory we may have used */
    if ( TextureImage[0] )
	    SDL_FreeSurface( TextureImage[0] );

    return Status;
}

/* function to reset our viewport after a window resize */
int resizeWindow( int width, int height )
{
    /* Height / width ration */
    GLfloat ratio;
 
    /* Protect against a divide by zero */
    if ( height == 0 )
	height = 1;

    ratio = ( GLfloat )width / ( GLfloat )height;

    /* Setup our viewport. */
    glViewport( 0, 0, ( GLint )width, ( GLint )height );

    /* change to the projection matrix and set our viewing volume. */
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    /* Set our perspective */
    gluPerspective( 45.0f, ratio, 0.1f, 100.0f );

    /* Make sure we're chaning the model view and not the projection */
    glMatrixMode( GL_MODELVIEW );

    /* Reset The View */
    glLoadIdentity( );

    return( TRUE );
}

/* function to handle key press events */
void handleKeyPress( SDL_keysym *keysym )
{
    switch ( keysym->sym )
	{
	case SDLK_t:
	    /* 't' key was pressed
	     * this toggles the twinkling of the stars
	     */
	    twinkle = !twinkle;
	    break;
	case SDLK_UP:
	    /* Up arrow key was pressed
	     * this changes the tilt of the stars
	     */
	    tilt -= 0.5f;
	    break;
	case SDLK_DOWN:
	    /* Down arrow key was pressed
	     * this changes the tilt of the stars
	     */
	    tilt += 0.5f;
	    break;
	case SDLK_i:
	    /* 'i' key was pressed
	     * zoom into the scene
	     */
	    zoom += 0.2f;
	    break;
	case SDLK_o:
	    /* 'o' key was pressed
	     * zoom out of the scene
	     */
	    zoom -= 0.2f;
	    break;
	default:
	    break;
	}

    return;
}

/* general OpenGL initialization function */
int initGL( GLvoid )
{
    SDL_GLES_Init(SDL_GLES_VERSION_1_1);

    /* specify size of depth buffer (NEW) */
    SDL_GLES_SetAttribute(SDL_GLES_DEPTH_SIZE, 24);

    context = SDL_GLES_CreateContext();
    SDL_GLES_MakeCurrent(context);

    /* Load in the texture */
    if ( !LoadGLTextures( ) )
	return FALSE;

    /* Enable Texture Mapping */
    glEnable( GL_TEXTURE_2D );

    /* Enable smooth shading */
    glShadeModel( GL_SMOOTH );

    /* Set the background black */
    glClearColor( 0.0f, 0.0f, 0.0f, 0.0f );

    /* Depth buffer setup */
    glClearDepthf( 1.0f );

    /* Really Nice Perspective Calculations */
    glHint( GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST );

    /* Set The Blending Function For Translucency */
    glBlendFunc( GL_SRC_ALPHA, GL_ONE );

    /* Enable Blending */
    glEnable( GL_BLEND );

    /* Create A Loop That Goes Through All The Stars */
    for ( loop = 0; loop < NUM; loop++ )
	{
	    /* Start All The Stars At Angle Zero */
	    stars[loop].angle = 0.0f;

	    /* Calculate Distance From The Center */
	    stars[loop].dist = ( ( float )loop / NUM ) * 5.0f;
	    /* Give star[loop] A Random Red Intensity */
	    stars[loop].r = rand( ) % 256;
	    /* Give star[loop] A Random Green Intensity */
	    stars[loop].g = rand( ) % 256;
	    /* Give star[loop] A Random Blue Intensity */
	    stars[loop].b = rand( ) % 256;
        }

    glEnableClientState(GL_VERTEX_ARRAY);
    glEnableClientState(GL_TEXTURE_COORD_ARRAY);

    return( TRUE );
}

/* Here goes our drawing code */
int drawGLScene( GLvoid )
{
    static int spin = 0;

    /* These are to calculate our fps */
    static GLint T0     = 0;
    static GLint Frames = 0;

    /* Clear The Screen And The Depth Buffer */
    glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

    /* Select Our Texture */
    glBindTexture( GL_TEXTURE_2D, texture[0] );
    glLoadIdentity( );

    /* Loop Through All The Stars */
    for ( loop = 0; loop < NUM; loop++ )
        {
	    /* Reset The View Before We Draw Each Star */
	    glLoadIdentity( );
	    /* Zoom Into The Screen (Using The Value In 'zoom') */
	    glTranslatef( 0.0f, 0.0f, zoom );

	    /* Tilt The View (Using The Value In 'tilt') */
	    glRotatef( tilt, 1.0f, 0.0f, 0.0f );
	    /* Rotate To The Current Stars Angle */
	    glRotatef( stars[loop].angle, 0.0f, 1.0f, 0.0f );

	    /* Move Forward On The X Plane */
	    glTranslatef( stars[loop].dist, 0.0f, 0.0f );

	    /* Cancel The Current Stars Angle */
	    glRotatef( -stars[loop].angle, 0.0f, 1.0f, 0.0f );
	    /* Cancel The Screen Tilt */
	    glRotatef( -tilt, 1.0f, 0.0f, 0.0f );

	    /* Begin Drawing The Textured Quad */
	    const GLfloat quadVertices[] = {
	      -1.0f, -1.0f, 0.0f,
	       1.0f, -1.0f, 0.0f,
	      -1.0f,  1.0f, 0.0f,
	       1.0f,  1.0f, 0.0f,
	    };
	    
	    const GLfloat texCoords[] = {
	      0.0f, 0.0f,
	      1.0f, 0.0f,
	      0.0f, 1.0f,
	      1.0f, 1.0f,
	    };

	    /* Twinkling Stars Enabled */
	    if ( twinkle )
                {
		    /* Assign A Color Using Bytes */
		    glColor4ub( stars[( NUM - loop ) - 1].r,
				stars[( NUM - loop ) - 1].g,
				stars[( NUM - loop ) - 1].b, 255 );

		    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
		    glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
		    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);
                }
	    
	    /* Rotate The Star On The Z Axis */
	    glRotatef( spin, 0.0f, 0.0f, 1.0f );

	    /* Assign A Color Using Bytes */
	    glColor4ub( stars[loop].r, stars[loop].g, stars[loop].b, 255 );

	    glVertexPointer(3, GL_FLOAT, 0, quadVertices);
	    glTexCoordPointer(2, GL_FLOAT, 0, texCoords);
	    glDrawArrays(GL_TRIANGLE_STRIP, 0, 4);

	    /* Used To Spin The Stars */
	    spin += 0.01f;
	    /* Changes The Angle Of A Star */
	    stars[loop].angle += ( float )loop / NUM;
	    /* Changes The Distance Of A Star */
	    stars[loop].dist -= 0.01f;

	    /* Is The Star In The Middle Yet */
	    if ( stars[loop].dist < 0.0f )
                {
		    /* Move The Star 5 Units From The Center */
		    stars[loop].dist += 5.0f;
		    /* Give It A New Red Value */
		    stars[loop].r = rand( ) % 256;
		    /* Give It A New Green Value */
		    stars[loop].g = rand( ) % 256;
		    /* Give It A New Blue Value */
		    stars[loop].b = rand( ) % 256;
                }
        }

    /* Draw it to the screen */
    SDL_GLES_SwapBuffers( );

    /* Gather our frames per second */
    Frames++;
    {
	GLint t = SDL_GetTicks();
	if (t - T0 >= 5000) {
	    GLfloat seconds = (t - T0) / 1000.0;
	    GLfloat fps = Frames / seconds;
	    printf("%d frames in %g seconds = %g FPS\n", Frames, seconds, fps);
	    T0 = t;
	    Frames = 0;
	}
    }

    return( TRUE );
}

int main( int argc, char **argv )
{
    /* main loop variable */
    int done = FALSE;
    /* used to collect events */
    SDL_Event event;
    /* whether or not the window is active */
    int isActive = TRUE;

    /* initialize SDL */
    if ( SDL_Init( SDL_INIT_VIDEO ) < 0 )
	{
	    fprintf( stderr, "Video initialization failed: %s\n",
		     SDL_GetError( ) );
	    Quit( 1 );
	}

    SDL_EnableKeyRepeat(250, 20);

    surface = SDL_SetVideoMode(0, 0, 16, SDL_SWSURFACE | SDL_FULLSCREEN);
    SDL_ShowCursor(0);

    SDL_WM_SetCaption("NeHe OpenGL lesson 9", "NeHe9");

    /* initialize OpenGL */
    initGL( );

    /* resize the initial window */
    resizeWindow( SCREEN_WIDTH, SCREEN_HEIGHT );

    /* wait for events */
    while ( !done )
	{
	    /* handle the events in the queue */

	    while ( SDL_PollEvent( &event ) )
		{
		    switch( event.type )
			{
			case SDL_ACTIVEEVENT:
			    /* Something's happend with our focus
			     * If we lost focus or we are iconified, we
			     * shouldn't draw the screen
			     */
			    if ( event.active.gain == 0 )
				isActive = FALSE;
			    else
				isActive = TRUE;
			    break;			    
			case SDL_KEYDOWN:
			    /* handle key presses */
			    handleKeyPress( &event.key.keysym );
			    break;
			case SDL_MOUSEBUTTONDOWN:
 			case SDL_QUIT:
			    /* handle quit requests */
			    done = TRUE;
			    break;
			default:
			    break;
			}
		}

	    /* draw the scene */
	    if ( isActive )
		drawGLScene( );
	}

    /* clean ourselves up and exit */
    Quit( 0 );

    /* Should never get here */
    return( 0 );
}
