// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#ifndef MAINCONTROLLER_H
#define MAINCONTROLLER_H

#include "data/location.h"
#include "data/settings.h"

#include <QObject>

class MainWindow;
class CinemaSchedule;
class TheaterController;
class MovieController;
class LocationController;
class ActionController;
class SettingsController;
class ItemModelSortController;
class TheaterModel;
class MovieModel;
class ConnectivityManager;
class QAbstractItemModel;
class QSortFilterProxyModel;

class MainController : public QObject
{
    Q_OBJECT
public:
    MainController(MainWindow *main_window, Settings *settings, CinemaSchedule *cinema_schedule,
                   TheaterController *theater_controller, MovieController *movie_controller,
                   LocationController *location_controller, ActionController *action_controller,
                   SettingsController *settings_controller,
                   ItemModelSortController *sort_controller, QThread *search_worker);
    ~MainController();

public:
    void Run();
    void Search();
    Q_SLOT void Search(Location location);

private:
    Q_SLOT void NetworkConnected();
    Q_SLOT void NetworkDisconnected();
    Q_SLOT void NetworkError();

    Q_SLOT void SettingsChanged(const Settings &settings);

    Q_SLOT void SearchTheaters();
    Q_SLOT void SearchMovies();
    Q_SLOT void OpenAboutDialog();
    void CancelTheaterSearch();
    void CancelMovieSearch();

    Q_SLOT void TheatersSearchStarted(int search_task_id);
    Q_SLOT void TheatersSearchReply(int search_task_id, bool intermediate);
    Q_SLOT void TheatersSearchError(int search_task_id);
    Q_SLOT void TheatersSearchFinished(int search_task_id, bool success);

    void SortTheaters(bool intermediate, const char *);
    Q_SLOT void TheatersSortFinished(QAbstractItemModel *model, int search_task_id, bool intermediate);
    Q_SLOT void TheatersSortErrorFinished(QAbstractItemModel *model, int search_task_id, bool intermediate);
    void SetTheaterModel(QAbstractItemModel *model);

    Q_SLOT void MoviesSearchStarted(int search_task_id);
    Q_SLOT void MoviesSearchReply(int search_task_id, bool intermediate);
    Q_SLOT void MoviesSearchError(int search_task_id);
    Q_SLOT void MoviesSearchFinished(int search_task_id, bool success);

    void SortMovies(bool intermediate, const char *);
    Q_SLOT void MoviesSortFinished(QAbstractItemModel *model, int search_task_id, bool intermediate);
    Q_SLOT void MoviesSortErrorFinished(QAbstractItemModel *model, int search_task_id, bool intermediate);
    void SetMovieModel(QAbstractItemModel *model);

private:
    MainWindow *_main_window;
    Settings *_settings;
    CinemaSchedule *_cinema_schedule;
    TheaterController *_theater_controller;
    MovieController *_movie_controller;
    LocationController *_location_controller;
    ActionController *_action_controller;
    SettingsController *_settings_controller;
    ItemModelSortController *_sort_controller;
    QThread *_search_worker;
    ConnectivityManager *_connectivity_manager;
    int _current_theater_search_task_id;
    int _current_movies_search_task_id;
    TheaterModel *_theater_model;
    MovieModel *_movie_model;
    QSortFilterProxyModel *_theater_proxy_model;
    QSortFilterProxyModel *_movie_proxy_model;
    Settings _last_search_settings;    
};

#endif // MAINCONTROLLER_H
