// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "moviescheduledelegate.h"

#include "movieschedulemodel.h"
#include "data/scheduleentry.h"
#include "data/cinema.h"
#include "scheduleentryitem.h"
#include "utils/timeutils.h"
#include "ui/styleutils.h"

#include <QPainter>

MovieScheduleDelegate::MovieScheduleDelegate(QWidget *parent)
    : QStyledItemDelegate(parent)
{
}

void MovieScheduleDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option,
                                  const QModelIndex &index) const
{
    if (qVariantCanConvert<ScheduleEntryItem>(index.data())) {
        ScheduleEntryItem item = qVariantValue<ScheduleEntryItem>(index.data());
        painter->save();
        painter->translate(option.rect.x(), option.rect.y());
        if (item.IsScheduleDateItem()) {
            _day_painter.Paint(painter, option, item.GetScheduleDate());
        } else {
            QRect bounding_rect;
            QFont bigfont(option.font);
            bigfont.setPointSizeF(bigfont.pointSizeF() * 1.5);
            painter->setFont(bigfont);
            painter->setPen(StyleUtils::INSTANCE()->GetScheduleTimeColor(option.palette, item.GetScheduleEntry()->GetStartTime()));
            painter->drawText(0, 0, 1000, option.rect.height() - 4,
                              Qt::AlignTop, TimeUtils::ToTimeString(item.GetScheduleEntry()->GetStartTime()), &bounding_rect);
            int x = bounding_rect.right() + 8;
            painter->translate(x, 0);
            _cinema_painter.Paint(painter, option, item.GetScheduleEntry()->GetCinema());
        }
        painter->restore();
    } else {
        QStyledItemDelegate::paint(painter, option, index);
    }
}

QSize MovieScheduleDelegate::sizeHint(const QStyleOptionViewItem &option,
                                      const QModelIndex &index) const
{
    if (qVariantCanConvert<ScheduleEntryItem>(index.data())) {
        ScheduleEntryItem item = qVariantValue<ScheduleEntryItem>(index.data());
        if (item.IsScheduleDateItem()) {
            return _day_painter.GetSizeHint(option);
        }
        return _cinema_painter.GetSizeHint(option);
    } else {
        return QStyledItemDelegate::sizeHint(option, index);
    }
}
