// Copyright 2010 Jochen Becher
//
// This file is part of MovieSchedule.
//
// MovieSchedule is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// MovieSchedule is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with MovieSchedule.  If not, see <http://www.gnu.org/licenses/>.

#include "locationdialog.h"
#include "ui_locationdialog.h"

LocationDialog::LocationDialog(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::LocationDialog)
{
    ui->setupUi(this);
    ui->_button_box->addButton(ui->Search, QDialogButtonBox::AcceptRole);
    connect(ui->_button_box, SIGNAL(accepted()), this, SLOT(SearchClicked()));
    connect(ui->_button_box, SIGNAL(rejected()), this, SLOT(deleteLater()));
    connect(ui->GPS, SIGNAL(clicked()), this, SLOT(GPSClicked()));
    connect(ui->Location1, SIGNAL(clicked()), this, SLOT(PreviousLocation1Clicked()));
    connect(ui->Location2, SIGNAL(clicked()), this, SLOT(PreviousLocation2Clicked()));
    connect(ui->Location3, SIGNAL(clicked()), this, SLOT(PreviousLocation3Clicked()));
    connect(ui->Location4, SIGNAL(clicked()), this, SLOT(PreviousLocation4Clicked()));
    connect(ui->Location5, SIGNAL(clicked()), this, SLOT(PreviousLocation5Clicked()));
    connect(this, SIGNAL(accepted()), this, SLOT(SearchClicked()));
    connect(this, SIGNAL(rejected()), this, SIGNAL(Cancel()));
    _location_buttons.append(ui->Location1);
    _location_buttons.append(ui->Location2);
    _location_buttons.append(ui->Location3);
    _location_buttons.append(ui->Location4);
    _location_buttons.append(ui->Location5);
}

LocationDialog::~LocationDialog()
{
    delete ui;
}

void LocationDialog::SetLocation(const Location &location)
{
    ui->Location->setText(location.GetLocationName());
}

void LocationDialog::SetPreviousLocations(const Locations &previous_locations)
{
    _previous_locations = previous_locations;
    QListIterator<Location> it(_previous_locations);
    int i = 0;
    while (it.hasNext() && i < _location_buttons.size()) {
        const Location &location = it.next();
        if (location.IsValid()) {
            _location_buttons[i]->setText(location.GetLocationName());
            _location_buttons[i]->setVisible(true);
            _location_buttons[i]->setEnabled(true);
            ++i;
        }
    }
    while (i < _location_buttons.size()) {
        _location_buttons[i]->setVisible(false);
        _location_buttons[i]->setEnabled(false);
        ++i;
    }
}

void LocationDialog::SetGPSEnabled(bool gps_enabled)
{
    ui->GPS->setVisible(gps_enabled);
    ui->GPS->setEnabled(gps_enabled);
}

void LocationDialog::show()
{
    ui->gridLayout->removeWidget(ui->GPS);
    for (int i = 0; i < _location_buttons.size(); ++i) {
        ui->gridLayout->removeWidget(_location_buttons[i]);
    }
    int row = 0;
    int column = 0;
    if (ui->GPS->isEnabled()) {
        ui->gridLayout->addWidget(ui->GPS, row, column);
        ++column;
    }
    for (int i = 0; i < _location_buttons.size(); ++i) {
        if (_location_buttons[i]->isEnabled()) {
            ui->gridLayout->addWidget(_location_buttons[i], row, column);
            ++column;
            if (column > 1) {
                ++row;
                column = 0;
            }
        }
    }
    for (int i = 0; i < _location_buttons.size(); ++i) {
        if (_location_buttons[i]->text() == ui->Location->text()) {
            ui->Location->setText("");
            break;
        }
    }
    adjustSize();
    ui->Search->setDefault(true);
    ui->Location->setFocus();
    QDialog::show();
}

void LocationDialog::SearchClicked()
{
    Location location;
    location.SetLocationName(ui->Location->text());
    if (location.IsValid()) {
        emit Search(location);
        hide();
        deleteLater();
    }
}

void LocationDialog::GPSClicked()
{
    emit SearchGPS();
    hide();
    deleteLater();
}

void LocationDialog::PreviousLocation1Clicked()
{
    PreviousLocationClicked(0);
}

void LocationDialog::PreviousLocation2Clicked()
{
    PreviousLocationClicked(1);
}

void LocationDialog::PreviousLocation3Clicked()
{
    PreviousLocationClicked(2);
}

void LocationDialog::PreviousLocation4Clicked()
{
    PreviousLocationClicked(3);
}

void LocationDialog::PreviousLocation5Clicked()
{
    PreviousLocationClicked(4);
}

void LocationDialog::PreviousLocationClicked(int i)
{
    Location location;
    location.SetLocationName(_location_buttons[i]->text());
    emit SearchPreviousLocation(location);
    hide();
    deleteLater();
}

void LocationDialog::changeEvent(QEvent *e)
{
    QDialog::changeEvent(e);
    switch (e->type()) {
    case QEvent::LanguageChange:
        ui->retranslateUi(this);
        break;
    default:
        break;
    }
}
