# -*- coding: utf-8 -*-
'''
MediaBox Widget UI Buttons
2010(c) Kirill Plyashkevich <ru.spirit@gmail.com>
'''

import gobject, gtk
from mediaboxwidgetlib.ui.Widgets import ImageWidget
from advpowcommon.ui.Cairo import Cairo

class AbstractButton(ImageWidget):

  def __init__(self, images, name, size, action=None):
    ImageWidget.__init__(self, name, size, images)
    self.__c_action = action
    self.add_events(gtk.gdk.BUTTON_PRESS_MASK | gtk.gdk.BUTTON_RELEASE_MASK | gtk.gdk.LEAVE_NOTIFY)
    self.connect('button-press-event', self.__do_button_press_event)
    self.connect('button-release-event', self.__do_button_release_event)
    self.connect('leave-notify-event', self.__do_leave_notify_event)

  def __do_button_press_event(self, widget, event):
    if self.get_wstate() == self.ENABLED and self.get_property('visible') and event.type == gtk.gdk.BUTTON_PRESS:
      self.press()
    return False

  def __do_button_release_event(self, widget, event):
    if self.get_wstate != self.DISABLED and self.get_property('visible'):
      if event.type == gtk.gdk.BUTTON_RELEASE and self.__c_action:
        gobject.idle_add(self.__run_action)
      self.enable()
    return False

  def __run_action(self):
    self.disable()
    self.__c_action()
    self.enable()

  def __do_leave_notify_event(self, widget, event):
    return self.__do_button_release_event(widget, event)

  def draw(self, cr):
    raise NotImplementedError('AbstractButton.draw() must be implemented')

  def load_image(self, file):
    raise NotImplementedError('AbstractButton.load_image() must be implemented')

class Button(AbstractButton):

  def __init__(self, name, size, icon='', text='', action=None):
    images = ('cursor_frame', 'button_1', 'button_2')
    AbstractButton.__init__(self, images, name, size, action)
    self.__button_text = ''
    self.__button_image = None
    self.__set_c_icon(icon)
    self.__set_c_text(text)

  def draw(self, cr):
    image = self.images[self.get_wstate()]
    if image:
      Cairo.draw_pixbuf(cr, image, (self.allocation[2] - self.size_width)/2, (self.allocation[3] - self.size_height)/2)
    Cairo.draw_text_with_icon(cr, self.__button_icon, self.__button_text, (self.allocation[2], self.allocation[3]), self.size_height/4, Cairo.ALIGN_CENTER, Cairo.ALIGN_MIDDLE, text_color=self._theme.get_color('mb_button_text'), padding=0)
    return False

  def load_image(self, file):
    pix1 = self.cacher.get(file).scale_simple(self.size_height, self.size_height, gtk.gdk.INTERP_BILINEAR)
    pix = pix1.scale_simple(self.size_width, self.size_height, gtk.gdk.INTERP_BILINEAR)
    pix2 = pix1.copy()
    pix3 = pix1.copy()
    pix1.copy_area(0, 0, self.size_height/2, self.size_height, pix, 0, 0)
    for x in range(self.size_height/2, self.size_width - self.size_height/2):
      pix2.copy_area(self.size_height/2, 0, 1, self.size_height, pix, x, 0)
    pix3.copy_area(self.size_height/2, 0, self.size_height/2, self.size_height, pix, self.size_width - self.size_height/2, 0)
    return pix

  def __set_c_text(self, text):
    self.__button_text = text

  def __set_c_icon(self, icon):
    if icon:
      pix = self.cacher.get(self._theme.get_image(icon))
      ih = self.size_height#-2#/2
      iw = int((float(pix.get_width())/pix.get_height())*ih)
      if pix.get_width() != iw or pix.get_height() != ih:
        self.__button_icon = pix.scale_simple(iw, ih, gtk.gdk.INTERP_BILINEAR)
      else:
        self.__button_icon = pix
    else:
      self.__button_icon = None

  def set_text(self, text):
    self.__set_c_text(text)
    self.redraw()

  def set_icon(self, icon):
    self.__set_c_icon(icon)
    self.redraw()

class ImageButton(AbstractButton):

  def __init__(self, images, name, size, action=None):
    AbstractButton.__init__(self, images, name, size, action)

  def draw(self, cr):
    image = self.images[self.get_wstate()]
    if image:
      Cairo.draw_pixbuf(cr, image, (self.allocation[2] - self.size_width)/2, (self.allocation[3] - self.size_height)/2)
    return False

  def load_image(self, file):
    pix = self.cacher.get(file)
    if pix.get_width() != self.size_width or pix.get_height() != self.size_height:
      pix = pix.scale_simple(self.size_width, self.size_height, gtk.gdk.INTERP_BILINEAR)
    return pix
