/**
 * maemo-barcode - barcode detection/recognition application
 *
 * Copyright (c) 2008 Simon Pickering <S.G.Pickering@bath.ac.uk>
 *
 * Various parts of barcode recognition and GStreamer manipulation code written by:
 *       Timothy Terriberry
 *       Adam Harwell
 *       Jonas Hurrelmann
 *
 * Original GStreamer code based on the maemo-examples package
 * Copyright (c) 2007-2008 Nokia Corporation. All rights reserved.
 * Copyright (c) 2006 INdT.
 * @author Talita Menezes <talita.menezes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 * @author Jami Pekkanen <jami.pekkanen@nokia.com>
 */

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gst/gstbin.h>
#include <gst/interfaces/xoverlay.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>
#include <assert.h>
#include <gdk/gdkx.h>

#include <time.h>
#include <dmtx.h>

#include <pthread.h>
#include <sched.h>

#include "common.h"
#include "camera_n8x0.h"
#include "batoo.h"
#include "maemo-barcode.h"
#include "web.h"
#include "decoder.h"

#ifndef USE_ZBAR
#include "image.h"
#include "binarize.h"
#include "qrcode.h"
extern clock_t QR1D_timer;
#endif

extern AppData *appdata;


extern clock_t dmtx_timer;

//static unsigned char buffer[WIDTH * HEIGHT];
//static unsigned char buffer_out[WIDTH * HEIGHT];

int scanning_status=0;
int processing=0;


/* Callback that gets called whenever pipeline's message bus has
 * a message */
gboolean bus_callback (GstBus * bus, GstMessage * message) {
    gchar *message_str;
    //const gchar *message_name;
    GError *error;
    
    

    /* Report errors to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_ERROR) {
        gst_message_parse_error (message, &error, &message_str);
        printf("GST error: %s\n", message_str);
        g_error ("GST error: %s\n", message_str);
        g_free (error);
        g_free (message_str);
    }

    /* Report warnings to the console */
    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_WARNING) {
        gst_message_parse_warning (message, &error, &message_str);
        printf("GST warning: %s\n", message_str);
        g_warning ("GST warning: %s\n", message_str);
        g_free (error);
        g_free (message_str);
    }

    /* See if the message type is GST_MESSAGE_APPLICATION which means
     * thet the message is sent by the client code (this program) and
     * not by gstreamer. */

    if (GST_MESSAGE_TYPE (message) == GST_MESSAGE_APPLICATION) {
        //char EAN[14], ISBN[11], title[255], author[255];
        //float price;
        //char place[255];
        //float number;
        //char temp[50];
        //unsigned int datetime;
        int message_type;
        char *message_output;
        BarcodeData *bcdata;

        fprintf(stdout, "Got a message from the GStreamer thread\n");

        gst_element_set_state (appdata->pipeline, GST_STATE_READY);
        scanning_status=0;
        
        bcdata = malloc(sizeof(BarcodeData)); // struct in common.h

        /* Get name of the message's structure */
        //message_name = gst_structure_get_name (gst_message_get_structure (message));
        bcdata->message_type = g_value_get_int(gst_structure_get_value(gst_message_get_structure (message), "type"));
        bcdata->message_output = g_strdup(g_value_get_string(gst_structure_get_value (gst_message_get_structure (message), "output")));

        //gst_structure_get_int(gst_message_get_structure(message), "type", &message_type);
        //message_output = gst_structure_get_string(gst_message_get_structure(message), "output");

        //printf("bus_callback(): message_name=%s\n", message_name);

        gtk_button_set_label(GTK_BUTTON(appdata->scan_button), "Scan!");
        
        //handle_barcode_data(message_output, message_type);
        g_idle_add(handle_barcode_data, bcdata);
        
        fprintf(stdout, "Just finished processing barcode\n");
        
        // unref/free this message
        // gst_message_unref(message); // perhaps not needed....?
        
        processing=0;
        
#ifndef USE_ZBAR        
        QR1D_timer=0;
#endif
        dmtx_timer=0;
                
        // do we need to free message_output now?
        //g_free (message_name);
        //g_free (message_output);
        
        
    }
    return TRUE;
}


static GstBusSyncReply handle_create_window (GstBus * bus, GstMessage * message, GstPipeline * pipeline)
{
 // ignore anything but 'prepare-xwindow-id' element messages
 if (GST_MESSAGE_TYPE (message) != GST_MESSAGE_ELEMENT)
   return GST_BUS_PASS;
 
 if (!gst_structure_has_name (message->structure, "prepare-xwindow-id"))
   return GST_BUS_PASS;
 
 //win = XCreateSimpleWindow (disp, root, 0, 0, 320, 240, 0, 0, 0);
 
 //XSetWindowBackgroundPixmap (disp, win, None);
 
 //XMapRaised (disp, win);
 
 //XSync (disp, FALSE);
  
 gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (GST_MESSAGE_SRC (message)), GDK_WINDOW_XWINDOW(appdata->screen->window));
  
 gst_message_unref (message);
  
 return GST_BUS_DROP;
}




gboolean initialize_pipeline(int *argc, char ***argv) {
    GstElement *pipeline, *camera_src, *screen_sink, *image_sink;
    GstElement *screen_queue, *image_queue;
    GstElement *csp_filter, *tee; // *image_filter, 
    GstCaps *caps;
    GstBus *bus;
    
    /* Initialize Gstreamer */
    gst_init (argc, argv);

    /* Create pipeline and attach a callback to it's
     * message bus */
    pipeline = gst_pipeline_new ("test-camera");
    if(!pipeline)
        printf("pipeline failed to init\n");

    bus = gst_pipeline_get_bus (GST_PIPELINE (pipeline));
    gst_bus_set_sync_handler (bus, (GstBusSyncHandler) handle_create_window, pipeline); 
    // for the xvimagesink stuff, from here: http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gst-plugins-base-libs/html/gst-plugins-base-libs-gstxoverlay.html#gst-x-overlay-set-xwindow-id

    gst_bus_add_watch (bus, (GstBusFunc) bus_callback, NULL);
    gst_object_unref (GST_OBJECT (bus));

    /* Save pipeline to the AppData structure */
    appdata->pipeline = pipeline;

    /* Create elements */

    /* Camera video stream comes from a Video4Linux driver */
    if (appdata->device_type==DEVICE_N900)
        camera_src = gst_element_factory_make("v4l2camsrc", "camera_src");
    else if (appdata->device_type==DEVICE_N800)
        camera_src = gst_element_factory_make("gconfv4l2src", "camera_src");
    else
        camera_src = gst_element_factory_make("v4l2src", "camera_src");
    
    if(!camera_src)
        printf("camera_src failed to init\n");


    /* Colorspace filter is needed to make sure that sinks understands
     * the stream coming from the camera */
    csp_filter = gst_element_factory_make ("ffmpegcolorspace", "csp_filter");
    if(!csp_filter)
        printf("csp_filter failed to init\n");

    /* Tee that copies the stream to multiple outputs */
    tee = gst_element_factory_make ("tee", "tee");
    if(!tee)
        printf("tee failed to init\n");

    /* Queue creates new thread for the stream */
    screen_queue = gst_element_factory_make ("queue", "screen_queue");
    if(!screen_queue)
        printf("screen_queue failed to init\n");

    /* Sink that shows the image on screen. Xephyr doesn't support XVideo
     * extension, so it needs to use ximagesink, but the device uses
     * xvimagesink */
    screen_sink = gst_element_factory_make ("xvimagesink", "screen_sink");
    if(!screen_sink)
        printf("screen_sink failed to init\n");
    g_object_set (G_OBJECT (screen_sink), "sync", FALSE, NULL);

    /* Creates separate thread for the stream from which the image
     * is captured */
    image_queue = gst_element_factory_make ("queue", "image_queue");
    if(!pipeline)
        printf("pipeline failed to init\n");

    /* Filter to convert stream to use format that the gdkpixbuf library
     * can use */
    //image_filter = gst_element_factory_make ("ffmpegcolorspace", "image_filter");
    //if(!image_filter)
    //    printf("image_filter failed to init\n");

    /* A dummy sink for the image stream. Goes to bitheaven */
    image_sink = gst_element_factory_make ("fakesink", "image_sink");
    if(!image_sink)
        printf("image_sink failed to init\n");
    g_object_set (G_OBJECT (image_sink), "sync", FALSE, NULL);

    /* Check that elements are correctly initialized */
    if (!(pipeline && camera_src && screen_sink && csp_filter && screen_queue && image_queue && image_sink)) {
        g_critical ("Couldn't create pipeline elements");
        return FALSE;
    }

    /* Set image sink to emit handoff-signal before throwing away
     * it's buffer */
    g_object_set (G_OBJECT (image_sink), "signal-handoffs", TRUE, "sync", FALSE, NULL);

    /* Add elements to the pipeline. This has to be done prior to
     * linking them */
    gst_bin_add_many (GST_BIN (pipeline), camera_src, csp_filter, tee, screen_queue, screen_sink, image_queue,image_sink, NULL);
    
    /* Specify what kind of video is wanted from the camera */
    if (appdata->device_type==DEVICE_N900){
        printf("Using N900 caps\n");
        caps = gst_caps_from_string("video/x-raw-yuv,format=(fourcc)UYVY,width=320,height=240,framerate=[1/10,10/1]");
//        caps = gst_caps_new_simple("video/x-raw-yuv",
//            "width", G_TYPE_INT, 320,
//            "height", G_TYPE_INT, 240,
//            "framerate", GST_TYPE_FRACTION, 323, 25,
//            NULL);    
    }else{
        printf("Using N8x0 caps\n");
        caps = gst_caps_new_simple ("video/x-raw-yuv",
                                "format", GST_TYPE_FOURCC, GST_MAKE_FOURCC ('U', 'Y', 'V', 'Y'),
                                "width", G_TYPE_INT, 320, "height", G_TYPE_INT, 240, "framerate", GST_TYPE_FRACTION, 8, 1,
                                NULL); // change to 2 fps
    }
    /* Link the camera source and colorspace filter using capabilities
     * specified */
    if (!gst_element_link_filtered (camera_src, csp_filter, caps))
        return FALSE;

    gst_caps_unref (caps);

    /* Connect Colorspace Filter -> Tee -> Screen Queue -> Screen Sink
     * This finalizes the initialization of the screen-part of the pipeline */
    if (!gst_element_link_many (csp_filter, tee, screen_queue, screen_sink, NULL))
        return FALSE;

    //caps = gst_caps_new_simple ("video/x-raw-yuv", "width", G_TYPE_INT, WIDTH, "height", G_TYPE_INT, HEIGHT, NULL);

    /* Link the image-branch of the pipeline. The pipeline is
     * ready after this */
    if (!gst_element_link_many (tee, image_queue, image_sink, NULL))
        return FALSE;
    //if (!gst_element_link_filtered (image_filter, image_sink, caps))
    //    return FALSE;

    //gst_caps_unref (caps);

    /* As soon as screen is exposed, window ID will be advised to the sink */
    //g_signal_connect (appdata->screen, "expose-event", G_CALLBACK (expose_cb), screen_sink);

    appdata->buffer_cb_id = g_signal_connect (G_OBJECT (image_sink), "handoff", G_CALLBACK (buffer_probe_callback), NULL);

    //gst_element_set_state (pipeline, GST_STATE_PLAYING);
    //gst_element_set_state (pipeline, GST_STATE_READY);


    init_dmtx_thread();

    return TRUE;
}



/* Destroy the pipeline on exit */
void destroy_pipeline (GtkWidget * widget) {
    /* Free the pipeline. This automatically also unrefs all elements
     * added to the pipeline */
    gst_element_set_state (appdata->pipeline, GST_STATE_NULL);
    gst_object_unref (GST_OBJECT (appdata->pipeline));
}


/* Callback to be called when the screen-widget is exposed */
/*
gboolean expose_cb (GtkWidget * widget, GdkEventExpose * event, gpointer data) {
    printf("Expose event\n");
    gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (data),  GDK_WINDOW_XWINDOW(widget->window)); // GST_X_OVERLAY () // GDK_WINDOW_XWINDOW
    printf("Expose event done\n");
    return FALSE;
}
*/



/* This callback will be registered to the image sink
 * after user requests a photo */
gboolean buffer_probe_callback (GstElement *image_sink, GstBuffer *buffer, GstPad *pad) 
{
    GstMessage *message;
    //gchar *message_name;
    unsigned char *data = (unsigned char *) GST_BUFFER_DATA (buffer);
    int type;
    //unsigned char output[14];
    unsigned char *output;
    unsigned char *src, *dst;
    unsigned char fourcc[5] = "Y800";
    int bpp = 8;
    guint z = appdata->frame_width * appdata->frame_height;
    
    printf("In buffer_probe_callback\n");
    
    if (processing==1) // make sure we don't get called again while we're already scanning
        return TRUE;
    
    // check the frame is large enough here
    
    processing=1; // set the var, to avoid being called again while we're still running the last one


    // write over ourselves
    src = data;
    dst = data;
    do {
        src++;
        *(dst++) = (*(src++));
    } while (--z);


    /* analyse the image and check the status */

    printf("About to call analyse_image()\n");
    type = analyse_image(data, &output, fourcc, bpp);

    if (type) {
        GstStructure *structure;

        // appdata->scan_in_progress=0; // cancel the scan
        printf("analyse_image() returned TRUE\n");

        /* Create and send an application message which will be
         * catched in the bus watcher function. This has to be
         * sent as a message because this callback is called in
         * a gstreamer thread and calling GUI-functions here would
         * lead to X-server synchronization problems */
        structure = gst_structure_new ("analysed", "type", G_TYPE_INT, type, "output", G_TYPE_STRING, output, NULL);
        printf("first bit\n");
        message = gst_message_new_application (GST_OBJECT (appdata->pipeline), structure);
        gst_element_post_message (appdata->pipeline, message);
        
        // I think it makes a copy of the data we pass, so we can free output now
        free(output);
        // or perhaps it infact takes ownership of the data we point to...


        // hmm, do we need to free the structure at all? In the other thread probably?

    } else {
        printf("analyse_image() returned FALSE\n");
        processing=0; // we've also finished processing this frame
    }
    
    /* Returning TRUE means that the buffer can is OK to be
     * sent forward. When using fakesink this doesn't really
     * matter because the data is discarded anyway */
    return TRUE;
}



