/**
 * maemo-barcode - barcode detection/recognition application
 *
 * Copyright (c) 2008 Simon Pickering <S.G.Pickering@bath.ac.uk>
 *
 * Various parts of barcode recognition and GStreamer manipulation code written by:
 *       Timothy Terriberry
 *       Adam Harwell
 *       Jonas Hurrelmann
 *
 * Original GStreamer code based on the maemo-examples package
 * Copyright (c) 2007-2008 Nokia Corporation. All rights reserved.
 * Copyright (c) 2006 INdT.
 * @author Talita Menezes <talita.menezes@indt.org.br>
 * @author Cidorvan Leite <cidorvan.leite@indt.org.br>
 * @author Jami Pekkanen <jami.pekkanen@nokia.com>
 */

#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gst/gst.h>
#include <gst/interfaces/xoverlay.h>
#include <hildon/hildon-banner.h>
#include <hildon/hildon-program.h>
#include <hildon/hildon.h>

#include <libosso.h>

#include <glib.h>
#include <ctype.h>
#include <time.h>
#include <stdio.h>
#include <gdk/gdkx.h>

#include "common.h"
#include "gui_fremantle.h"
#include "lists.h"
#include "maemo-barcode.h"
#include "web.h"
#ifdef N900
#include "camera_n900.h"
#include <gst/interfaces/photography.h>
#else
#include "camera_n8x0.h"
#endif /* N900 */

#include <mce/dbus-names.h>
#include <mce/mode-names.h>


//gtk_widget_queue_draw(widget);

extern int scanning_status;
extern int processing;

extern AppData *appdata;

extern clock_t QR1D_timer;
extern clock_t dmtx_timer;

// Orientation functions modified from those in conboy.

#define MCE_MATCH_RULE "type='signal',interface='" MCE_SIGNAL_IF "',member='" MCE_DEVICE_ORIENTATION_SIG "'"

static void set_orientation(const gchar* orientation)
{
	/* If we disabled the accelerometers, then we don't want to react on it if
	 * it was turned on by another application.
	 *
	 * E.g. when a dialog is open we want to disable auto rotation and if there
	 * is another application that turned on the accelerators we still don't want
	 * the input.
	 */
	if (appdata->accelerometers == FALSE)
		return;

	g_printerr("INFO: Setting to orientation: %s\n", orientation);

	HildonPortraitFlags flags;


	if (strcmp(orientation, "landscape") == 0) {
		flags = HILDON_PORTRAIT_MODE_SUPPORT;
		appdata->isportrait = FALSE;
	} else if (strcmp(orientation, "portrait") == 0) {
		flags = HILDON_PORTRAIT_MODE_REQUEST;
		appdata->isportrait = TRUE;
	} else {
		g_printerr("ERROR: Orientation must be 'landscape' or 'portrait', not '%s'.\n", orientation);
		return;
	}

	/* Switch the orientation of all open windows */
	hildon_gtk_window_set_portrait_flags(GTK_WINDOW(appdata->window), flags);
	
	/* Tell the windows to redraw themselves, etc. */
	rotate_scan_window();


}



gboolean device_is_portrait_mode()
{
    osso_rpc_t ret;
    gboolean result = FALSE;

    if (osso_rpc_run_system(appdata->osso, MCE_SERVICE, MCE_REQUEST_PATH,
        MCE_REQUEST_IF, MCE_DEVICE_ORIENTATION_GET, &ret, DBUS_TYPE_INVALID) == OSSO_OK) {
        g_printerr("INFO: DBus said orientation is: %s\n", ret.value.s);

        if (strcmp(ret.value.s, MCE_ORIENTATION_PORTRAIT) == 0) {
            result = TRUE;
        }

        osso_rpc_free_val(&ret);

    } else {
        g_printerr("ERROR: Call do DBus failed\n");
    }

    return result;
}


static DBusHandlerResult dbus_handle_mce_message(DBusConnection *con, DBusMessage *msg, gpointer data)
{
    DBusMessageIter iter;
    const gchar *mode = NULL;
    /* Could also catch other MCE messages here. */

    if (dbus_message_is_signal(msg, MCE_SIGNAL_IF, MCE_DEVICE_ORIENTATION_SIG)) {
        if (dbus_message_iter_init(msg, &iter)) {
            dbus_message_iter_get_basic(&iter, &mode);
            g_printerr("INFO: New orientation is now: %s\n", mode);
            
            set_orientation(mode);
            
        }
    }
    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}

void orientation_enable_accelerometers()
{
	if (appdata->accelerometers == TRUE)
		return;

	/* TODO: Replace string "req_accelerometer_enable" with constant once we switched to final SDK */
	if (osso_rpc_run_system(appdata->osso, MCE_SERVICE, MCE_REQUEST_PATH,
			MCE_REQUEST_IF,	"req_accelerometer_enable", NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_printerr("INFO: Accelerometers enabled\n");
		appdata->accelerometers = TRUE;
	} else {
		g_printerr("WARN: Cannot enable accelerometers\n");
	}
}

void orientation_disable_accelerometers()
{
	if (appdata->accelerometers == FALSE)
		return;

	/* TODO: Replace string "req_accelerometer_disable" with constant once we switched to final SDK */
	if (osso_rpc_run_system(appdata->osso, MCE_SERVICE, MCE_REQUEST_PATH,
			MCE_REQUEST_IF,	"req_accelerometer_disable", NULL, DBUS_TYPE_INVALID) == OSSO_OK) {
		g_printerr("INFO: Accelerometers disabled\n");
		appdata->accelerometers = FALSE;
	} else {
		g_printerr("WARN: Cannot disable accelerometers\n");
	}
}

void orientation_init()
{
    printf("Entered orientation_init()\n");


    printf("orientation_init(): about to run orientation_enable_accelerometers()\n");
	/* Enable accelerometers */
	orientation_enable_accelerometers();

    printf("orientation_init(): about to run device_is_portrait_mode()\n");
	/* Get current orientation from DBus and save into appdata */
	appdata->isportrait = device_is_portrait_mode(appdata->osso);

    printf("orientation_init(): about to dbus_bus_add_match()\n");
	/* Add a matchin rule */
	dbus_bus_add_match(appdata->con, MCE_MATCH_RULE, NULL);

    printf("orientation_init(): about to dbus_connection_add_filter()\n");
	/* Add the callback, which should be called, once the device is rotated */
	dbus_connection_add_filter(appdata->con, dbus_handle_mce_message, NULL, NULL);
}


void initialize_gui(HildonProgram ** program, HildonWindow ** window, int *argc, char ***argv, gchar * app_name) {
    g_thread_init (NULL);
    gdk_threads_init ();
    /* Initialize GTK+ */

    hildon_gtk_init (argc, argv);

    /* Create HildonProgram and set application name */
    *program = HILDON_PROGRAM (hildon_program_get_instance ());
    g_set_application_name (app_name);

    /* Create the toplevel HildonWindow */
    *window = HILDON_WINDOW (hildon_stackable_window_new ());

    orientation_init();

    /* Connect destroying of the main window to gtk_main_quit */
    g_signal_connect (G_OBJECT (*window), "delete_event", G_CALLBACK (destroy_gui), NULL);

}

void destroy_gui(GtkWidget *window) {
    //close_db(appdata->db);
    gtk_main_quit();
}


/* ==================================================================================================
 *
 * Search window stuff
 *
 * ================================================================================================== */


/* Create search results window */
GtkWidget *create_search_window() {
    GtkWidget *vbuttonbox, *hbox;
    GtkWidget *button;

    //GtkWidget *view;

    // overall container
    hbox = gtk_hbox_new (FALSE, 0);


    // create the grid (single column actually) view of the search output
    appdata->search_view = create_search_view();

    // insert the grid containing a list of the databases
    gtk_container_add(GTK_CONTAINER(hbox), appdata->search_view);



    // create the vbox containing buttons
    vbuttonbox = gtk_vbutton_box_new();

    // insert some buttons into vbox
    button = gtk_button_new_with_label ("Edit");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (edit_data_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Delete");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (delete_data_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Web fill");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (webfill_all_data_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);


    // insert vbox into overall hbox
    gtk_container_add(GTK_CONTAINER(hbox), vbuttonbox);

    return hbox;
}



/* Create search dialog */
void search_dialog() {
    /* In this dialog, allow the user to search from individual tables or all
       search on individual fields or all
       use wildcards
       
       We need to return the relevant data somehow... how?
    */

    GtkWidget *dialog, *hbox;
    GtkWidget *label, *text_entry;
    GtkWidget *combo_box_tables;
    GtkWidget *combo_box_fields;
    gint response;
    //GtkTreeIter iter;
    sqlite3_stmt *stmt;
    gchar *command;
    int rc;
    char *pretty_name;

    dialog = gtk_dialog_new_with_buttons ("Search",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    //content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    // now we want to add some contents to the dialog

    // create some containers for the various bits and bobs

    hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("Search table(s):");
    combo_box_tables = gtk_combo_box_new_text();
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_tables), "All Tables");

    // now we need to read which tables are available, and loop through adding them one at a time

    // loop through all the existing lists and add them to the combo box.
    command = sqlite3_mprintf("select * from lists;");
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(appdata->db, command, stmt);
    if (rc==SQLITE_OK) {
        while (1) {
            rc = sqlite3_step(stmt);
            if ( rc==SQLITE_ROW ) { // add the row
                //key = (int)sqlite3_column_int(stmt, COL_KEY);
                //sql_name = (char *)sqlite3_column_text(stmt, COL_SQL_NAME);
                pretty_name = (char *)sqlite3_column_text(stmt, COL_PRETTY_NAME);

                gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_tables), pretty_name); // I'm assuming that each pretty_name will be unique, but then if a user can't tell them apart who cares...
                // does the combobox make a copy here, or does it use the same object?
            } else
                break;
        }
    }
    sqlite3_free(command);


    // add these objects
    gtk_container_add (GTK_CONTAINER (hbox),label );
    gtk_container_add (GTK_CONTAINER (hbox), combo_box_tables);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog


    // create a combobox for the fields to be searched
    combo_box_fields = gtk_combo_box_new_text();
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "All fields");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "EAN");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "ISBN");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Title");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Author");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Price");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Place");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Number");
    gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_fields), "Time/Date");


    // we'll keep it simple for the time being, just a text box, but this should allow wildcards (handled by the sqlite engine)
    // it won't allow date/time/number/price ranges, just =, >, <, etc
    // perhaps we should have two entry boxes, so the user can chose 2 (or more) fields and do searches involving multiple matches
    // later :)
    hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("Search fields(s):");
    gtk_container_add (GTK_CONTAINER (hbox),label );
    gtk_container_add (GTK_CONTAINER (hbox), combo_box_fields);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog

    hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("Search for:");
    text_entry = gtk_entry_new ();
    gtk_container_add (GTK_CONTAINER (hbox),label );
    gtk_container_add (GTK_CONTAINER (hbox), text_entry);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog

    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // do something as the dialog was accepted





        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);
}

/* Create search results dialog */



/* ==================================================================================================
 *
 * Scan data window stuff
 *
 * ================================================================================================== */

/* Create scan window contents */
GtkWidget *create_scan_window() {
    GtkWidget *table;
    GtkWidget *save_button;
    //PangoFontDescription *pfd;

    // create the packing structures
    appdata->scan_window_vbox_left = gtk_vbox_new (FALSE, 0);
    appdata->scan_window_vbox_right = gtk_vbox_new (FALSE, 0);
    
    g_object_ref(appdata->scan_window_vbox_left);
    g_object_ref(appdata->scan_window_vbox_right);

    
    appdata->scan_window_table = gtk_table_new (2, 2, FALSE);

    save_button = gtk_button_new_with_label("Save");

    /* Connect the "clicked" signal of the button to our callback */
    g_signal_connect (G_OBJECT (save_button), "clicked", G_CALLBACK (save_button_pressed), NULL);

    // create the labels and other description stuff which will be on the right hand side of the screen
    strcpy(appdata->EAN_text, "0000000000000");
    //appdata->EANlabel = gtk_label_new (appdata->EAN_text);
    appdata->EANlabel = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_text(GTK_ENTRY(appdata->EANlabel), appdata->EAN_text);
    gtk_entry_set_editable(GTK_ENTRY(appdata->EANlabel), FALSE);

    strcpy(appdata->title_text, "unknown");
    //appdata->titlelabel = gtk_label_new (appdata->title_text);
    //gtk_label_set_line_wrap(appdata->titlelabel, TRUE); // set line wrapping so we can see all the output
    appdata->titlelabel = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_text(GTK_ENTRY (appdata->titlelabel), appdata->title_text);
    gtk_entry_set_editable(GTK_ENTRY (appdata->titlelabel), FALSE);

    strcpy(appdata->author_text, "unknown");
    //appdata->authorlabel = gtk_label_new (appdata->author_text);
    appdata->authorlabel = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_text(GTK_ENTRY (appdata->authorlabel), appdata->author_text);
    gtk_entry_set_editable(GTK_ENTRY (appdata->authorlabel), FALSE);

    strcpy(appdata->price_text, "unknown");
    //appdata->pricelabel = gtk_label_new (appdata->price_text);
    appdata->pricelabel = hildon_entry_new(HILDON_SIZE_AUTO);
    hildon_entry_set_text(GTK_ENTRY (appdata->pricelabel), appdata->price_text);
    gtk_entry_set_editable(GTK_ENTRY (appdata->pricelabel), FALSE);

    // pack the labels, etc
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_right), appdata->EANlabel, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_right), appdata->titlelabel, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_right), appdata->authorlabel, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_right), appdata->pricelabel, TRUE, TRUE, 0);

// temporarily get rid of this until the sb stuff is working
//    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_right), save_button, TRUE, TRUE, 0);


    // create the camera view window
    appdata->screen = gtk_drawing_area_new ();
    gtk_widget_set_size_request (appdata->screen, 320, 240);
    gtk_widget_set_double_buffered (appdata->screen, FALSE);

    // allow user to tap the widget to autofocus - or not, seems to lockup the UI doing this....
    gtk_widget_add_events(appdata->screen, GDK_BUTTON_PRESS_MASK);
    g_signal_connect (G_OBJECT (appdata->screen), "button-press-event", G_CALLBACK (do_refocus), NULL);

    // create the scan button
    appdata->scan_button = gtk_button_new_with_label("Scan!");
    /* Connect the "clicked" signal of the button to our callback */
    g_signal_connect (G_OBJECT (appdata->scan_button), "clicked", G_CALLBACK (scan_button_pressed), NULL);

    // start packing these objects
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_left), appdata->screen, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (appdata->scan_window_vbox_left), appdata->scan_button, TRUE, TRUE, 0);


    gtk_table_attach ((GtkTable *) appdata->scan_window_table, appdata->scan_window_vbox_left, 0, 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
    if(!appdata->isportrait)    
        gtk_table_attach ((GtkTable *) appdata->scan_window_table, appdata->scan_window_vbox_right, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
    else
        gtk_table_attach ((GtkTable *) appdata->scan_window_table, appdata->scan_window_vbox_right, 0, 1, 1, 2, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

    return appdata->scan_window_table;
}

void update_scan_window() {
    //gtk_label_set_text ((GtkLabel *)appdata->EANlabel, appdata->EAN_text);
    //gtk_label_set_text((GtkLabel *)appdata->titlelabel, appdata->title_text);
    //gtk_label_set_text((GtkLabel *)appdata->authorlabel, appdata->author_text);
    //gtk_label_set_text((GtkLabel *)appdata->pricelabel, appdata->price_text);
    hildon_entry_set_text(GTK_ENTRY(appdata->EANlabel), appdata->EAN_text);
    hildon_entry_set_text(GTK_ENTRY(appdata->titlelabel), appdata->title_text);
    hildon_entry_set_text(GTK_ENTRY(appdata->authorlabel), appdata->author_text);
    hildon_entry_set_text(GTK_ENTRY(appdata->pricelabel), appdata->price_text);
}

void rotate_scan_window()
{
    GtkWidget *box;
    gtk_widget_hide_all(appdata->tab[0]);

    gtk_container_remove(appdata->scan_window_table, appdata->scan_window_vbox_right);
    if(!appdata->isportrait)    
        gtk_table_attach ((GtkTable *) appdata->scan_window_table, appdata->scan_window_vbox_right, 1, 2, 0, 1, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
    else
        gtk_table_attach ((GtkTable *) appdata->scan_window_table, appdata->scan_window_vbox_right, 0, 1, 1, 2, GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

    // need to refresh the appdata->screen widget somehow...
    
    gtk_widget_show_all(appdata->scan_window_table);
    gtk_widget_show_all(appdata->tab[0]);
    //gdk_window_invalidate_rect(appdata->screen->window, NULL, FALSE);
}


/* ==================================================================================================
 * Scan data window callbacks
 * ================================================================================================== */

void scan_button_pressed( GtkWidget *button) {
    // button has been pressed
    if (scanning_status) {
        // we therefore want to stop the scan
        processing=0;
        scanning_status = 0;
        gst_element_set_state (appdata->pipeline, GST_STATE_PAUSED); //READY
        gtk_button_set_label(GTK_BUTTON(button), "Scan!"); // should we wait until the pipeline shuts down? Probably not, it will be pretty quick
#ifndef USE_ZBAR
        QR1D_timer = 0;
#endif
        dmtx_timer = 0;
#ifdef N900
        gst_photography_set_autofocus(GST_PHOTOGRAPHY(appdata->camera_src), FALSE);
#endif 
    } else {
        // we therefore want to start the scan
        processing=0;
        scanning_status = 1;
#ifdef N900
        gst_element_set_state (appdata->pipeline, GST_STATE_PAUSED);
        gst_photography_set_scene_mode (GST_PHOTOGRAPHY (appdata->camera_src), GST_PHOTOGRAPHY_SCENE_MODE_CLOSEUP);
        gst_photography_set_autofocus(GST_PHOTOGRAPHY(appdata->camera_src), TRUE);
#endif               
        gst_element_set_state (appdata->pipeline, GST_STATE_PLAYING);
        gtk_button_set_label(GTK_BUTTON(button), "Cancel");
        //gst_x_overlay_set_xwindow_id (GST_X_OVERLAY (appdata->screen),  GDK_WINDOW_XWINDOW(appdata->screen->window)); //GDK_WINDOW_XWINDOW
        //gst_x_overlay_expose(GST_X_OVERLAY (appdata->screen));
#ifndef USE_ZBAR
        QR1D_timer = clock();
#endif
        dmtx_timer = clock(); 
    }
}


void save_button_pressed( GtkWidget *button) {
    // button has been pressed

    GtkWidget *dialog, *hbox;
    GtkWidget *label;
    GtkWidget *combo_box_lists;
    //GtkTreeSelection *select;
    GtkTreeModel *model;
    //GtkTreeIter  iter;
    GtkTreeSelection *select;

    sqlite3_stmt *stmt;
    gchar *command;

    char *lists_sqlite_name;
    char sqlite_name[255];
    char *pretty_name;
    char EAN[14];
    char ISBN[11];
    char title[255];
    char author[255];
    float price;
    char place[255];
    char *datetimestr;
    float number;
    unsigned int datetime;

    GtkWidget *label_EAN, *text_entry_EAN;
    GtkWidget *label_ISBN, *text_entry_ISBN;
    GtkWidget *label_TITLE, *text_entry_TITLE;
    GtkWidget *label_AUTHOR, *text_entry_AUTHOR;
    GtkWidget *label_PRICE, *spinner_PRICE;
    GtkWidget *label_PLACE, *text_entry_PLACE;
    GtkWidget *label_NUMBER, *spinner_NUMBER;
    GtkWidget *label_DATE, *text_entry_DATE;
    GtkAdjustment *spinner_price_adj, *spinner_number_adj;

    int rc=1, response;
    const char *str;

    GtkTreeIter    iter;

    spinner_price_adj = (GtkAdjustment *) gtk_adjustment_new (price, 0.0, 10000000.0, 0.01, 1.0, 1.0);
    spinner_number_adj = (GtkAdjustment *) gtk_adjustment_new (number, 0.0, 10000000.0, 0.1, 1.0, 1.0);

    printf("Save button pressed!\n");



// do weblookup and fill in the other fields now?
// only problem is that we might need to pause while the lookup happens, which is not ideal!


    // we know the EAN
    // appdata->label_text
    strcpy(EAN, appdata->EAN_text);
    strcpy(ISBN,""); // we could try filling in the ISBN if valid here!
    strcpy(title,"");
    strcpy(author,"");
    price=0.0;
    strcpy(place,""); // here? From GPS?
    number=1.0;

    time(&datetime); // now
    datetimestr = ctime(&datetime);

    // open a dialog and ask which list to save it in
    dialog = gtk_dialog_new_with_buttons ("Save to list",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);


    hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("List name:");
    // create a combobox for the lists
    combo_box_lists = gtk_combo_box_new_text();

    // loop through all the existing lists and add them to the combo box.
    command = sqlite3_mprintf("select * from lists;");
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //rc = submit_query(appdata->db, command, stmt);
    if (rc==SQLITE_OK) {
        //int key;
        //char *sql_name;
        //char *pretty_name;
        while (1) {
            rc = sqlite3_step(stmt);
            if ( rc==SQLITE_ROW ) { // add the row
                //key = (int)sqlite3_column_int(stmt, COL_KEY);
                //sql_name = (char *)sqlite3_column_text(stmt, COL_SQL_NAME);
                pretty_name = (char *)sqlite3_column_text(stmt, COL_PRETTY_NAME);

                gtk_combo_box_append_text(GTK_COMBO_BOX(combo_box_lists), pretty_name); // I'm assuming that each pretty_name will be unique, but then if a user can't tell them apart who cares...
                // does the combobox make a copy here, or does it use the same object?
            } else
                break;
        }
    }
    rc = sqlite3_finalize(stmt);
    sqlite3_free(command);

    // add these to the hbox
    gtk_container_add (GTK_CONTAINER (hbox), label );
    gtk_container_add (GTK_CONTAINER (hbox),  combo_box_lists);

    // add hbox to the dialog vbox
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog




// add in elements to allow the user to edit the data about this entry
// do a lookup first, so we add looked-up data to the list in question


    hbox = gtk_hbox_new (FALSE, 0);
    label_EAN = gtk_label_new ("EAN:");
    text_entry_EAN = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_EAN), EAN);
    gtk_container_add(GTK_CONTAINER(hbox), label_EAN);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_EAN);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_ISBN = gtk_label_new ("ISBN:");
    text_entry_ISBN = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_ISBN), ISBN);
    gtk_container_add(GTK_CONTAINER(hbox), label_ISBN);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_ISBN);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_TITLE = gtk_label_new ("TITLE:");
    text_entry_TITLE = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_TITLE), title);
    gtk_container_add(GTK_CONTAINER(hbox), label_TITLE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_TITLE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_AUTHOR = gtk_label_new ("AUTHOR:");
    text_entry_AUTHOR = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_AUTHOR), author);
    gtk_container_add(GTK_CONTAINER(hbox), label_AUTHOR);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_AUTHOR);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_PRICE = gtk_label_new ("PRICE:");
    spinner_PRICE = gtk_spin_button_new (spinner_price_adj, 0.01, 2);
    gtk_container_add(GTK_CONTAINER(hbox), label_PRICE);
    gtk_container_add(GTK_CONTAINER(hbox), spinner_PRICE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_PLACE = gtk_label_new ("PLACE:");
    text_entry_PLACE = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_PLACE), place);
    gtk_container_add(GTK_CONTAINER(hbox), label_PLACE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_PLACE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_NUMBER = gtk_label_new ("NUMBER:");
    spinner_NUMBER = gtk_spin_button_new (spinner_number_adj, 0.1, 1);
    gtk_container_add(GTK_CONTAINER(hbox), label_NUMBER);
    gtk_container_add(GTK_CONTAINER(hbox), spinner_NUMBER);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_DATE = gtk_label_new ("DATE/TIME:");
    text_entry_DATE = gtk_entry_new (); // perhaps a gtkcalendar?
    sprintf(datetimestr, "%u",datetime);
    gtk_entry_set_text(GTK_ENTRY(text_entry_DATE), datetimestr);
    gtk_container_add(GTK_CONTAINER(hbox), label_DATE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_DATE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);



    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // find which table name was highlighted
        pretty_name = gtk_combo_box_get_active_text(GTK_COMBO_BOX(combo_box_lists)); // does this return the entire row? Or just some selected text?


        // get sql name from pretty name
        get_sql_from_pretty_name(pretty_name, sqlite_name);


        // extract the values from the textboxes
        str = gtk_entry_get_text(GTK_ENTRY(text_entry_EAN));
        strcpy(EAN, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_ISBN));
        strcpy(ISBN, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_TITLE));
        strcpy(title, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_AUTHOR));
        strcpy(author, str);

        price =  gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner_PRICE));

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_PLACE));
        strcpy(place, str);

        number =  gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner_NUMBER));

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_DATE));
        sscanf(str, "%u", &datetime);

        
        // now just add the current data to this list
        add_entry(sqlite_name, EAN, ISBN, title, author, &price, place, &number, &datetime);

        
        if(strcmp(appdata->current_data_table_name, sqlite_name)!=0){
            printf("save_button_pressed(): Not the current table, don't update display\n");
            gtk_widget_destroy(dialog);
            return; // is not the displayed table, so no need to update atm...
        }else
            printf("save_button_pressed(): Is the current table, update display\n");
        
        if(!(appdata->data_store)){
            printf("save_button_pressed(): Data store does not exist!\n");
            gtk_widget_destroy(dialog);
            return;
        }
        
        gtk_list_store_append (appdata->data_store, &iter); // add a new "row"
        gtk_list_store_set (appdata->data_store, &iter,
                            //COL_KEY, key, // leave this line out and get it auto-assigned?
                            COL_EAN, EAN,
                            COL_ISBN, ISBN,
                            COL_TITLE, title,
                            COL_AUTHOR, author,
                            COL_PRICE, price,
                            COL_PLACE, place,
                            COL_NUMBER, number,
                            COL_DATE, datetime,-1);


        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);

}

/* ==================================================================================================
 *
 * Lists window stuff
 *
 * ================================================================================================== */

/* Create database list window contents */
GtkWidget *create_lists_window() {
    GtkWidget *vbuttonbox, *hbox;
    GtkWidget *button;

    //GtkWidget *view;

    // overall container
    hbox = gtk_hbox_new (FALSE, 0);

    // create the grid (single column actually) view of the databases
    appdata->lists_view = create_lists_view_and_model ();
    

    // insert the grid containing a list of the databases
    gtk_container_add(GTK_CONTAINER(hbox), appdata->lists_view);
    
    // create the vbox containing buttons
    vbuttonbox = gtk_vbutton_box_new();

    // insert some buttons into vbox
    button = gtk_button_new_with_label ("New");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (create_new_list_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Rename");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (rename_list_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Delete");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (delete_list_button), NULL);
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    // insert vbox into overall hbox
    gtk_container_add(GTK_CONTAINER(hbox), vbuttonbox);

    return hbox;
}


/* ==================================================================================================
 * lists window callbacks
 * ================================================================================================== */

// handle clicks on the New List button
void create_new_list_button() {
    GtkWidget *dialog, *hbox;
    GtkWidget *label, *text_entry;
    gint response;

    //GtkListStore  *store;
    GtkTreeIter    iter;
    const char *pretty_name;
    char sql_name[255];

    dialog = gtk_dialog_new_with_buttons ("Create new list",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    //content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    // now we want to add some contents to the dialog

    // create some containers for the various bits and bobs
    hbox = gtk_hbox_new (FALSE, 0);

    label = gtk_label_new ("List name:");
    text_entry = gtk_entry_new ();

    gtk_container_add (GTK_CONTAINER (hbox), label );
    gtk_container_add (GTK_CONTAINER (hbox), text_entry);

    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog

    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // do something as the dialog was accepted

        // grab text from the dialog
        pretty_name = gtk_entry_get_text(GTK_ENTRY(text_entry));

        // check we have something in the text box
        if (strlen(pretty_name)==0)
            break;

        // let's turn this pretty name into a valid sql name
        deprettify_name(pretty_name, sql_name);

        printf("create_new_list_button(): Pretty name: %s; SQL name: %s\n", pretty_name, sql_name);
        // so let's add this new list then
        add_list(pretty_name, sql_name);

        // now add entry to the list store
        // get the SQL name which has been assigned to this list

        gtk_list_store_append (appdata->lists_store, &iter); // add a new "row"
        gtk_list_store_set (appdata->lists_store, &iter,
                            COL_KEY, 0,
                            COL_SQL_NAME, sql_name,
                            COL_PRETTY_NAME, pretty_name,
                            -1);
        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);
}


// handle clicks on the Rename List button
void rename_list_button() {
    // create a list box with the existing lists, and a button to click to allow rename

    GtkWidget *dialog, *hbox;
    GtkWidget *label, *text_entry;
    gint response;
    const char *list_name;

    GtkTreeSelection *select;
    char *sqlite_name, *pretty_name;
    int key;
    GtkTreeModel *model;
    GtkTreeIter  iter;

    select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->lists_view));

    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) {
        gtk_tree_model_get(model, &iter, COL_SQL_NAME, &sqlite_name, -1); // get the SQL name for this list
        gtk_tree_model_get(model, &iter, COL_PRETTY_NAME, &pretty_name, -1); // get the SQL name for this list
        gtk_tree_model_get(model, &iter, COL_KEY, &key, -1); // get the SQL name for this list
        printf("rename_list_button(): SQLite name=%s\n", sqlite_name);
        printf("rename_list_button(): pretty name=%s\n", pretty_name);
        printf("rename_list_button(): key=%i\n", key);
        // gtk_list_store_remove(store, &iter); // remove it from the display
        //g_free(sqlite_name);
    } else {
        return;
    }

    dialog = gtk_dialog_new_with_buttons ("Rename existing list",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    //content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    // now we want to add some contents to the dialog

    // create some containers for the various bits and bobs
    hbox = gtk_hbox_new (FALSE, 0);

    label = gtk_label_new ("List name:");
    text_entry = gtk_entry_new ();

    printf("1\n");

    gtk_entry_set_text(GTK_ENTRY(text_entry), pretty_name);

    printf("3\n");

    gtk_container_add (GTK_CONTAINER (hbox),label );
    gtk_container_add (GTK_CONTAINER (hbox), text_entry); // add vertical box to horizontal box

    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), hbox); // add horizontal box to the dialog

    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // do something as the dialog was accepted

        // grab text from the dialog and overwrite the string we used earlier
        list_name = gtk_entry_get_text(GTK_ENTRY(text_entry));

        // check we have something in the text box
        if (strlen(list_name)==0)
            break;
        else
            printf("rename_list_button(): Renaming to %s\n", list_name);

        // so let's add this new list then
        rename_list(sqlite_name, list_name);

        // now rename the list in the treeview
        //gtk_list_store_remove(store, &iter); // remove it from the display
        gtk_list_store_set (appdata->lists_store, &iter, COL_PRETTY_NAME, list_name, -1);

        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);
    g_free(sqlite_name);
}


// handle clicks on the Delete List button
void delete_list_button() {
    // delete the list passed in after asking if the user is sure!

    GtkWidget *dialog, *hbox;
    GtkWidget *label; //, *text_entry;
    gint response;
    GString *text;
    char *pretty_name;

    GtkTreeSelection *select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->lists_view));
    char *sql_name;
    GtkTreeModel *model;
    GtkTreeIter  iter;

    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) {
        gtk_tree_model_get(model, &iter, COL_SQL_NAME, &sql_name, -1); // get the SQL name for this list
        gtk_tree_model_get(model, &iter, COL_PRETTY_NAME, &pretty_name, -1); // get the SQL name for this list
        printf("delete_list_button(): sql_name=%s\n", sql_name);
        // gtk_list_store_remove(store, &iter); // remove it from the display
    }

    dialog = gtk_dialog_new_with_buttons ("Create new list",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    //content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
    // now we want to add some contents to the dialog

    // create some containers for the various bits and bobs
    hbox = gtk_hbox_new (FALSE, 0);

    // get the pretty name from the sql list name
    //get_pretty_from_sql_name(sql_name, pretty_list);

    // create some text asking if the user really wants to delete this table
    text = g_string_new("");
    g_string_printf(text, "Delete table '%s'?", pretty_name);

    // create the label to hold this text
    label = gtk_label_new (text->str);

    // add it to the dialog
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), label);

    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // do something as the dialog was accepted

        // so let's delete this table then
        remove_list(sql_name);

        // now remove the list in the treeview
        gtk_list_store_remove(appdata->lists_store, &iter); // remove it from the display

        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);
    g_string_free(text, TRUE);

}




/* ==================================================================================================
 *
 * Data window stuff
 *
 * ================================================================================================== */

/* Create list entries (i.e. data) window contents */
GtkWidget *create_data_window() {
    GtkWidget *hbox, *vbuttonbox, *button; //, *view;
    GtkAdjustment *hadjustment;
    GtkAdjustment *vadjustment;
    GtkWidget *ScrolledWindow;
    char *current_sql_list_name;

    printf("Entered create_data_window()\n");    
    
    //vbox = gtk_vbox_new (FALSE, 0);
    hbox = gtk_hbox_new (FALSE, 0);

    current_sql_list_name = GetCurrentListViewTable();
    //strcpy(appdata->current_data_table_name, current_sql_list_name);
    
    // create the grid (single column actually) view of the databases
    appdata->data_view = create_data_view_and_model(current_sql_list_name); // need to pass it the name of the database to look at

    g_free(current_sql_list_name);


    hadjustment = gtk_tree_view_get_hadjustment(GTK_TREE_VIEW(appdata->data_view));
    vadjustment = gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(appdata->data_view));
    ScrolledWindow = gtk_scrolled_window_new(hadjustment, vadjustment);

    
    gtk_container_add(GTK_CONTAINER(ScrolledWindow), appdata->data_view);
    //gtk_box_pack_start(GTK_CONTAINER(ScrolledWindow), appdata->data_view, TRUE, TRUE, 0);
    // insert the grid containing a list of the databases
    gtk_container_add(GTK_CONTAINER(hbox), ScrolledWindow);
    //gtk_box_pack_start(GTK_CONTAINER(hbox), ScrolledWindow, TRUE, TRUE, 0);

    // insert the vbox containing buttons
    vbuttonbox = gtk_vbutton_box_new();

    // insert some buttons into vbox
    button = gtk_button_new_with_label ("Edit");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (edit_data_button), NULL); // pass it the data in question!
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Delete");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (delete_data_button), NULL); // pass it the data in question!
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    button = gtk_button_new_with_label ("Web fill");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked", GTK_SIGNAL_FUNC (webfill_all_data_button), NULL); // pass it the data in question!
    gtk_container_add(GTK_CONTAINER(vbuttonbox), button);
    //gtk_widget_show(button);

    // insert vbox into overall hbox
    gtk_container_add(GTK_CONTAINER(hbox), vbuttonbox);



    return hbox;
}

/* ==================================================================================================
 * Data window callbacks
 * ================================================================================================== */

void edit_data_button() { // pass it the data in question!
    // create list boxes with the existing data, and a button to click to allow saving changes

    // check these lengths - EAN and ISBN are limited in length
    char EAN[14], ISBN[14], title[255], author[255], place[255], datetimestr[255];
    float price, number;
    unsigned int datetime;
    int key;

    GtkWidget *dialog, *hbox;
    GtkWidget *label_EAN, *text_entry_EAN;
    GtkWidget *label_ISBN, *text_entry_ISBN;
    GtkWidget *label_TITLE, *text_entry_TITLE;
    GtkWidget *label_AUTHOR, *text_entry_AUTHOR;
    GtkWidget *label_PRICE, *spinner_PRICE;
    GtkWidget *label_PLACE, *text_entry_PLACE;
    GtkWidget *label_NUMBER, *spinner_NUMBER;
    GtkWidget *label_DATE, *text_entry_DATE;

    gint response;
    //GString *list_name;
    const char *str;
    int ret;

    GtkTreeSelection *select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->data_view));
    //char *sqlite_name;
    GtkTreeModel *model;
    GtkTreeIter  iter;

    GtkAdjustment *spinner_price_adj, *spinner_number_adj;


    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) {
        gtk_tree_model_get(model, &iter, COL_KEY, &key, -1); // get the key value for this data row
    } else {
        return; // no row selected
    }

    ret = get_data_row(appdata->current_data_table_name, &key, EAN, ISBN, title, author, &price, place, &number, &datetime);
    if (ret==SQLITE_ROW)
        // we got data, otherwise no

    spinner_price_adj = (GtkAdjustment *) gtk_adjustment_new (price, 0.0, 10000000.0, 0.01, 1.0, 1.0);
    spinner_number_adj = (GtkAdjustment *) gtk_adjustment_new (number, 0.0, 10000000.0, 0.1, 1.0, 1.0);

    dialog = gtk_dialog_new_with_buttons ("Edit data",
                                          GTK_WINDOW(appdata->window),
                                          GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                          GTK_STOCK_OK,
                                          GTK_RESPONSE_ACCEPT,
                                          GTK_STOCK_CANCEL,
                                          GTK_RESPONSE_REJECT,
                                          NULL);

    // now we want to add some contents to the dialog

    // create some containers for the various bits and bobs

    // might want to replace these generic text edit boxes with some specific ones (e.g. formatted edit, dates, currency, etc.)
    hbox = gtk_hbox_new (FALSE, 0);
    label_EAN = gtk_label_new ("EAN:");
    text_entry_EAN = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_EAN), EAN);
    gtk_container_add(GTK_CONTAINER(hbox), label_EAN);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_EAN);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_ISBN = gtk_label_new ("ISBN:");
    text_entry_ISBN = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_ISBN), ISBN);
    gtk_container_add(GTK_CONTAINER(hbox), label_ISBN);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_ISBN);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_TITLE = gtk_label_new ("TITLE:");
    text_entry_TITLE = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_TITLE), title);
    gtk_container_add(GTK_CONTAINER(hbox), label_TITLE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_TITLE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_AUTHOR = gtk_label_new ("AUTHOR:");
    text_entry_AUTHOR = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_AUTHOR), author);
    gtk_container_add(GTK_CONTAINER(hbox), label_AUTHOR);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_AUTHOR);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_PRICE = gtk_label_new ("PRICE:");
    spinner_PRICE = gtk_spin_button_new (spinner_price_adj, 0.01, 2);
    gtk_container_add(GTK_CONTAINER(hbox), label_PRICE);
    gtk_container_add(GTK_CONTAINER(hbox), spinner_PRICE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_PLACE = gtk_label_new ("PLACE:");
    text_entry_PLACE = gtk_entry_new ();
    gtk_entry_set_text(GTK_ENTRY(text_entry_PLACE), place);
    gtk_container_add(GTK_CONTAINER(hbox), label_PLACE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_PLACE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_NUMBER = gtk_label_new ("NUMBER:");
    spinner_NUMBER = gtk_spin_button_new (spinner_number_adj, 0.1, 1);
    gtk_container_add(GTK_CONTAINER(hbox), label_NUMBER);
    gtk_container_add(GTK_CONTAINER(hbox), spinner_NUMBER);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label_DATE = gtk_label_new ("DATE/TIME:");
    text_entry_DATE = gtk_entry_new (); // perhaps a gtkcalendar?
    sprintf(datetimestr, "%u",datetime);
    gtk_entry_set_text(GTK_ENTRY(text_entry_DATE), datetimestr);
    gtk_container_add(GTK_CONTAINER(hbox), label_DATE);
    gtk_container_add(GTK_CONTAINER(hbox), text_entry_DATE);
    gtk_container_add(GTK_CONTAINER(GTK_DIALOG (dialog)->vbox), hbox);

    /*
       // this gives us the pretty name for the current database, we might want to know this
       get_pretty_from_sql_name(sqlite_name, list_name->str);
    */

    // should add two more buttons to this dialog - to perform a web search and update on this information, and to delete the currently selected record


    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_ACCEPT:
        // do something as the dialog was accepted

        // grab text from the dialog and overwrite the string we used earlier
        // I assume the point which is assigned doesn't need to be freed

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_EAN));
        strcpy(EAN, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_ISBN));
        strcpy(ISBN, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_TITLE));
        strcpy(title, str);

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_AUTHOR));
        strcpy(author, str);

        price =  gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner_PRICE));

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_PLACE));
        strcpy(place, str);

        number =  gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinner_NUMBER));

        str = gtk_entry_get_text(GTK_ENTRY(text_entry_DATE));
        sscanf(str, "%u", &datetime);

        // so let's replace the data
        change_entry(appdata->current_data_table_name, key, EAN, ISBN, title, author, &price, place, &number, &datetime);

        // now refresh the data held in the treeview
        gtk_list_store_set (appdata->data_store, &iter, COL_KEY, key, COL_EAN, EAN, COL_ISBN, ISBN, COL_TITLE, title, \
                            COL_AUTHOR, author, COL_PRICE, price, COL_PLACE, place, COL_NUMBER, number, COL_DATE, datetime,-1);

        break;
    default:
        break; // dialog was cancelled, so do nothing
    }
    gtk_widget_destroy(dialog);
    //g_string_free(list_name, TRUE);
}

void delete_data_button() { // pass it the data in question!
    // delete the currently highlighted record

    GtkWidget *dialog; //, *hbox;

    gint response;
    //GString *list_name;

    GtkTreeSelection *select = gtk_tree_view_get_selection (GTK_TREE_VIEW (appdata->data_view));
    GtkTreeModel *model;
    GtkTreeIter  iter;

    int key;

    if (gtk_tree_selection_get_selected(GTK_TREE_SELECTION(select), &model, &iter)) {
        gtk_tree_model_get(model, &iter, COL_KEY, &key, -1); // get the key value for this data row
    } else {
        return; // nothing selected
    }



    dialog = gtk_message_dialog_new (GTK_WINDOW(appdata->window),
                                     GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                                     GTK_MESSAGE_QUESTION,
                                     GTK_BUTTONS_YES_NO,
                                     "Delete record?");
    gtk_widget_show_all (dialog);

    response = gtk_dialog_run(GTK_DIALOG(dialog));
    switch (response) {
    case GTK_RESPONSE_YES:
        // do something as the dialog was accepted

        // so let's remove the data from the table
        remove_entry(appdata->current_data_table_name, key);

        // now remove the row from the treeview
        gtk_list_store_remove(appdata->data_store, &iter);

        break;
    default:
        break; // dialog was cancelled or "no" was clicked, so do nothing
    }
    gtk_widget_destroy(dialog);
}


void webfill_all_data_button() { // pass it the data in question!
    // lookup each EAN/ISBN in the current table using a web service and fill in the missing information (e.g. title, author, etc.)
    sqlite3_stmt *stmt;
    gchar *command;
    int rc;
    int key;
    char EAN[14];
    char ISBN[11];
    char title[255];
    char author[255];
    float price;
    char place[255];
    float number;
    unsigned int datetime;
    //GtkTreeIter  iter;

    command = sqlite3_mprintf("select * from '%q';", appdata->current_data_table_name);
    rc = sqlite3_prepare_v2(appdata->db, command, -1, &stmt, NULL); //submit_query(db, command, stmt);
    if (rc) {
        printf("webfill_all_data_button: error: %s \n", sqlite3_errmsg(appdata->db));
        return;
    }
    printf("webfill_all_data_button: select * from %s \n", appdata->current_data_table_name);
    sqlite3_free(command);
    while (get_next_data_row(stmt, &key, EAN, ISBN, title, author, &price, place, &number, &datetime)==SQLITE_ROW) {
        /*
        if (lookup_data(EAN, ISBN, title, author, &price, place, &number, &datetime)) {
            // we've looked up and updated this data
            // now save it back in the database
            change_entry(appdata->current_data_table_name, key, EAN, ISBN, title, author, &price, place, &number, &datetime); // should probably error check here too!

            // now refresh the data held in the treeview
            gtk_list_store_set (appdata->data_store, &iter, COL_KEY, key, COL_EAN, EAN, COL_ISBN, ISBN, COL_TITLE, title, \
                                COL_AUTHOR, author, COL_PRICE, price, COL_PLACE, place, COL_NUMBER, number, COL_DATE, datetime,-1);
        } else {
            // couldn't find this one, so do nothing
        }
        */
    }
    rc=sqlite3_finalize(stmt); //finish_query(stmt);
}


/* Create scanned book entry window contents */
/* Create scanned CD entry window contents */
/* Create scanned DVD entry window contents */
/* Create scanned other entry window contents */

GtkWidget *create_info_window(char *EAN) { // pass it the data in question!
    GtkWidget *button, *hbox;
    hbox = gtk_hbox_new (FALSE, 0);
    button = gtk_button_new_with_label ("Place holder!");
    // insert vbox into overall hbox
    gtk_container_add(GTK_CONTAINER(hbox), button);

    return hbox;
}




