/*
Mancala - A Historical Board Game
Copyright (C) 2009-2010 A.H.M.Mahfuzur Rahman 65mahfuz90@gmail.com
Copyright (c) 2010 Reto Zingg g.d0b3rm4n@gmail.com

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "Cup.h"

#include <math.h>

#include <QFont>
#include <QSize>
#include <QPainter>
#include <QPointF>
#include <QGraphicsSceneMouseEvent>
#include <QTimer>

// #include <kdebug.h>

#include "Stone.h"
#include "Board.h"
#include "GameInfo.h"


Cup::Cup(GameInfo* info, int cupNumber, QGraphicsSvgItem *parent)
    : QGraphicsSvgItem(parent)

{
    m_index = cupNumber;
    m_board = qgraphicsitem_cast<Board*>(parent);
    m_gameInfo = info;
    m_text = new QGraphicsSimpleTextItem(parent);

    setSharedRenderer(m_board->renderer());
    setElementId("cup");
    setZValue(2);

    drawInitially();
    drawTextInitially(m_gameInfo->initialStonesPerCup());

    //connect(this,SIGNAL(signalChangeText()),this,SLOT(slotChangeText()));
}

void Cup::drawInitially(){

    qreal gapWidth = m_board->boundingRect().width() * ( 0.2 / ( m_gameInfo->numCupsPerRow() + m_gameInfo->numKalahs() + 1 ) );

    m_size.setWidth( m_board->boundingRect().width() * 0.8 / ( m_gameInfo->numCupsPerRow() + m_gameInfo->numKalahs() ) );
    m_size.setHeight( m_board->boundingRect().height() * 0.35 );

    qreal posX,posY;

    if(m_index < m_gameInfo->numCupsPerRow())
        posX = m_board->boundingRect().x() + gapWidth * ( m_index % m_gameInfo->numCupsPerRow() +
        m_gameInfo->numKalahs()) + ( m_index % m_gameInfo->numCupsPerRow() + 1) * m_size.width();

    else{
        posX = m_board->boundingRect().x() + gapWidth *
        ((((2*m_gameInfo->numCupsPerRow()-1)-m_index) % m_gameInfo->numCupsPerRow()) + m_gameInfo->numKalahs() )
        +(((2*m_gameInfo->numCupsPerRow()-1)-m_index) % m_gameInfo->numCupsPerRow() + 1) * m_size.width();
    }
    posY = m_board->boundingRect().y() + 0.1 * m_board->boundingRect().height();
    if(m_index >= m_gameInfo->numCupsPerRow())
        posY += 0.1 * m_board->boundingRect().height() + m_size.height();


    m_pos.setX(posX);
    m_pos.setY(posY);

    qreal cupScaleX = m_size.width() / this->boundingRect().width();
    qreal cupScaleY = m_size.height()/ this->boundingRect().height();

    m_mid_point.setX( m_pos.x() + m_size.width() / 2 );
    m_mid_point.setY( m_pos.y() + m_size.height() / 2 );

    setPos(m_pos);
    scale(cupScaleX,cupScaleY);

}


void Cup::drawTextInitially(int numStone){

    QString str;
    while(numStone != 0){
            //qDebug() << "Number " << numStone % 10 << " Char " <<(QChar)(numStone % 10 + '0') ;
            str.push_front( (QChar)(numStone % 10 + '0') );
            numStone /= 10;
        }

    m_text->setText(str);
    m_text->setFont( QFont("Comic Sans MS", 24, QFont::Bold) );
    m_text->setPen(QPen(QColor("darkslateblue"), 1.5, Qt::SolidLine, Qt::RoundCap,
                   Qt::RoundJoin));

    QLinearGradient gradient(0, 0, 0, 100);
    gradient.setColorAt(0.0, "mediumslateblue");
    gradient.setColorAt(1.0, "cornsilk");
    m_text->setBrush(gradient);

    m_text->setVisible(false);

    int textX = m_pos.x() + m_size.width();
    textX -= m_text->boundingRect().size().width() + 15;

    int textY =  m_pos.y() + m_size.height();
    textY -= m_text->boundingRect().size().height() - 15;

    m_text->scale(0.6,0.6);
    m_text->setPos( textX,textY );
    m_text->setZValue(4);
    m_text->setVisible(true);

}

//add Stone to cup
void Cup::addStone(Stone* stone){

    QPointF* stonePos;
    qreal m_diameter_stone = qMin(m_size.width()/5, m_size.height()/5);

    stone->setSize(QSize(m_diameter_stone,m_diameter_stone));
    stonePos = calculateStonePosition(stone);

    //kDebug() << "Stone Position: " << stonePos->x() << " " <<stonePos->y();

    stone->setPosition(stonePos->x(),stonePos->y());

    m_stone.append(stone);
}


//Add a stone to the stoneList
void Cup::updateStoneList(Stone* stoneItem){

    QPointF *stonePos;
    m_stone.append(stoneItem);

    //calculate and update stone position
    stonePos = calculateStonePosition(stoneItem);
    stoneItem->animateStones(QPointF(stonePos->x(),stonePos->y()));
    update();

    QTimer::singleShot(500, this, SLOT(slotChangeText()));

}

//Remove a stone from the stoneList
Stone* Cup::updateStoneList(){

    Stone* stone;

    if(m_stone.count()){
        stone = m_stone.front();
        m_stone.pop_front();
    }
    else{
        // kDebug() << "List is empty already";
        return NULL;
    }
    slotChangeText();

    return stone;

}

void Cup::slotChangeText(){

    m_text->setVisible(false);
    if( m_stone.count() ){
        m_text->setText( QString("%1").arg( m_stone.count() ));
        m_text->setVisible(true);
        update();
    }
}


QPointF* Cup::calculateStonePosition(Stone* stone){

    QPointF *returnPoint = new QPointF();

    qreal diameter = qMin( m_size.width(),m_size.height());

    int initialLength = qrand() % (int) diameter / 3;
    int angle = qrand() % 360;

    qreal posX = middlePoint().x() + initialLength * cos(angle * M_PI / 180);
    qreal posY = middlePoint().y() + initialLength * sin(angle * M_PI / 180);

    if( ( posX + stone->size().width()) > ( position().x() + 0.7 * m_size.width() ) )
        posX -= stone->size().width();
    else if( posX < ( m_pos.x()+ 0.3 * m_size.width() ) )
        posX += stone->size().width();

    if( ( posY + stone->size().height() ) > ( m_pos.y() + 0.7 * m_size.height() ) )
        posY -= stone->size().height();
    else if( posY < (m_pos.y() + 0.3 < m_size.height() ) )
        posY += stone->size().height();

    returnPoint->setX(posX);
    returnPoint->setY(posY);

    return returnPoint;

}

//Reimplementing the shape item
QPainterPath Cup::shape() const
 {
     QPainterPath path;
     path.addEllipse(boundingRect());
     return path;
 }
