#!/bin/sh

# Copyright (c) 2009 Nokia Corporation. All rights reserved.
#
# Permission is hereby granted, free of charge, to any person
# obtaining a copy of this software and associated documentation
# files (the "Software"), to deal in the Software without
# restriction, including without limitation the rights to use,
# copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following
# conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
# OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
# HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
# WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
# OTHER DEALINGS IN THE SOFTWARE.

set -eu
#exec 2>> /tmp/maddev-if.log; date >&2; set -x; : "$@"

LC_ALL=C LANG=C
export LC_ALL LANG
PATH=$HOME/bin:/bin:/sbin:/usr/bin:/usr/sbin
export PATH

USB_IF=usb0
WLAN_IF=wlan0
GPRS_IF=gprs0
UDHCPC_SCRIPT=/etc/udhcpc/libicd_network_ipv4.script

# 1. If no input parameters, return information about
#    usb0, wlan0 and gprs0 interfaces
# 2. If one input parameter, setup the interface and allocate the
#    the IP address dynamically using dhcp
#    Input parameters:
#    <interface name> either 'usb0', wlan0' or 'gprs0'
# 3. If more than one input parameters, setup the interface with statically
#    define IP and set a default gateway if defined
#    Input parameters:
#    <interface name> either 'usb0', wlan0' or 'gprs0'
#    <IP address>     IPv4 address
#    <IP mask>        in dotted decimal format
#    <default gw>     optional parameter:
#                     IPv4 address of default gateway,
#                     if missing default route is not set
#
#  Return codes:
#    0 - success
#    1 - error

warn () { echo "$@" >&2; }
die () { warn "$@"; exit 1; }

kill_dhcpclients ()
{
	udhcpcpids=`ps ax | awk '/[u]dhcpc.*'"$1"'/ {print $1}'`
	if [ "x$udhcpcpids" != "x" ] ; then
		devrootsh kill -9 $udhcpcpids
	fi
}

set_ip_and_mask ()
{
	set x `ifconfig $1 2>/dev/null \
		| /bin/sed -n 's/.*inet addr:\\([^ ]*\\).*Mask:\\([^ ]*\\).*/\\1 \\2/p'`
	ip=${2:-} mask=${3:-}
}


if [ $# -gt 0 ]; then

  ifname=$1
  case $ifname in
	$USB_IF | $WLAN_IF | $GPRS_IF)
		;;
	*)
		warn "Interface $ifname unknown!"
		die "Only following interfaces can be modified: $USB_IF, $WLAN_IF, $GPRS_IF"
		;;
  esac

  if [ $# -eq 1 ]
  then
	# Allocate dynamic IP address to interface

	kill_dhcpclients $ifname
	devrootsh /sbin/udhcpc -i $ifname -s $UDHCPC_SCRIPT >/dev/null 2>&1 &
	# DHCP client process exit is considered as a failure
	udhcpcpid=$!
	sleep 2
	devrootsh kill -0 $udhcpcpid 2>/dev/null ||
		wait $udhcpcpid || die "DHCP client exited with failure!"

  elif [ $# -ge 3 ]
  then
	# Allocate static IP address to interface

	ip=$2 ipmask=$3 default_gw=${4:-}

	kill_dhcpclients $ifname
	devrootsh ifconfig $ifname up $ip netmask $ipmask ||
		die "ifconfig failed to configure $ifname"

        # Delete the default route for this interface if it exists
        defroute=`route -n | awk '$1 == "0.0.0.0" && $8 == "'"$ifname"'"'`
	if [ "x$defroute" != "x" ] ; then
		devrootsh route del default dev $ifname ||
			die "Deleting default route for $ifname failed!"
	fi

	if [ "x${default_gw:-}" != "x" ] ; then
		if [ "$default_gw" == "0.0.0.0" ] ; then
			devrootsh route add default dev $ifname >/dev/null 2>&1
		else
			devrootsh route add default gw $default_gw dev $ifname >/dev/null 2>&1
		fi || die "Setting default gateway for $ifname failed!"
	fi

  fi
fi

# Return interface information to standard output in following format:
# <interface name> <IP address> <IP mask> <flags>
#
#   <interface name> either 'usb0', wlan0' or 'gprs0'
#   <IP address>     IPv4 address, '-' if not defined
#   <IP mask>        in dotted decimal format, '-' if not defined
#   <flags>          'R' if default route
#                    'D' if IP address is allocated using DHCP
#                    '-' non of flags set

defroute=`route -n | awk '$1 == "0.0.0.0" { print $8; exit }'`

for ifname in $USB_IF $WLAN_IF $GPRS_IF; do
	set_ip_and_mask $ifname
	flags=
	if [ "x$defroute" = "x$ifname" ] ; then
		flags=R
	fi
	if [ "x$ip" != "x" ] ; then
		if ps ax | grep "[u]dhcpc.*$ifname" >/dev/null 2>&1 ; then
			flags="$flags"D
		fi
	fi

	echo "$ifname" "${ip:--}" "${mask:--}" "${flags:--}"
done
