/*
 * tangle-scrolling-actor.c
 *
 * This file is part of Tangle Toolkit - A graphical actor library based on Clutter Toolkit
 *
 * (c) 2010 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 */

#include "tangle-scrolling-actor.h"

/**
 * SECTION:tangle-scrolling-actor
 * @Short_description: A wrapper actor that scrolls its content
 * @Title: TangleScrollingActor
 */

G_DEFINE_TYPE(TangleScrollingActor, tangle_scrolling_actor, TANGLE_TYPE_WRAPPER_ACTOR);

enum {
	PROP_0,
	PROP_SCROLLING_OFFSET_X,
	PROP_SCROLLING_OFFSET_Y,
	PROP_SCROLLING_WIDTH,
	PROP_SCROLLING_HEIGHT,
	PROP_SCROLLING_THRESHOLD_X,
	PROP_SCROLLING_THRESHOLD_Y,
	PROP_MAX_OVERSHOOT_X,
	PROP_MAX_OVERSHOOT_Y
};

struct _TangleScrollingActorPrivate {
	gfloat scrolling_offset_x;
	gfloat scrolling_offset_y;
	gfloat scrolling_threshold_x;
	gfloat scrolling_threshold_y;
	gfloat max_overshoot_x;
	gfloat max_overshoot_y;

	gfloat scrolling_width;
	gfloat scrolling_height;
	
	gfloat start_motion_x;
	gfloat start_motion_y;
	gfloat start_scrolling_offset_x;
	gfloat start_scrolling_offset_y;

	gulong captured_event_handler_id;

	guint descendant_interacting : 1;	
	guint descendant_dragging : 1;
	guint scrolling_effective : 1;
};

static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data);
static void send_interaction_event(TangleActor* actor, gboolean began);

ClutterActor* tangle_scrolling_actor_new(ClutterActor* wrapped) {

	return CLUTTER_ACTOR(g_object_new(TANGLE_TYPE_SCROLLING_ACTOR, "wrapped", wrapped, NULL));
}

gfloat tangle_scrolling_actor_get_scrolling_offset_x(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_offset_x;
}

void tangle_scrolling_actor_set_scrolling_offset_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_x) {
	if (scrolling_actor->priv->scrolling_offset_x != scrolling_offset_x) {
		scrolling_actor->priv->scrolling_offset_x = scrolling_offset_x;
		if (scrolling_actor->priv->scrolling_offset_x < -scrolling_actor->priv->max_overshoot_x) {
			scrolling_actor->priv->scrolling_offset_x = -scrolling_actor->priv->max_overshoot_x;
		} else if (scrolling_actor->priv->scrolling_offset_x > scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x) {
			scrolling_actor->priv->scrolling_offset_x = scrolling_actor->priv->scrolling_width + scrolling_actor->priv->max_overshoot_x;
		}
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scrolling_actor));
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling_offset-x");
	}	
}

gfloat tangle_scrolling_actor_get_scrolling_offset_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_offset_y;
}

void tangle_scrolling_actor_set_scrolling_offset_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_offset_y) {
	if (scrolling_actor->priv->scrolling_offset_y != scrolling_offset_y) {
		scrolling_actor->priv->scrolling_offset_y = scrolling_offset_y;
		if (scrolling_actor->priv->scrolling_offset_y < -scrolling_actor->priv->max_overshoot_y) {
			scrolling_actor->priv->scrolling_offset_y = -scrolling_actor->priv->max_overshoot_y;
		} else if (scrolling_actor->priv->scrolling_offset_y > scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y) {
			scrolling_actor->priv->scrolling_offset_y = scrolling_actor->priv->scrolling_height + scrolling_actor->priv->max_overshoot_y;
		}
		clutter_actor_queue_relayout(CLUTTER_ACTOR(scrolling_actor));
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling_offset-y");
	}
}

gfloat tangle_scrolling_actor_get_scrolling_width(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_width;
}

gfloat tangle_scrolling_actor_get_scrolling_height(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_height;
}

void tangle_scrolling_actor_set_scrolling_threshold_x(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold_x) {
	if (scrolling_actor->priv->scrolling_threshold_x != scrolling_threshold_x) {
		scrolling_actor->priv->scrolling_threshold_x = scrolling_threshold_x;
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-threshold-x");
	}
}

gfloat tangle_scrolling_actor_get_scrolling_threshold_y(TangleScrollingActor* scrolling_actor) {

	return scrolling_actor->priv->scrolling_threshold_y;
}

void tangle_scrolling_actor_set_scrolling_threshold_y(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold_y) {
	if (scrolling_actor->priv->scrolling_threshold_y != scrolling_threshold_y) {
		scrolling_actor->priv->scrolling_threshold_y = scrolling_threshold_y;
		g_object_notify(G_OBJECT(scrolling_actor), "scrolling-threshold-y");
	}
}

void tangle_scrolling_actor_set_scrolling_threshold(TangleScrollingActor* scrolling_actor, gfloat scrolling_threshold) {
	tangle_scrolling_actor_set_scrolling_threshold_x(scrolling_actor, scrolling_threshold);
	tangle_scrolling_actor_set_scrolling_threshold_y(scrolling_actor, scrolling_threshold);
}

static gboolean tangle_scrolling_actor_button_press_event(ClutterActor* actor, ClutterButtonEvent* event) {
	TangleScrollingActor* scrolling_actor;
	gfloat x, y;
		
	scrolling_actor = TANGLE_SCROLLING_ACTOR(actor);

	if (!scrolling_actor->priv->captured_event_handler_id) {
		scrolling_actor->priv->start_motion_x = event->x;
		scrolling_actor->priv->start_motion_y = event->y;
		scrolling_actor->priv->start_scrolling_offset_x = scrolling_actor->priv->scrolling_offset_x;
		scrolling_actor->priv->start_scrolling_offset_y = scrolling_actor->priv->scrolling_offset_y;
		scrolling_actor->priv->scrolling_effective = FALSE;
		scrolling_actor->priv->captured_event_handler_id = g_signal_connect(clutter_actor_get_stage(actor), "captured-event", G_CALLBACK(on_captured_event), scrolling_actor);
	}

	return FALSE;
}


static void tangle_scrolling_actor_allocate_wrapped(TangleWrapperActor* wrapper_actor, ClutterActor* wrapped, const ClutterActorBox* box, ClutterAllocationFlags flags) {
	TangleScrollingActor* scrolling_actor;
	gfloat old_width, old_height;
	ClutterActorBox actor_box;

	scrolling_actor = TANGLE_SCROLLING_ACTOR(wrapper_actor);

	old_width = scrolling_actor->priv->scrolling_width;
	old_height = scrolling_actor->priv->scrolling_height;
	clutter_actor_get_preferred_size(wrapped, NULL, NULL, &scrolling_actor->priv->scrolling_width, &scrolling_actor->priv->scrolling_height);
	if (old_width != scrolling_actor->priv->scrolling_width) {
		g_object_notify(G_OBJECT(wrapper_actor), "scrolling-width");
	}
	if (old_height != scrolling_actor->priv->scrolling_height) {
		g_object_notify(G_OBJECT(wrapper_actor), "scrolling-height");
	}

	actor_box.x1 = box->x1 - scrolling_actor->priv->scrolling_offset_x;
	actor_box.y1 = box->y1 - scrolling_actor->priv->scrolling_offset_y;
	actor_box.x2 = actor_box.x1 + scrolling_actor->priv->scrolling_width;
	actor_box.y2 = actor_box.y1 + scrolling_actor->priv->scrolling_height;

	clutter_actor_allocate(wrapped, &actor_box, flags);
	
	scrolling_actor->priv->scrolling_width -= box->x2 - box->x1;
	scrolling_actor->priv->scrolling_height -= box->y2 - box->y1;
}

static void tangle_scrolling_actor_handle_descendant_event(TangleActor* actor, TangleEvent* event) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(actor);
	
	if (event->any.sender != actor) {
		switch (event->type) {
			case TANGLE_EVENT_INTERACTION_BEGAN:
				scrolling_actor->priv->descendant_interacting = TRUE;
				if (event->interaction.dragging) {
					scrolling_actor->priv->descendant_dragging = TRUE;
				}
				break;
			case TANGLE_EVENT_INTERACTION_ENDED:
				scrolling_actor->priv->descendant_interacting = FALSE;
				scrolling_actor->priv->descendant_dragging = FALSE;
				break;
			default:
				break;
		}
	}

	TANGLE_ACTOR_CLASS(tangle_scrolling_actor_parent_class)->handle_descendant_event(actor, event);
}

static void tangle_scrolling_actor_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);

	switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			tangle_scrolling_actor_set_scrolling_offset_x(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_OFFSET_Y:
			tangle_scrolling_actor_set_scrolling_offset_y(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			tangle_scrolling_actor_set_scrolling_threshold_x(scrolling_actor, g_value_get_float(value));
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			tangle_scrolling_actor_set_scrolling_threshold_y(scrolling_actor, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void tangle_scrolling_actor_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        TangleScrollingActor* scrolling_actor;

	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);

        switch (prop_id) {
		case PROP_SCROLLING_OFFSET_X:
			g_value_set_float(value, scrolling_actor->priv->scrolling_offset_x);
			break;
		case PROP_SCROLLING_OFFSET_Y:
			g_value_set_float(value, scrolling_actor->priv->scrolling_offset_y);
			break;
		case PROP_SCROLLING_WIDTH:
			g_value_set_float(value, scrolling_actor->priv->scrolling_width);
			break;
		case PROP_SCROLLING_HEIGHT:
			g_value_set_float(value, scrolling_actor->priv->scrolling_height);
			break;
		case PROP_SCROLLING_THRESHOLD_X:
			g_value_set_float(value, scrolling_actor->priv->scrolling_threshold_x);
			break;
		case PROP_SCROLLING_THRESHOLD_Y:
			g_value_set_float(value, scrolling_actor->priv->scrolling_threshold_y);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void tangle_scrolling_actor_dispose(GObject* object) {
	TangleScrollingActor* scrolling_actor;
	
	scrolling_actor = TANGLE_SCROLLING_ACTOR(object);
	
	if (scrolling_actor->priv->captured_event_handler_id) {
		g_signal_handler_disconnect(scrolling_actor, scrolling_actor->priv->captured_event_handler_id);
		scrolling_actor->priv->captured_event_handler_id = 0;
	}

	G_OBJECT_CLASS(tangle_scrolling_actor_parent_class)->dispose(object);
}

static void tangle_scrolling_actor_class_init(TangleScrollingActorClass* scrolling_actor_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(scrolling_actor_class);
	ClutterActorClass* clutter_actor_class = CLUTTER_ACTOR_CLASS(scrolling_actor_class);
	TangleActorClass* actor_class = TANGLE_ACTOR_CLASS(scrolling_actor_class);
	TangleWrapperActorClass* wrapper_actor_class = TANGLE_WRAPPER_ACTOR_CLASS(scrolling_actor_class);

	gobject_class->dispose = tangle_scrolling_actor_dispose;
	gobject_class->set_property = tangle_scrolling_actor_set_property;
	gobject_class->get_property = tangle_scrolling_actor_get_property;
	
	clutter_actor_class->button_press_event = tangle_scrolling_actor_button_press_event;
	
	actor_class->handle_descendant_event = tangle_scrolling_actor_handle_descendant_event;

	wrapper_actor_class->allocate_wrapped = tangle_scrolling_actor_allocate_wrapped;

	/**
	 * TangleScrollingActor:scrolling-offset-x:
	 *
	 * The offset to the left side of the wrapped actor's bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_X,
	                                g_param_spec_float("scrolling-offset-x",
	                                                    "Scrolling offset X",
	                                                    "The offset to the left side of the wrapped actor's bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-offset-y:
	 *
	 * The offset to the top side of the wrapped actor's bounding box.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_OFFSET_Y,
	                                g_param_spec_float("scrolling-offset-y",
	                                                    "Scrolling offset Y",
	                                                    "The offset to the top side of the wrapped actor's bounding box",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-width:
	 *
	 * The maximum value for the scrollng-offset-x: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_WIDTH,
	                                g_param_spec_float("scrolling-width",
	                                                    "Scrolling width",
	                                                    "The maximum value for the scrollng-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-height:
	 *
	 * The maximum value for the scrollng-offset-y: property.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_HEIGHT,
	                                g_param_spec_float("scrolling-height",
	                                                    "Scrolling height",
	                                                    "The maximum value for the scrollng-offset-y property",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));	/**
	 * TangleScrollingActor:scrolling-threshold-x:
	 *
	 * The horizontal threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_X,
	                                g_param_spec_float("scrolling-threshold-x",
	                                                    "Scrolling threshold X",
	                                                    "The horizontal threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
	/**
	 * TangleScrollingActor:scrolling-threshold-y:
	 *
	 * The vertical threshold before start scrolling.
	 */
	g_object_class_install_property(gobject_class, PROP_SCROLLING_THRESHOLD_Y,
	                                g_param_spec_float("scrolling-threshold-y",
	                                                    "Scrolling threshold Y",
	                                                    "The vertical threshold before start scrolling",
	                                                   -G_MAXFLOAT, G_MAXFLOAT, 0.0,
	                                                   G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));


	g_type_class_add_private(gobject_class, sizeof(TangleScrollingActorPrivate));
}

static void tangle_scrolling_actor_init(TangleScrollingActor* scrolling_actor) {
	scrolling_actor->priv = G_TYPE_INSTANCE_GET_PRIVATE(scrolling_actor, TANGLE_TYPE_SCROLLING_ACTOR, TangleScrollingActorPrivate);

	clutter_actor_set_reactive(CLUTTER_ACTOR(scrolling_actor), TRUE);
	g_object_set(CLUTTER_ACTOR(scrolling_actor), "clip-to-allocation", TRUE, NULL);
}

#define ABS_F(x) ((x) < 0 ? -(x) : (x))
static gboolean on_captured_event(ClutterActor* actor, ClutterEvent* event, gpointer user_data) {
	TangleScrollingActor* scrolling_actor;
	gboolean handled = TRUE;
	gfloat x, y, dx, dy;
	
	if (event->type == CLUTTER_MOTION || event->type == CLUTTER_BUTTON_RELEASE) {
		scrolling_actor = TANGLE_SCROLLING_ACTOR(user_data);
		
		g_object_ref(scrolling_actor);

		clutter_event_get_coords(event, &x, &y);
		dx = x - scrolling_actor->priv->start_motion_x;
		dy = y - scrolling_actor->priv->start_motion_y;

		if (scrolling_actor->priv->scrolling_effective) {
			tangle_scrolling_actor_set_scrolling_offset_x(scrolling_actor, scrolling_actor->priv->start_scrolling_offset_x - dx);
			tangle_scrolling_actor_set_scrolling_offset_y(scrolling_actor, scrolling_actor->priv->start_scrolling_offset_y - dy);
			if (event->type == CLUTTER_BUTTON_RELEASE) {
				g_signal_handler_disconnect(actor, scrolling_actor->priv->captured_event_handler_id);
				scrolling_actor->priv->captured_event_handler_id = 0;
				send_interaction_event(TANGLE_ACTOR(scrolling_actor), FALSE);
			}
		} else {
			if (event->type == CLUTTER_MOTION && !scrolling_actor->priv->descendant_dragging &&
			    (!scrolling_actor->priv->descendant_interacting ||
			     ABS_F(dy) > scrolling_actor->priv->scrolling_threshold_x ||
			     ABS_F(dx) > scrolling_actor->priv->scrolling_threshold_y)) {
				scrolling_actor->priv->scrolling_effective = TRUE;
				send_interaction_event(TANGLE_ACTOR(scrolling_actor), TRUE);
			} else if (event->type == CLUTTER_BUTTON_RELEASE) {
				g_signal_handler_disconnect(actor, scrolling_actor->priv->captured_event_handler_id);
				scrolling_actor->priv->captured_event_handler_id = 0;
			}
		}

		g_object_unref(scrolling_actor);
	}

	return FALSE;
}

static void send_interaction_event(TangleActor* actor, gboolean began) {
	TangleEvent* event;
	
	event = tangle_event_new((began ? TANGLE_EVENT_INTERACTION_BEGAN : TANGLE_EVENT_INTERACTION_ENDED),
	                         TANGLE_ACTOR(actor));
	event->interaction.x_axis = TRUE;
	event->interaction.y_axis = TRUE;
	event->interaction.dragging = TRUE;
	tangle_actor_send_event(actor, event);
	tangle_event_free(event);
}
