/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-provider/microfeeddatabase.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedobject.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedfile.h>
#include <microfeed-common/microfeedfileindextable.h>

#include <string.h>
#include <sys/file.h>
#include <stdio.h>

struct _MicrofeedDatabase {
	MicrofeedObject object;
	char* public_name;
	char* private_name;
	char* directory;
	MicrofeedFile* file;
	MicrofeedFileIndexTable* table;
	MicrofeedStore* database_tables;
};

struct _MicrofeedDatabaseTable {
	MicrofeedObject object;
	MicrofeedDatabase* database;
	char* name;
	MicrofeedStore* indices;
	int removed : 1;
};

struct _MicrofeedDatabaseIndex {
	MicrofeedObject object;
	MicrofeedDatabaseTable* database_table;
	char* name;
	MicrofeedFileIndexTable* table;
};

struct _MicrofeedDatabaseIterator {
	MicrofeedDatabaseIndex* database_index;
	MicrofeedFileIndexTableIterator* table_iterator;
};

static void database_free_function(MicrofeedDatabase* database);
static void database_table_free_function(MicrofeedDatabaseTable* database_table);
static void database_index_free_function(MicrofeedDatabaseIndex* database);
static void replace_data(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size, int partial);
static void remove_block(MicrofeedDatabaseTable* database_table, void* block);
	
static MicrofeedClass microfeed_database_class = {
	"MicrofeedDatabase",
	(MicrofeedObjectFreeFunction)database_free_function
};

static MicrofeedClass microfeed_database_table_class = {
	"MicrofeedDatabaseTable",
	(MicrofeedObjectFreeFunction)database_table_free_function
};
	
static MicrofeedClass microfeed_database_index_class = {
	"MicrofeedDatabaseIndex",
	(MicrofeedObjectFreeFunction)database_index_free_function
};

static MicrofeedDatabase* new_database(const char* public_name, const char* private_name, const char* directory) {
	MicrofeedDatabase* database = NULL;
	char* path;
	MicrofeedFile* file;
	MicrofeedFileIndexTable* table;

	path = microfeed_util_string_concatenate(directory, "/", private_name, NULL);
	if ((file = microfeed_file_new(path, 1))) {
		if ((table = microfeed_file_index_table_new_named_indices(file, 0, 1))) {
			database = microfeed_object_new(&microfeed_database_class, MicrofeedDatabase);
			database->public_name = strdup(public_name);
			database->private_name = strdup(private_name);
			database->directory = strdup(directory);
			database->file = file;
			database->table = table;
			database->database_tables = microfeed_store_new_sorted_weak_references((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)microfeed_database_table_get_name);
		} else {
			microfeed_file_free(file);
		}
	}
	free(path);
	
	return database;
}

MicrofeedDatabase* microfeed_database_new(const char* name, const char* directory) {
	MicrofeedDatabase* database = NULL;
	char buffer[1024];
	char* source_path;
	char* destination_path;
	int fd;
	struct flock lock;

	snprintf(buffer, 1024, "%s%c%ld", name, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR, (long)getpid());
	if (microfeed_util_create_directory_recursively(directory)) {
		source_path = microfeed_util_string_concatenate(directory, "/", name, NULL);
		destination_path = microfeed_util_string_concatenate(directory, "/", buffer, NULL);
		if ((fd = open(source_path, O_RDWR)) >= 0) {
			lock.l_type = F_RDLCK;
			lock.l_whence = SEEK_SET;
			lock.l_start = 0;
			lock.l_len = 0;
			if (fcntl(fd, F_GETLK, &lock) == 0) {
				/* Note: file_copy opens and closes the source file, and thus, releases the lock. */
				if (microfeed_util_file_copy(source_path, destination_path)) {
					if (!(database = new_database(name, buffer, directory))) {
						unlink(destination_path);
					}
				}
			}
			/* This, at least, releases the lock. */
			close(fd);
		} else {
			database = new_database(name, buffer, directory);
		}
		microfeed_memory_free(source_path);
		microfeed_memory_free(destination_path);
	}

	return database;
}

static void database_free_function(MicrofeedDatabase* database) {
	char* source_path;
	char* destination_path;
	int error;
	
	microfeed_store_clean_weak_references(database->database_tables);
	microfeed_assert(microfeed_store_get_size(database->database_tables) == 0);
	microfeed_file_index_table_free(database->table);
	microfeed_file_free(database->file);

	source_path = microfeed_util_string_concatenate(database->directory, "/", database->private_name, NULL);
	destination_path = microfeed_util_string_concatenate(database->directory, "/", database->public_name, NULL);
	unlink(destination_path);
	link(source_path, destination_path);
	unlink(source_path);
	microfeed_memory_free(source_path);
	microfeed_memory_free(destination_path);

	free(database->public_name);
	free(database->private_name);
	free(database->directory);
	microfeed_store_free(database->database_tables);
}

MicrofeedDatabaseTable* microfeed_database_get_table(MicrofeedDatabase* database, const char* name) {
	MicrofeedDatabaseTable* database_table = NULL;
	
	microfeed_object_lock(database, MicrofeedDatabase);

	if (!(database_table = microfeed_store_get(database->database_tables, name, MicrofeedDatabaseTable))) {
		database_table = microfeed_object_new(&microfeed_database_table_class, MicrofeedDatabaseTable);
		database_table->database = microfeed_object_ref(database, MicrofeedDatabase);
		database_table->name = strdup(name);
		database_table->indices = microfeed_store_new_sorted_weak_references((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)microfeed_database_index_get_name);

		microfeed_store_insert(database->database_tables, database_table);
	}

	microfeed_object_unlock(database, MicrofeedDatabase);

	return database_table;
}

int microfeed_database_get_named_data(MicrofeedDatabase* database, const char* name, void** data_return, size_t* data_size_return) {
	int retvalue = 0;
	MicrofeedFileIndex index;

	microfeed_object_lock(database, MicrofeedDatabase);

	if ((index = microfeed_file_index_table_get_named_index(database->table, name)) != MICROFEED_FILE_INDEX_INVALID) {
		*data_size_return = microfeed_file_get_index_size(database->file, index);
		*data_return = malloc(*data_size_return);
		memcpy(*data_return, microfeed_file_get_block_impl(database->file, index), *data_size_return);
		retvalue = 1;
	}
	
	microfeed_object_unlock(database, MicrofeedDatabase);

	return retvalue;
}

int microfeed_database_set_named_data(MicrofeedDatabase* database, const char* name, void* data, size_t data_size) {
	int retvalue = 0;
	MicrofeedFileIndex index;
	void* block;

	microfeed_object_lock(database, MicrofeedDatabase);
	
	if ((index = microfeed_file_index_table_get_named_index(database->table, name)) != MICROFEED_FILE_INDEX_INVALID) {
		block = microfeed_file_get_block_impl(database->file, index);
		if (microfeed_file_block_get_size(block) < data_size) {
			block = microfeed_file_resize_block_impl(database->file, block, data_size);
		}
		memcpy(block, data, data_size);
	} else {
		block = microfeed_file_allocate_block_impl(database->file, data_size);
		memcpy(block, data, data_size);
		index = microfeed_file_get_index(database->file, block);
		microfeed_assert(microfeed_file_index_table_insert_named_index(database->table, name, index));
	}
	
	microfeed_object_unlock(database, MicrofeedDatabase);

	return retvalue;

}

void microfeed_database_remove_named_data(MicrofeedDatabase* database, const char* name) {

	microfeed_object_lock(database, MicrofeedDatabase);

	microfeed_file_index_table_remove_named_index(database->table, name);

	microfeed_object_unlock(database, MicrofeedDatabase);
}

static void database_table_free_function(MicrofeedDatabaseTable* database_table) {
	microfeed_object_unref(database_table->database, MicrofeedDatabase);
	free(database_table->name);
	microfeed_store_clean_weak_references(database_table->indices);
	microfeed_assert(microfeed_store_get_size(database_table->indices) == 0);
	microfeed_store_free(database_table->indices);
}

MicrofeedDatabaseIndex* microfeed_database_table_get_index(MicrofeedDatabaseTable* database_table, const char* name, int create, MicrofeedDatabaseIndexFunction index_function, MicrofeedDatabaseCompareFunction compare_function) {
	MicrofeedDatabaseIndex* database_index;
	char* full_name;
	MicrofeedFileIndex index;
	MicrofeedFileIndexTable* table = NULL;
	int error;
	
	microfeed_object_lock(database_table->database, MicrofeedDatabase);

	if (!(database_index = microfeed_store_get(database_table->indices, name, MicrofeedDatabaseIndex))) {
		full_name = microfeed_util_string_concatenate(database_table->name, ":", name, NULL);
		if ((index = microfeed_file_index_table_get_named_index(database_table->database->table, full_name)) != MICROFEED_FILE_INDEX_INVALID) {
			table = microfeed_file_index_table_new_sorted(database_table->database->file, index, 0, (MicrofeedFileIndexTableCompareKeysFunction)compare_function, (MicrofeedFileIndexTableGetKeyFunction)index_function);
		} else if (create) {
			if ((table = microfeed_file_index_table_new_sorted(database_table->database->file, MICROFEED_FILE_INDEX_INVALID, 1, (MicrofeedFileIndexTableCompareKeysFunction)compare_function, (MicrofeedFileIndexTableGetKeyFunction)index_function))) {
				if (!microfeed_file_index_table_insert_named_index(database_table->database->table, full_name, microfeed_file_index_table_get_file_index(table))) {
					microfeed_file_index_table_free(table);
					table = NULL;
				}
			}
		}
		if (table) {
			database_index = microfeed_object_new(&microfeed_database_index_class, MicrofeedDatabaseIndex);
			database_index->database_table = microfeed_object_ref(database_table, MicrofeedDatabaseTable);
			database_index->name = strdup(name);
			database_index->table = table;

			microfeed_store_insert(database_table->indices, database_index);
		}
		free(full_name);
	}

	microfeed_object_unlock(database_table->database, MicrofeedDatabase);
	
	return database_index;	
}

static void database_index_free_function(MicrofeedDatabaseIndex* database_index) {
	MicrofeedFileIndexTableIterator* iterator;
	void* block;
	char* full_name;

	if (database_index->database_table->removed) {
		microfeed_store_clean_weak_references(database_index->database_table->indices);
		if (microfeed_store_get_size(database_index->database_table->indices) == 1) {
			for (iterator = microfeed_file_index_table_iterate(database_index->table, NULL, 0, 0);
			     (block = microfeed_file_index_table_iterator_get_block_impl(iterator));
			     microfeed_file_index_table_iterator_next(iterator)) {
				microfeed_file_free_block(database_index->database_table->database->file, block);     
			}
			microfeed_file_index_table_iterator_free(iterator);
		}
		full_name = microfeed_util_string_concatenate(database_index->database_table->name, ":", database_index->name, NULL);
		microfeed_file_index_table_remove_named_index(database_index->database_table->database->table, full_name);
		free(full_name);
	}

	microfeed_object_unref(database_index->database_table, MicrofeedDatabaseTable);
	free(database_index->name);
	microfeed_file_index_table_free(database_index->table);
}

const char* microfeed_database_table_get_name(MicrofeedDatabaseTable* database_table) {
	
	return database_table->name;
}

int microfeed_database_table_get_data_count(MicrofeedDatabaseTable* database_table) {
	int count = 0;
	MicrofeedDatabaseIndex* database_index;
	
	if (microfeed_store_get_size(database_table->indices)) {
		database_index = microfeed_store_get_index(database_table->indices, 0, MicrofeedDatabaseIndex);
		count = microfeed_file_index_table_get_size(database_index->table);
		microfeed_object_unref(database_index, MicrofeedDatabaseIndex);
	}
	
	return count;
}

time_t microfeed_database_table_get_last_updated_time(MicrofeedDatabaseTable* database_table) {
	time_t timestamp = 0;
	MicrofeedStoreIterator* iterator;
	MicrofeedDatabaseIndex* index;
	time_t t;

	for (iterator = microfeed_store_iterate(database_table->indices, NULL);
	     (index = microfeed_store_iterator_get(iterator, MicrofeedDatabaseIndex));
	     microfeed_store_iterator_next(iterator)) {
		t = microfeed_file_index_table_get_last_updated_time(index->table);
		if (t > timestamp) {
			timestamp = t;
		}
	}
	microfeed_store_iterator_free(iterator);

	return timestamp;
}

void microfeed_database_table_set_to_be_removed(MicrofeedDatabaseTable* database_table) {
	microfeed_assert(microfeed_store_get_size(database_table->indices) > 0);

	database_table->removed = 1;
}

MicrofeedDatabaseTable* microfeed_database_index_get_table(MicrofeedDatabaseIndex* database_index) {
	
	return database_index->database_table;
}

const char* microfeed_database_index_get_name(MicrofeedDatabaseIndex* database_index) {
	
	return database_index->name;
}

void microfeed_database_index_replace_data(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size) {
	replace_data(database_index, data, data_size, 0);
}

void microfeed_database_index_replace_data_partial(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size) {
	replace_data(database_index, data, data_size, 1);
}

int microfeed_database_index_get_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size, void** data_return, size_t* data_size_return) {
	int retvalue = 0;
	void* block;

	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	if ((block = microfeed_file_index_table_get_block_impl(database_index->table, key, key_size))) {
		*data_size_return = microfeed_file_block_get_size(block);
		*data_return = malloc(*data_size_return);
		memcpy(*data_return, block, *data_size_return);
		retvalue = 1;
	}
	
	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
	
	return retvalue;	
}

int microfeed_database_index_get_data_partial(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size, void** data_return, size_t* data_size_return) {
	int retvalue = 0;
	void* block;
	size_t size;

	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	if ((block = microfeed_file_index_table_get_block_impl(database_index->table, key, key_size))) {
		if ((size = microfeed_file_block_get_size(block)) < *data_size_return) {
			*data_size_return = size;
		}
		*data_return = malloc(*data_size_return);
		memcpy(*data_return, block, *data_size_return);
		retvalue = 1;
	}
	
	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
	
	return retvalue;	
}

int microfeed_database_index_is_data(MicrofeedDatabaseIndex* database_index, const void* key, size_t key_size) {
	int retvalue = 0;

	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	if (microfeed_file_index_table_get_block_impl(database_index->table, key, key_size)) {
		retvalue = 1;
	}
	
	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
	
	return retvalue;

}

MicrofeedDatabaseIterator* microfeed_database_index_iterate(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, int backwards) {
	MicrofeedDatabaseIterator* iterator;

	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	iterator = microfeed_memory_allocate(MicrofeedDatabaseIterator);
	iterator->database_index = database_index;
	iterator->table_iterator = microfeed_file_index_table_iterate(database_index->table, start_key, start_key_size, backwards);

	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);

	return iterator;
}
	
void microfeed_database_index_remove_data(MicrofeedDatabaseIndex* database_index, const void* key, const size_t key_size) {
	void* block;
	
	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	if ((block = microfeed_file_index_table_get_block_impl(database_index->table, key, key_size))) {
		remove_block(database_index->database_table, block);
	}

	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
}

void microfeed_database_index_remove_data_range(MicrofeedDatabaseIndex* database_index, const void* start_key, const size_t start_key_size, const void* end_key, const size_t end_key_size) {
	MicrofeedFileIndexTableIterator* iterator;
	void* end_block = NULL;
	void* block;

	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);
	
	if (end_key && end_key_size) {
		end_block = microfeed_file_index_table_get_block_impl(database_index->table, end_key, end_key_size);
	}
	for (iterator = microfeed_file_index_table_iterate(database_index->table, start_key, start_key_size, 0);
	     (block = microfeed_file_index_table_iterator_get_block_impl(iterator));
	     microfeed_file_index_table_iterator_next(iterator)) {
		remove_block(database_index->database_table, block);
		if (block == end_block) {
			break;
		}
	}
	microfeed_file_index_table_iterator_free(iterator);

	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
}

void microfeed_database_iterator_free(MicrofeedDatabaseIterator* iterator) {
	MicrofeedDatabase* database;
	
	database = iterator->database_index->database_table->database;
	
	microfeed_object_lock(database, MicrofeedDatabase);

	microfeed_file_index_table_iterator_free(iterator->table_iterator);
	iterator->database_index = NULL;
	microfeed_memory_free(iterator);

	microfeed_object_unlock(database, MicrofeedDatabase);
}
	
int microfeed_database_iterator_get(MicrofeedDatabaseIterator* iterator, void** data_return, size_t* data_size_return) {
	int success = 0;
	void* block;

	microfeed_object_lock(iterator->database_index->database_table->database, MicrofeedDatabase);
	
	if (block = microfeed_file_index_table_iterator_get_block_impl(iterator->table_iterator)) {
		*data_size_return = microfeed_file_block_get_size(block);
		*data_return = malloc(*data_size_return);
		memcpy(*data_return, block, *data_size_return);

		success = 1;
	}
	
	microfeed_object_unlock(iterator->database_index->database_table->database, MicrofeedDatabase);

	return success;
}

void microfeed_database_iterator_next(MicrofeedDatabaseIterator* iterator) {
	microfeed_object_lock(iterator->database_index->database_table->database, MicrofeedDatabase);

	microfeed_file_index_table_iterator_next(iterator->table_iterator);
	
	microfeed_object_unlock(iterator->database_index->database_table->database, MicrofeedDatabase);
}

int microfeed_database_compare_keys_direct(const void* key1, size_t key1_size, const void* key2, size_t key2_size) {

	return microfeed_file_index_table_compare_keys_direct(key1, key1_size, key2, key2_size);
}

static void replace_data(MicrofeedDatabaseIndex* database_index, const void* data, size_t data_size, int partial) {
	const void* key;
	size_t key_size;
	void* block;
	MicrofeedStoreIterator* iterator;
	MicrofeedDatabaseIndex* index;
	const void* block_key;
	size_t block_key_size;
	MicrofeedFileIndex block_index;
	MicrofeedFileIndex existing_index;
	
	microfeed_object_lock(database_index->database_table->database, MicrofeedDatabase);

	microfeed_file_index_table_get_block_key(database_index->table, data, data_size, &key, &key_size);
	if ((block = microfeed_file_index_table_get_block_impl(database_index->table, key, key_size))) {
		if ((partial && microfeed_file_block_get_size(block) < data_size) || !partial) {
			block = microfeed_file_resize_block_impl(database_index->database_table->database->file, block, data_size);
		}
		block_index = microfeed_file_get_index(database_index->database_table->database->file, block);
		for (iterator = microfeed_store_iterate(database_index->database_table->indices, NULL);
		     (index = microfeed_store_iterator_get(iterator, MicrofeedDatabaseIndex));
		     microfeed_store_iterator_next(iterator)) {
			if (index != database_index &&
			    (existing_index = microfeed_file_index_table_update_index(index->table, block_index, data, data_size)) != MICROFEED_FILE_INDEX_INVALID) {
				printf("*** This should not happen: index existed!\n");
				abort();
				microfeed_file_free_index(database_index->database_table->database->file, existing_index);
			}
		}
		microfeed_store_iterator_free(iterator);
		block = microfeed_file_get_block_impl(database_index->database_table->database->file, block_index);
		memcpy(block, data, data_size);
		microfeed_file_index_table_set_last_updated_time(database_index->table, time(NULL));
	} else {
		block = microfeed_file_allocate_block_impl(database_index->database_table->database->file, data_size);
		memcpy(block, data, data_size);
		block_index = microfeed_file_get_index(database_index->database_table->database->file, block);
		for (iterator = microfeed_store_iterate(database_index->database_table->indices, NULL);
		     (index = microfeed_store_iterator_get(iterator, MicrofeedDatabaseIndex));
		     microfeed_store_iterator_next(iterator)) {
			if ((existing_index = microfeed_file_index_table_replace_index(index->table, block_index)) != MICROFEED_FILE_INDEX_INVALID) {
				printf("*** This should not happen: index existed!\n");
				abort();
				microfeed_file_free_index(database_index->database_table->database->file, existing_index);
			}
		}
		microfeed_store_iterator_free(iterator);
	}
	
	microfeed_object_unlock(database_index->database_table->database, MicrofeedDatabase);
}

static void remove_block(MicrofeedDatabaseTable* database_table, void* block) {
	MicrofeedStoreIterator* iterator;
	MicrofeedDatabaseIndex* index;
	
	for (iterator = microfeed_store_iterate(database_table->indices, NULL);
	     (index = microfeed_store_iterator_get(iterator, MicrofeedDatabaseIndex));
	     microfeed_store_iterator_next(iterator)) {
		microfeed_file_index_table_remove_block(index->table, block);
	}
	microfeed_store_iterator_free(iterator);
	microfeed_file_free_block(database_table->database->file, block);
}
