/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE
#include <microfeed-provider/microfeedhttp.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedstore.h>

#include <curl/curl.h>
#include <openssl/hmac.h>
#include <openssl/crypto.h>
 
#include <string.h>

typedef struct _Buffer Buffer;

typedef struct _OAuth OAuth;

struct _MicrofeedHttp {
	Buffer* buffer;
	CURL* curl;
	char* cookie_filename;
	char* userpass;
	time_t server_time;
	time_t reply_start_time;
	OAuth* oauth;
};

struct _Buffer {
	void* data;
	size_t size;
	size_t reserved;
};

struct _OAuth {
	char* request_token_url;
	char* user_authorization_url;
	char* access_token_url;
	char* consumer_key;
	char* consumer_secret;
	MicrofeedHttpOAuthAuthorizeCallback authorize_callback;
	MicrofeedHttpOAuthAccessCallback access_callback;
	void* user_data;
	MicrofeedHttpOAuthSignatureMethod signature_method;
	char* access_token;
	char* access_token_secret;
};

static unsigned int serial = 1;
static MicrofeedMutex** mutexes = NULL;

static Buffer* buffer_new();
static void buffer_free(Buffer* buffer);
static void buffer_append(Buffer* buffer, const void* data, size_t size);

static OAuth* oauth_new(const char* request_token_url, const char* user_authorization_url, const char* access_token_url, const char* consumer_key, const char* consumer_secret, MicrofeedHttpOAuthAuthorizeCallback authorize_callback, MicrofeedHttpOAuthAccessCallback access_callback, void* user_data);
static void oauth_free(OAuth* oauth);

static size_t curl_header(void* ptr, size_t size, size_t nmemb, void* data);
static size_t curl_write(void* ptr, size_t size, size_t nmemb, void* data);
static char* append_oauth_authentication(MicrofeedHttp* http, const char* method, const char* url, const char* parameters);
static CURLcode do_http_get_data(MicrofeedHttp* http, const char* url);
static void openssl_locking_function(int mode, int n, const char *file, int line);
static unsigned long openssl_id_function(void);

void microfeed_http_init(int multithreaded) {
	int i;
	CURLcode code;

	if (multithreaded) {
		mutexes = (MicrofeedMutex**)malloc(CRYPTO_num_locks() * sizeof(MicrofeedMutex*)); 
		for (i = 0; i < CRYPTO_num_locks(); i++) {
			mutexes[i] = microfeed_mutex_new();
		} 
		CRYPTO_set_locking_callback(openssl_locking_function); 
		CRYPTO_set_id_callback(openssl_id_function);
	}

	if ((code = curl_global_init(CURL_GLOBAL_ALL)) != 0) {
		fprintf(stderr, "ERROR: curl_global_init failed: %d\n", code);
	}
}

void microfeed_http_cleanup() {
	int i;
	
	if (mutexes) {
		CRYPTO_set_locking_callback(NULL); 
		CRYPTO_set_id_callback(NULL);		
		for (i = 0; i < CRYPTO_num_locks(); i++) {
        	microfeed_mutex_free(mutexes[i]);
		}
		free(mutexes);
		mutexes = NULL;
	}
	curl_global_cleanup();
}

MicrofeedHttp* microfeed_http_new() {
	MicrofeedHttp* http = NULL;
	CURLcode code;
	
	http = microfeed_memory_allocate(MicrofeedHttp);
	http->buffer = buffer_new();

	if ((http->curl = curl_easy_init())) {
		if ((code = curl_easy_setopt(http->curl, CURLOPT_WRITEFUNCTION, curl_write)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_WRITEDATA, http->buffer)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_HEADERFUNCTION, curl_header)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_HEADERDATA, http)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_COOKIEFILE, "")) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_FOLLOWLOCATION, (long)1)) || 
		    (code = curl_easy_setopt(http->curl, CURLOPT_TIMEOUT, (long)300)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_NOSIGNAL, (long)1)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_USERAGENT, "Microfeed/" PACKAGE_VERSION))) {
			curl_easy_cleanup(http->curl);
			buffer_free(http->buffer);
			microfeed_memory_free(http);
			http = NULL;
		}
	} else {
		buffer_free(http->buffer);
		microfeed_memory_free(http);
		http = NULL;
	}

	return http;
}

void microfeed_http_free(MicrofeedHttp* http) {
	if (http) {
		curl_easy_cleanup(http->curl);
		buffer_free(http->buffer);
		free(http->cookie_filename);
		free(http->userpass);
		free(http);
	}
}

void microfeed_http_free_string(MicrofeedHttp* http, char* ptr) {
	curl_free(ptr);
}
	
/* NULL if error. Returned buffer may contain NULL bytes. Returned buffer must NOT be freed; it is reused in the next call. */
const void* microfeed_http_get_data(MicrofeedHttp* http, const char* url, size_t* size_return) {
	char* url_with_oauth = NULL;
	CURLcode code;
	long response;

	*size_return = 0;
	if (http->oauth) {
		if ((url_with_oauth = append_oauth_authentication(http, "GET", url, NULL))) {
			code = do_http_get_data(http, url_with_oauth);
			free(url_with_oauth);
			if (code) {
				
				return NULL;
			}
			if (http->oauth && (curl_easy_getinfo(http->curl, CURLINFO_RESPONSE_CODE, &response) ||
		                        response == 0 || response == 401)) {
				free(http->oauth->access_token);
				free(http->oauth->access_token_secret);
				http->oauth->access_token = http->oauth->access_token_secret = NULL;	
				if (http->oauth->access_callback) {
					http->oauth->access_callback(http, http->oauth->access_token, http->oauth->access_token_secret, http->oauth->user_data);
				}
				return NULL;
			}
		}
	} else {
		if (do_http_get_data(http, url)) {
			
			return NULL;
		}
	}
	*size_return = http->buffer->size;

	return http->buffer->data;
}

MicrofeedJson* microfeed_http_get_json(MicrofeedHttp* http, const char* url) {
	MicrofeedJson* json = NULL;
	const char* data;
	size_t size;

	if ((data = microfeed_http_get_data(http, url, &size))) {
		json = microfeed_json_new_from_data(http->buffer->data, http->buffer->size);
	}

	return json;
}

time_t microfeed_http_get_reply_start_time(MicrofeedHttp* http) {

	return http->reply_start_time;
}

time_t microfeed_http_get_server_time(MicrofeedHttp* http) {

	return http->server_time;
}

char* microfeed_http_post_data(MicrofeedHttp* http, const char* url, size_t* size_return, const char* post_data) {
	CURLcode code;
	struct curl_slist* slist = NULL;
	char* post_data_with_oauth = NULL;
	long response;

	*size_return = 0;
	http->buffer->size = 0;
	if (!(slist = curl_slist_append(slist, "Content-Type: application/x-www-form-urlencoded; charset=utf-8"))) {

		return NULL;
	}
	if (http->oauth) {
		if (!(post_data_with_oauth = append_oauth_authentication(http, "POST", url, post_data))) {
			
			return NULL;
		}
		if ((code = curl_easy_setopt(http->curl, CURLOPT_POST, 1)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_POSTFIELDS, post_data_with_oauth)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_URL, url)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, slist))) {
			free(post_data_with_oauth);
			curl_slist_free_all(slist);
		    
			return NULL;
		}
	} else {
		if ((code = curl_easy_setopt(http->curl, CURLOPT_POST, 1)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_POSTFIELDS, post_data)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_URL, url)) ||
		    (code = curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, slist))) {
			curl_slist_free_all(slist);
		    
			return NULL;
		}
	}
	http->reply_start_time = http->server_time = 0;
	if ((code = curl_easy_perform(http->curl)) != 0) {
		curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, NULL);
		curl_slist_free_all(slist);
		free(post_data_with_oauth);

		return NULL;
	}
	if (http->oauth && (curl_easy_getinfo(http->curl, CURLINFO_RESPONSE_CODE, &response) ||
		                response == 0 || response == 401)) {
		free(http->oauth->access_token);
		free(http->oauth->access_token_secret);
		http->oauth->access_token = http->oauth->access_token_secret = NULL;	
		if (http->oauth->access_callback) {
			http->oauth->access_callback(http, http->oauth->access_token, http->oauth->access_token_secret, http->oauth->user_data);
		}
		
		return NULL;
	}
	free(post_data_with_oauth);
	curl_easy_setopt(http->curl, CURLOPT_HTTPHEADER, NULL);
	curl_slist_free_all(slist);

	*size_return = http->buffer->size;

	return http->buffer->data;
}

time_t microfeed_http_parse_date(MicrofeedHttp* http, const char* datestring) {
	time_t retvalue;
	
	if ((retvalue = curl_getdate(datestring, NULL)) == -1) {
		retvalue = 0;
	}
	
	return retvalue;
}

MicrofeedJson* microfeed_http_post_json(MicrofeedHttp* http, const char* url, const char* post_data) {
	MicrofeedJson* json = NULL;
	const char* data;
	size_t size;

	if ((data = microfeed_http_post_data(http, url, &size, post_data))) {
		json = microfeed_json_new_from_data(http->buffer->data, http->buffer->size);
	}
	
	return json;
}

int microfeed_http_set_cookie_file(MicrofeedHttp* http, const char* filename) {
	int retvalue = 1;
	
	if (http->cookie_filename) {
		free(http->cookie_filename);
	}
	http->cookie_filename = strdup(filename);
	if (curl_easy_setopt(http->curl, CURLOPT_COOKIEFILE, http->cookie_filename) || 
	    curl_easy_setopt(http->curl, CURLOPT_COOKIEJAR, http->cookie_filename)) {

		retvalue = 0;
	}

	return retvalue;
}

int microfeed_http_set_basic_authentication(MicrofeedHttp* http, const char* userpass) {
	int retvalue = 1;
	
	if (http->userpass) {
		free(http->userpass);
	}
	if (userpass) {
		http->userpass = strdup(userpass);
		if (curl_easy_setopt(http->curl, CURLOPT_HTTPAUTH, CURLAUTH_BASIC) ||
		    curl_easy_setopt(http->curl, CURLOPT_USERPWD, http->userpass)) {

			retvalue = 0;
		}
	} else {
		http->userpass = NULL;
		if (curl_easy_setopt(http->curl, CURLOPT_USERPWD, NULL)) {

			retvalue = 0;
		}
	}	
	
	return retvalue;
}

int microfeed_http_set_oauth_authentication(MicrofeedHttp* http,  const char* request_token_url, const char* user_authorization_url, const char* access_token_url, const char* consumer_key, const char* consumer_secret, MicrofeedHttpOAuthAuthorizeCallback authorize_callback, MicrofeedHttpOAuthAccessCallback access_callback, void* user_data) {
	oauth_free(http->oauth);
	http->oauth = oauth_new(request_token_url, user_authorization_url, access_token_url, consumer_key, consumer_secret, authorize_callback, access_callback, user_data);

	return 1;
}

void microfeed_http_unset_oauth_authentication(MicrofeedHttp* http) {
	oauth_free(http->oauth);
	http->oauth = NULL;
}


void microfeed_http_set_oauth_access_token(MicrofeedHttp* http, const char* access_token, const char* access_token_secret) {
	free(http->oauth->access_token);
	free(http->oauth->access_token_secret);
	http->oauth->access_token = strdup(access_token);
	http->oauth->access_token_secret = strdup(access_token_secret);
}

/* ********************************* *
 *  Buffer
 * ********************************* */ 

static Buffer* buffer_new() {
	Buffer* buffer;
	
	buffer = microfeed_memory_allocate(Buffer);
	buffer->reserved = 1;
	buffer->data = malloc(buffer->reserved);
	*((char*)buffer->data) = 0;
	
	return buffer;
}

static void buffer_free(Buffer* buffer) {
	free(buffer->data);
	microfeed_memory_free(buffer);
}

static void buffer_append(Buffer* buffer, const void* data, size_t size) {
	if (buffer->size + size >= buffer->reserved) {
		buffer->reserved += buffer->size + size + 1;
		buffer->data = realloc(buffer->data, buffer->reserved);
	}
	memcpy(buffer->data + buffer->size, data, size);
	buffer->size += size;
	*((char*)(buffer->data + buffer->size)) = 0;
}

/* ********************************* *
 *  OAuth
 * ********************************* */ 

static OAuth* oauth_new(const char* request_token_url, const char* user_authorization_url, const char* access_token_url, const char* consumer_key, const char* consumer_secret, MicrofeedHttpOAuthAuthorizeCallback authorize_callback, MicrofeedHttpOAuthAccessCallback access_callback, void* user_data) {
	OAuth* oauth;
	
	oauth = microfeed_memory_allocate(OAuth);
	oauth->request_token_url = strdup(request_token_url);
	oauth->user_authorization_url = strdup(user_authorization_url);
	oauth->access_token_url = strdup(access_token_url);
	oauth->consumer_key = strdup(consumer_key);
	oauth->consumer_secret = strdup(consumer_secret);
	oauth->authorize_callback = authorize_callback;
	oauth->access_callback = access_callback;
	oauth->user_data = user_data;
	oauth->signature_method = MICROFEED_HTTP_OAUTH_SIGNATURE_METHOD_HMAC_SHA1;
	
	return oauth;
}

void oauth_free(OAuth* oauth) {
	if (oauth) {
		free(oauth->request_token_url);
		free(oauth->user_authorization_url);
		free(oauth->access_token_url);
		free(oauth->consumer_key);
		free(oauth->consumer_secret);
		free(oauth->access_token);
		free(oauth->access_token_secret);
		microfeed_memory_free(oauth);	
	}
}

/* ********************************* *
 *  Internal functions
 * ********************************* */ 

static size_t curl_header(void* ptr, size_t size, size_t nmemb, void* data) {
	MicrofeedHttp* http;
	char* s;
	
	http = (MicrofeedHttp*)data;
	
	if (!http->reply_start_time) {
		http->reply_start_time = time(NULL);
	}

	if (size * nmemb > 5 && !strncasecmp(ptr, "date:", 5)) {
		s = strndup(ptr + 5, size * nmemb - 5);
		http->server_time = curl_getdate(s, NULL);
		free(s);
	}
	
	return size * nmemb;
}

static size_t curl_write(void* ptr, size_t size, size_t nmemb, void* data) {
	Buffer* buffer;
	
	buffer = (Buffer*)data;
	buffer_append(buffer, ptr, size * nmemb);
	
	return size * nmemb;
}

static char* hmac_sha1(const char* s, const char* key, size_t* length_return) {
	char* hmac;
	unsigned int u;

	u = EVP_MAX_MD_SIZE;
	hmac = (char*)malloc(u);
	HMAC(EVP_sha1(), key, strlen(key), (const unsigned char*)s, strlen(s), hmac, &u);
	hmac = (char*)realloc(hmac, u);
	*length_return = (size_t)u;
	
	return hmac;
}

static void copy_parameters_into_store(const char* parameters, MicrofeedStore* store) {
	const char* end;
	char* copy;
	char* old;
	
	while (*parameters) {
		while (*parameters == '?' || *parameters == '&') {
			parameters++;
		}
		if (*parameters) {
			end = strchrnul(parameters, '&');
			copy = strndup(parameters, end - parameters);
			if ((old = microfeed_store_replace(store, copy, char))) {
				free(old);
			}
			parameters = end;
		}
	}
}

static char* generate_oauth_signature(const char* method, const char* url, const char* parameters, const char* oauth_parameters, const char* consumer_secret, const char* token_secret) {
	MicrofeedStore* parameters_store;
	char* s;
	char* copy;
	char* escaped_url;
	Buffer* buffer;
	MicrofeedStoreIterator* iterator;
	char* escaped_parameters;
	char* signature_base_string;
	char* key;
	char* signature;
	size_t length;
	char* base64;
	char* escaped;
	
	parameters_store = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, microfeed_store_get_key_direct);
	if ((s = strchr(url, '?'))) {
		copy = strndup(url, s - url);
		escaped_url = microfeed_util_string_percent_encoding_escape(copy);
		free(copy);
		copy_parameters_into_store(s + 1, parameters_store);
	} else {
		escaped_url = microfeed_util_string_percent_encoding_escape(url);
	}
	if (parameters) {
		copy_parameters_into_store(parameters, parameters_store);
	}
	copy_parameters_into_store(oauth_parameters, parameters_store);
	
	buffer = buffer_new();
	for (iterator = microfeed_store_iterate(parameters_store, NULL); (s = microfeed_store_iterator_get(iterator, char)); microfeed_store_iterator_next(iterator)) {
		if (buffer->size) {
			buffer_append(buffer, "&", 1);
		}
		buffer_append(buffer, s, strlen(s));	
	}	
	escaped_parameters = microfeed_util_string_percent_encoding_escape((char*)buffer->data);
	buffer_free(buffer);
	signature_base_string = microfeed_util_string_concatenate(method, "&", escaped_url, "&", escaped_parameters, NULL);

	free(escaped_parameters);
	free(escaped_url);
	if (token_secret) {
		key = microfeed_util_string_concatenate(consumer_secret, "&", token_secret, NULL);
	} else {
		key = microfeed_util_string_concatenate(consumer_secret, "&", NULL);
	}
	signature = hmac_sha1(signature_base_string, key, &length);
	free(key);
	base64 = microfeed_util_string_base64_encode(signature, length);
	free(signature);
	escaped = microfeed_util_string_percent_encoding_escape(base64);
	free(base64);
	
	return escaped;	
} 

static void parse_oauth_reply(const char* s, size_t length, char** token_return, char** token_secret_return) {
	const char* s1;
	const char* s2;

	while ((s1 = memchr(s, '=', length))) {
		if (!(s2 = memchr(s1 + 1, '&', length - (s1 - s)))) {
			s2 =s + length;
		}
		if (!strncmp(s, "oauth_token", s1 - s)) {
			free(*token_return);
			*token_return = strndup(s1 + 1, s2 - s1 -1);
		} else if (!strncmp(s, "oauth_token_secret", s1 - s)) {
			free(*token_secret_return);
			*token_secret_return = strndup(s1 + 1, s2 - s1 -1);				
		}
		if (s2 - s == length) {
			break;
		}
		s = s2 + 1;
		length -= s2 - s + 1;
	}
}

static char* append_oauth_authentication(MicrofeedHttp* http, const char* method, const char* url, const char* parameters) {
	char* result = NULL;
	char nonce[128];
	char timestamp[128];
	char* oauth_parameters;
	char* signature;
	char* oauth_url;
	char* request_token = NULL;
	char* request_token_secret = NULL;
	long response;
	
	if (!http->oauth->access_token) {
		snprintf(timestamp, 128, "%ld", time(NULL));
		snprintf(nonce, 128, "%u", serial++);	
		oauth_parameters = microfeed_util_string_concatenate("oauth_consumer_key=", http->oauth->consumer_key,
		 "&oauth_nonce=", nonce, "&oauth_signature_method=HMAC-SHA1&oauth_timestamp=", timestamp,
		 "&oauth_version=1.0", NULL);
		signature = generate_oauth_signature("GET", http->oauth->request_token_url, NULL, oauth_parameters, http->oauth->consumer_secret, NULL);
		oauth_url = microfeed_util_string_concatenate(http->oauth->request_token_url, (strchr(http->oauth->request_token_url, '?') ? "&" : "?"), oauth_parameters, "&oauth_signature=", signature, NULL);
		if (!do_http_get_data(http, oauth_url) && http->buffer->size > 30) {
			if (!curl_easy_getinfo(http->curl, CURLINFO_RESPONSE_CODE, &response) &&
			    response != 0 && response != 401) {
				parse_oauth_reply(http->buffer->data, http->buffer->size, &request_token, &request_token_secret);
			}
		}
		free(oauth_url);
		if (request_token && request_token_secret) {
			oauth_url = microfeed_util_string_concatenate(http->oauth->user_authorization_url, (strchr(http->oauth->user_authorization_url, '?') ? "&" : "?"), "oauth_token=", request_token, NULL);
			if (http->oauth->authorize_callback(http, oauth_url, http->oauth->user_data)) {
				snprintf(timestamp, 128, "%ld", time(NULL));
				snprintf(nonce, 128, "%u", serial++);	
				oauth_parameters = microfeed_util_string_concatenate("oauth_consumer_key=", http->oauth->consumer_key,
				 "&oauth_nonce=", nonce, "&oauth_signature_method=HMAC-SHA1&oauth_timestamp=", timestamp,
				 "&oauth_token=", request_token, "&oauth_version=1.0", NULL);
				signature = generate_oauth_signature("GET", http->oauth->access_token_url, NULL, oauth_parameters, http->oauth->consumer_secret, request_token_secret);
				oauth_url = microfeed_util_string_concatenate(http->oauth->access_token_url, (strchr(http->oauth->request_token_url, '?') ? "&" : "?"), oauth_parameters, "&oauth_signature=", signature, NULL);
				if (!do_http_get_data(http, oauth_url) && http->buffer->size > 30) {
					if (!curl_easy_getinfo(http->curl, CURLINFO_RESPONSE_CODE, &response) &&
					    response != 0 && response != 401) {
						parse_oauth_reply(http->buffer->data, http->buffer->size, &http->oauth->access_token, &http->oauth->access_token_secret);
					}
				}				
			}
		}
		if (http->oauth->access_token && http->oauth->access_token_secret) {
			if (http->oauth->access_callback) {
				http->oauth->access_callback(http, http->oauth->access_token, http->oauth->access_token_secret, http->oauth->user_data);
			}
		} else {
			free(http->oauth->access_token);
			free(http->oauth->access_token_secret);
			http->oauth->access_token = http->oauth->access_token_secret = NULL;
		}
		
	}
	if (http->oauth->access_token) {
		snprintf(timestamp, 128, "%ld", time(NULL));
		snprintf(nonce, 127, "%u", serial++);	
		oauth_parameters = microfeed_util_string_concatenate("oauth_consumer_key=", http->oauth->consumer_key,
		 "&oauth_nonce=", nonce, "&oauth_signature_method=HMAC-SHA1&oauth_timestamp=", timestamp,
		 "&oauth_token=", http->oauth->access_token, "&oauth_version=1.0", NULL);
		signature = generate_oauth_signature(method, url, parameters, oauth_parameters, http->oauth->consumer_secret, http->oauth->access_token_secret);
		if (parameters) {
			result = microfeed_util_string_concatenate(parameters, "&", oauth_parameters, "&oauth_signature=", signature, NULL);
		} else {
			result = microfeed_util_string_concatenate(url, (strchr(http->oauth->request_token_url, '?') ? "&" : "?"), oauth_parameters, "&oauth_signature=", signature, NULL);		
		}
		free(signature);
	}

	return result;
}

static CURLcode do_http_get_data(MicrofeedHttp* http, const char* url) {
	CURLcode code;

	http->buffer->size = 0;
	if (!(code = curl_easy_setopt(http->curl, CURLOPT_POST, 0)) &&
	    !(code = curl_easy_setopt(http->curl, CURLOPT_URL, url))) {
		http->reply_start_time = http->server_time = 0;
		code = curl_easy_perform(http->curl);
	}
	
	return code;
}

static void openssl_locking_function(int mode, int n, const char *file, int line) {
	if (mode & CRYPTO_LOCK) {
		microfeed_mutex_lock(mutexes[n]);
	} else {
		microfeed_mutex_unlock(mutexes[n]);
	} 
}

static unsigned long openssl_id_function(void) {
	
	return microfeed_thread_get_id(microfeed_thread_get_current());
}
