/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedobject.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#define DELTA 10

struct _MicrofeedStore {
	void** data;
	unsigned int size;
	unsigned int reserved;
	MicrofeedStoreCompareKeysFunction compare_keys;
	MicrofeedStoreGetKeyFunction get_key;
	MicrofeedStoreIterator* iterators;
	int unsorted : 1;
	int weak_references : 1;
};

struct _MicrofeedStoreIterator {
	MicrofeedStoreIterator* previous;
	MicrofeedStoreIterator* next;
	
	MicrofeedStore* store;
	unsigned int index;
	void* current_data;
};

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index_return, void** data_return);
static void remove_index(MicrofeedStore* store, unsigned int index);

MicrofeedStore* microfeed_store_new_sorted_data(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_memory_allocate(MicrofeedStore);
	store->compare_keys = compare_keys;
	store->get_key = get_key;
	
	return store;
}

MicrofeedStore* microfeed_store_new_unsorted_data(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_store_new_sorted_data(compare_keys, get_key);
	store->unsorted = 1;
	
	return store;
}

MicrofeedStore* microfeed_store_new_sorted_weak_references(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_store_new_sorted_data(compare_keys, get_key);
	store->weak_references = 1;
	
	return store;
}

MicrofeedStore* microfeed_store_new_unsorted_weak_references(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key) {
	MicrofeedStore* store;
	
	store = microfeed_store_new_sorted_data(compare_keys, get_key);
	store->unsorted = 1;
	store->weak_references = 1;
	
	return store;
}

void microfeed_store_free(MicrofeedStore* store) {
	MicrofeedStoreIterator* iterator;
	
	if (store->weak_references) {
		while (store->size > 0) {
			microfeed_weak_reference_unref((MicrofeedWeakReference*)store->data[store->size - 1]);
			store->size--;
		}	
	}
	
	for (iterator = store->iterators; iterator; iterator = iterator->next) {
		iterator->store = NULL;
	}	
	free(store->data);
	store->data = NULL;
	store->iterators = NULL;
	store->size = store->reserved = 0;
	free(store);
}

int microfeed_store_is_sorted(MicrofeedStore* store) {
	
	return !store->unsorted;
}

int microfeed_store_has_weak_references(MicrofeedStore* store) {

	return store->weak_references;
}

void microfeed_store_foreach(MicrofeedStore* store, MicrofeedStoreForeachFunction foreach, void* user_data) {
	int i;
	MicrofeedObject* object;

	if (store->weak_references) {
		for (i = 0; i < store->size; i++) {
			if ((object = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[i]))) {
				foreach(object, user_data);
				microfeed_object_unref_generic(object);
			} else {
				remove_index(store, i--);
			}
		}
	} else {
		for (i = 0; i < store->size; i++) {
			foreach(store->data[i], user_data);
		}
	}
}

void* microfeed_store_get_impl(MicrofeedStore* store, const void* key) {
	unsigned int index;
	void* data;

	get_index(store, key, &index, &data);

	return data;
}

void* microfeed_store_get_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;
	
	if (index < store->size) {
		if (store->weak_references) {
			data = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[index]);
		} else {
			data = store->data[index];
		}
	}
	
	return data;	
}

unsigned int microfeed_store_get_size(MicrofeedStore* store) {

	return store->size;
}

int microfeed_store_insert(MicrofeedStore* store, void* data) {
	int retvalue = 0;
	unsigned int index;
	void* existing;
	MicrofeedStoreIterator* iterator;

	if (!get_index(store, store->get_key(data), &index, &existing)) {
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}
	
		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
			        (store->size - index) * sizeof(void*));
		}
		store->size++;
		if (store->weak_references) {
			store->data[index] = microfeed_object_get_weak_reference_generic(MICROFEED_OBJECT(data));
		} else {
			store->data[index] = data;
		}

		retvalue = 1;
		
		for (iterator = store->iterators; iterator; iterator = iterator->next) {
			if (iterator->index <= index) {
				iterator->index++;
			}
		}
	}
	
	return retvalue;
}

void* microfeed_store_replace_impl(MicrofeedStore* store, void* data) {
	unsigned int index;
	void* existing;
	
	if (!get_index(store, store->get_key(data), &index, &existing)) {
		if (store->size == store->reserved) {
			store->reserved += DELTA;
			store->data = realloc(store->data, store->reserved * sizeof(void*));
		}

		if (index < store->size) {
			memmove(store->data + index + 1, store->data + index,
		        	(store->size - index) * sizeof(void*));
		}
		store->size++;
	} if (store->weak_references) {
		microfeed_weak_reference_unref((MicrofeedWeakReference*)existing);
	}
	if (store->weak_references) {
		store->data[index] = microfeed_object_get_weak_reference_generic(MICROFEED_OBJECT(data));
	} else {
		store->data[index] = data;
	}
	
	return existing;
}

int microfeed_store_remove(MicrofeedStore* store, const void* data) {
	int retvalue = 0;
	unsigned int index;
	void* existing;

	if (get_index(store, store->get_key(data), &index, &existing)) {
		remove_index(store, index);
		retvalue = 1;
	}

	return retvalue;
}

void* microfeed_store_remove_key_impl(MicrofeedStore* store, const void* key) {
	void* data;
	unsigned int index;

	if (get_index(store, key, &index, &data)) {
		remove_index(store, index);
	}

	return data;
}

void* microfeed_store_remove_index_impl(MicrofeedStore* store, unsigned int index) {
	void* data = NULL;

	if (index < store->size) {
		if (store->weak_references) {
			data = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[index]);
		} else {
			data = store->data[index];
		}
		remove_index(store, index);
	}

	return data;
}

void microfeed_store_remove_and_free_all(MicrofeedStore* store, MicrofeedStoreFreeDataFunction free_data) {
	MicrofeedObject* object;

	if (store->weak_references) {
		while (store->size > 0) {
			if ((object = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[store->size - 1]))) {
				free_data(object);
			}
			microfeed_weak_reference_unref((MicrofeedWeakReference*)store->data[store->size -1]);
			store->size--;
		}	
	} else {
		while (store->size > 0) {
			free_data(store->data[store->size - 1]);
			store->size--;
		}
	}
	if (store->reserved > DELTA) {
		store->reserved = DELTA;
		store->data = realloc(store->data, store->reserved * sizeof(void*));
	}
}

MicrofeedStoreIterator* microfeed_store_iterate(MicrofeedStore* store, const void* start_key) {
	MicrofeedStoreIterator* iterator;
	unsigned int index;
	void* existing;
	
	iterator = microfeed_memory_allocate(MicrofeedStoreIterator);
	iterator->store = store;
	if (start_key) {
		if (get_index(store, start_key, &index, &existing)) {
			iterator->index = index;
		} else {
			iterator->index = store->size;
		}
	} else {
		iterator->index = 0;
	}
	if (store->iterators) {
		store->iterators->previous = iterator;
		iterator->next = store->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	store->iterators = iterator;
	
	return iterator;
}

void microfeed_store_sort(MicrofeedStore* store, MicrofeedStoreCompareDatasFunction compare_datas, void* user_data) {
	unsigned int i;
	int j;
	void* data;
	
/* TODO: Does not support weak references */
microfeed_assert(!store->weak_references);

	if (store->unsorted) {
		for (i = 1; i < store->size; i++) {	
			data = store->data[i];
			for (j = i - 1; j >= 0 && compare_datas(store->data[j], data, j, i, user_data) > 0; j = j - 1) {
				store->data[j + 1] = store->data[j];
			}
			store->data[j + 1] = data;
		}
	}
}

void microfeed_store_clean_weak_references(MicrofeedStore* store) {
	int i;
	MicrofeedObject* object;
	
	if (store->weak_references) {
		for (i = 0; i < store->size; i++) {
			if ((object = microfeed_weak_reference_get_object_generic(store->data[i]))) {
				microfeed_object_unref_generic(object);
			} else {
				remove_index(store, i--);
			}
		}
	}
}

void microfeed_store_iterator_free(MicrofeedStoreIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->store) {
		iterator->store->iterators = iterator->next;
	}
	if (iterator->store->weak_references && iterator->current_data) {
		microfeed_object_unref_generic(iterator->current_data);
	}

	iterator->store = NULL;
	iterator->current_data = NULL;
	microfeed_memory_free(iterator);
}

void* microfeed_store_iterator_get_impl(MicrofeedStoreIterator* iterator) {
	if (!iterator->current_data && iterator->store && iterator->index < iterator->store->size) {
		if (iterator->store->weak_references) {
			iterator->current_data = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)iterator->store->data[iterator->index]);
			if (!iterator->current_data) {
				remove_index(iterator->store, iterator->index++);

				return microfeed_store_iterator_get_impl(iterator);
			}
		} else {
			iterator->current_data = iterator->store->data[iterator->index];
		}
	}
	
	return iterator->current_data;
}

void microfeed_store_iterator_next(MicrofeedStoreIterator* iterator) {
	if (iterator->store && (iterator->index < iterator->store->size || iterator->index == (unsigned int)-1)) {
		iterator->index++;
	}
	if (iterator->store->weak_references && iterator->current_data) {
		microfeed_object_unref_generic(iterator->current_data);
	}
	iterator->current_data = NULL;
}

int microfeed_store_compare_keys_direct(const void* key1, const void* key2) {

	return (key1 == key2 ? 0 : (key1 < key2 ? -1 : 1));
}

const void* microfeed_store_get_key_direct(const void* data) {
	
	return data;
}

static int get_index(MicrofeedStore* store, const void* key, unsigned int* index_return, void** data_return) {
	int retval = 0;
	MicrofeedObject* object;
	const void* k;
	int result;
	int i, min, max;

	*data_return = NULL;
	if (store->weak_references) {
		if (store->unsorted) {
			*index_return = store->size;
			if (key) {
				for (i = 0; i < store->size; i++) {
					if ((object = microfeed_weak_reference_get_object_generic(store->data[i]))) {
						k = store->get_key(object);
						if (k && store->compare_keys(key, k) == 0) {
							*index_return = i;
							*data_return = object;
							retval = 1;
							break;
						} else {
							microfeed_object_unref_generic(object);
						}
					} else {
						remove_index(store, i--);
					}
				}
			}
		} else {
			if (store->size == 0) {
				*index_return = 0;
			} else if (!(object = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[0]))) {
				remove_index(store, 0);
				retval = get_index(store, key, index_return, data_return);
			} else if ((result = store->compare_keys(key, store->get_key(object))) == 0) {
				*index_return = 0;
				*data_return = object;
				retval = 1;
			} else if (microfeed_object_unref_generic(object), result < 0) {
				*index_return = 0;
			} else if (store->size == 1) {
				*index_return = 1;
			} else if (!(object = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[store->size - 1]))) {
				remove_index(store, store->size - 1);
				retval = get_index(store, key, index_return, data_return);
			} else if ((result = store->compare_keys(key, store->get_key(object))) == 0) {
				*index_return = store->size -1;
				*data_return = object;
				retval = 1;
			} else if (microfeed_object_unref_generic(object), result > 0) {
				*index_return = store->size;
			} else if (store->size == 2) {
				*index_return = store->size - 1;
			} else {
				min = i = 0;
				max = store->size - 1;

				while (min <= max) {
					i = (min + max) / 2;
					if (!(object = microfeed_weak_reference_get_object_generic((MicrofeedWeakReference*)store->data[i]))) {
						remove_index(store, i);
						max--;
					} else if ((result = store->compare_keys(key, store->get_key(object))) == 0) {
						*data_return = object;
						retval = 1;
						break;
					} else if (microfeed_object_unref_generic(object), result < 0) {
						max = i - 1;
					} else {
						i++;
						min = i;
					}
				}

				*index_return = i;
			}
		}
	} else {
		if (store->unsorted) {
			*index_return = store->size;
			if (key) {
				for (i = 0; i < store->size; i++) {
					k = store->get_key(store->data[i]);
					if (k && store->compare_keys(key, k) == 0) {
						*index_return = i;
						*data_return = store->data[i];
						retval = 1;
						break;
					}
				}
			}
		} else {
			if (store->size == 0) {
				*index_return = 0;
			} else if ((result = store->compare_keys(key, store->get_key(store->data[0]))) == 0) {
				*index_return = 0;
				*data_return = store->data[0];
				retval = 1;	
			} else if (result < 0) {
				*index_return = 0;
			} else if (store->size == 1) {
				*index_return = 1;
			} else if ((result = store->compare_keys(key, store->get_key(store->data[store->size - 1]))) == 0) {
				*index_return = store->size -1;
				*data_return = store->data[store->size - 1];
				retval = 1;
			} else if (result > 0) {
				*index_return = store->size;
			} else if (store->size == 2) {
				*index_return = store->size - 1;
			} else {
				min = i = 0;
				max = store->size - 1;

				while (min <= max) {
					i = (min + max) / 2;
					if ((result = store->compare_keys(key, store->get_key(store->data[i]))) == 0) {
						*data_return = store->data[i];
						retval = 1;
						break;
					} else if (result < 0) {
						max = i - 1;
					} else {
						i++;
						min = i;
					}
				}

				*index_return = i;
			}
		}
	}

	return retval;
}

static void remove_index(MicrofeedStore* store, unsigned int index) {
	MicrofeedStoreIterator* iterator;

	if (store->weak_references) {
		microfeed_weak_reference_unref((MicrofeedWeakReference*)store->data[index]);
	}

	if (index + 1 < store->size) {
		memmove(store->data + index, store->data + index + 1,
		        (store->size - index - 1) * sizeof(void*));
	}

	store->size--;
	if (store->size + 2 * DELTA == store->reserved) {
		store->reserved -= DELTA;
		store->data = realloc(store->data, store->reserved * sizeof(void*));
	}

	for (iterator = store->iterators; iterator; iterator = iterator->next) {
		if (iterator->index <= index) {
			iterator->index--;
		}
	}
}
