/*
 * apiclient.cpp -- Contains all methods related to Last.fm API
 *
 * Copyright (C) 2010 Andrea Grandi
 * Authors: Andrea Grandi <a.grandi@gmail.com>
 *
 * This file is part of LastGo and is published under the GNU GPLv2
 * See the README file for more details.
 */

#include "apiclient.h"
#include <QXmlStreamReader>
#include <QDomDocument>
#include <QDebug>
#include <QMap>

ApiClient::ApiClient(QString key, QString secret, QObject *parent) :
    QObject(parent)
{
    api_key = key;
    api_secret = secret;
}

void ApiClient::authGetMobileSession(QString username, QString password)
{
    Utils *u = new Utils();

    QString token = u->getAuthToken(username, password);

    QMap<QString, QString> params;
    params["username"] = username;
    params["authToken"] = token;
    params["api_key"] = api_key;
    params["method"] = "auth.getMobileSession";

    QString signature = u->buildSignature(params, api_secret);
    params["api_sig"] = signature;

    QString api_url = u->buildApiUrl(LASTFM_BASE_URL, params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(authGetMobileSessionReply(QString)));
    r->apiRequest(api_url);
}

void ApiClient::authGetMobileSessionReply(QString data)
{
    QXmlStreamReader xml;
    xml.addData(data);

    QMap<QString, QString> response;
    QString status;

    while (!xml.atEnd())
    {
        xml.readNext();

        if (xml.isStartElement())
        {
            if(xml.name() == "lfm")
            {
                status = xml.attributes().value("status").toString();
                response["status"] = status;
            }
            else
            {
                if(status == "ok")
                {
                    if(xml.name() == "name")
                        response[xml.name().toString()] = xml.readElementText();
                    if(xml.name() == "key")
                        response[xml.name().toString()] = xml.readElementText();
                    if(xml.name() == "subscriber")
                        response[xml.name().toString()] = xml.readElementText();
                }
                else if(status == "failed")
                {
                    if(xml.name() == "error")
                    {
                        response["error"] = xml.attributes().value("code").toString();
                        response["message"] = xml.readElementText();
                    }
                }
            }
        }
    }

    emit authGetMobileSessionData(response);
}

void ApiClient::userShout(QString user, QString message, QString sk)
{
    Utils *u = new Utils();

    QMap<QString, QString> params;
    params["user"] = user;
    params["message"] = message;
    params["method"] = "user.shout";
    params["api_key"] = api_key;
    params["sk"] = sk;

    QString signature = u->buildSignature(params, api_secret);
    params["api_sig"] = signature;

    QByteArray post_params = u->buildPostParameters(params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(userShoutReply(QString)));
    r->apiRequestPost(LASTFM_BASE_URL, post_params);
}

void ApiClient::userShoutReply(QString data)
{
    QXmlStreamReader xml;
    xml.addData(data);

    QMap<QString, QString> response;
    QString status;

    while (!xml.atEnd())
    {
        xml.readNext();

        if (xml.isStartElement())
        {
            if(xml.name() == "lfm")
            {
                status = xml.attributes().value("status").toString();
                response["status"] = status;
            }
            else
            {
                if(status == "failed")
                {
                    if(xml.name() == "error")
                    {
                        response["error"] = xml.attributes().value("code").toString();
                        response["message"] = xml.readElementText();
                    }
                }
            }
        }
    }

    emit userShoutData(response);
}

void ApiClient::radioTune(QString station, QString sk)
{
    Utils *u = new Utils();

    QMap<QString, QString> params;
    params["station"] = station;
    params["method"] = "radio.tune";
    params["api_key"] = api_key;
    params["sk"] = sk;

    QString signature = u->buildSignature(params, api_secret);
    params["api_sig"] = signature;

    QByteArray post_params = u->buildPostParameters(params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(radioTuneReply(QString)));
    r->apiRequestPost(LASTFM_BASE_URL, post_params);
}

void ApiClient::radioTuneReply(QString data)
{
    QXmlStreamReader xml;
    xml.addData(data);

    QMap<QString, QString> response;
    QString status;

    while (!xml.atEnd())
    {
        xml.readNext();

        if (xml.isStartElement())
        {
            if(xml.name() == "lfm")
            {
                status = xml.attributes().value("status").toString();
                response["status"] = status;
            }
            else
            {
                if(status == "ok")
                {
                    if(xml.name() == "type")
                        response[xml.name().toString()] = xml.readElementText();
                    if(xml.name() == "name")
                        response[xml.name().toString()] = xml.readElementText();
                    if(xml.name() == "url")
                        response[xml.name().toString()] = xml.readElementText();
                    if(xml.name() == "supportsdiscovery")
                        response[xml.name().toString()] = xml.readElementText();
                }
                else if(status == "failed")
                {
                    if(xml.name() == "error")
                    {
                        response["error"] = xml.attributes().value("code").toString();
                        response["message"] = xml.readElementText();
                    }
                }
            }
        }
    }

    emit radioTuneData(response);
}

void ApiClient::radioGetPlaylist(QString sk)
{
    Utils *u = new Utils();

    QMap<QString, QString> params;
    params["method"] = "radio.getplaylist";
    params["api_key"] = api_key;
    params["sk"] = sk;

    QString signature = u->buildSignature(params, api_secret);
    params["api_sig"] = signature;

    QString api_url = u->buildApiUrl(LASTFM_BASE_URL, params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(radioGetPlaylistReply(QString)));
    r->apiRequest(api_url);
}

void ApiClient::radioGetPlaylistReply(QString data)
{
    QDomDocument doc("Playlist");
    doc.setContent(data.toUtf8());

    QDomElement elem = doc.documentElement();
    QString status = elem.attribute("status");

    QList<LastfmTrack*> track_list;

    if(status == "ok")
    {
        QDomNodeList tracks = elem.elementsByTagName("playlist").
                              at(0).toElement().elementsByTagName("trackList").
                              at(0).toElement().elementsByTagName("track");

        for(int i = 0; i < tracks.count(); i++)
        {
            LastfmTrack *track = new LastfmTrack();

            track->Location = tracks.at(i).toElement().elementsByTagName("location").at(0).toElement().text();
            track->Title = tracks.at(i).toElement().elementsByTagName("title").at(0).toElement().text();
            track->Identifier = tracks.at(i).toElement().elementsByTagName("identifier").at(0).toElement().text();
            track->Album = tracks.at(i).toElement().elementsByTagName("album").at(0).toElement().text();
            track->Creator = tracks.at(i).toElement().elementsByTagName("creator").at(0).toElement().text();
            track->Duration = tracks.at(i).toElement().elementsByTagName("duration").at(0).toElement().text();
            track->ArtistPage = tracks.at(i).toElement().elementsByTagName("artistpage").at(0).toElement().text();
            track->AlbumPage = tracks.at(i).toElement().elementsByTagName("albumpage").at(0).toElement().text();
            track->TrackPage = tracks.at(i).toElement().elementsByTagName("trackpage").at(0).toElement().text();
            track->Image = tracks.at(i).toElement().elementsByTagName("image").at(0).toElement().text();

            track_list.append(track);
        }

        emit radioGetPlaylistData(track_list);
    }
    else
    {
        QMap<QString, QString> error;
        QDomElement error_node = elem.elementsByTagName("error").at(0).toElement();
        error["message"] = error_node.text();
        error["error"] = error_node.attribute("code");

        emit radioGetPlaylistError(error);
    }
}

void ApiClient::scrobblerHandshake(QString protocol, QString clientId, QString clientVersion,
                                   QString user, QString timestamp, QString token, QString api_key, QString sk)
{
    Utils *u = new Utils();

    QMap<QString, QString> params;
    params["hs"] = "true";
    params["p"] = protocol;
    params["c"] = clientId;
    params["v"] = clientVersion;
    params["u"] = user;
    params["t"] = timestamp;
    params["a"] = token;
    params["api_key"] = api_key;
    params["sk"] = sk;

    QByteArray post_params = u->buildPostParameters(params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(scrobblerHandshakeReply(QString)));
    r->apiRequestPost(LASTFM_HANDSHAKE_URL, post_params);
}

void ApiClient::scrobblerHandshakeReply(QString data)
{
    QMap<QString, QString> reply;
    QTextStream stream(&data);

    QString reply_status = "";
    reply_status = stream.readLine();

    if(reply_status == "OK")
    {
        reply["status"] = reply_status;
        reply["session_id"] = stream.readLine();
        reply["playing_url"] = stream.readLine();
        reply["submission_url"] = stream.readLine();
    }
    else
        reply["status"] = reply_status;

    emit scrobblerHandshakeData(reply);
}

void ApiClient::scrobblerPlaying(QString playingUrl, QString sessionId, QString artist, QString track, QString album,
                                 QString trackLenght, QString trackNumber, QString mbTrackId)
{
    Utils *u = new Utils();

    int lenght = trackLenght.toInt();
    lenght = lenght / 1000;

    QMap<QString, QString> params;
    params["s"] = sessionId;
    params["a"] = artist;
    params["t"] = track;
    params["b"] = album;
    params["l"] = QString::number(lenght);;
    params["n"] = trackNumber;
    params["m"] = mbTrackId;

    QByteArray post_params = u->buildPostParameters(params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(scrobblerPlayingReply(QString)));
    r->apiRequestPost(playingUrl, post_params);
}

void ApiClient::scrobblerPlayingReply(QString data)
{
    emit scrobblerPlayingData(data);
}

void ApiClient::scrobblerSubmission(QString scrobblingUrl, QString sessionId, QString artist,
                                    QString track, QString timestamp, QString source, QString rating,
                                    QString trackLenght, QString album, QString trackNumber, QString mbTrackId)
{
    Utils *u = new Utils();

    int lenght = trackLenght.toInt();
    lenght = lenght / 1000;

    QMap<QString, QString> params;
    params["s"] = sessionId;
    params["a[0]"] = artist;
    params["t[0]"] = track;
    params["i[0]"] = timestamp;
    params["o[0]"] = source;
    params["r[0]"] = rating;
    params["l[0]"] = QString::number(lenght);
    params["b[0]"] = album;
    params["n[0]"] = trackNumber;
    params["m[0]"] = mbTrackId;

    QByteArray post_params = u->buildPostParameters(params);

    Requests *r = new Requests(this);
    connect(r, SIGNAL(dataAvailable(QString)), this, SLOT(scrobblerSubmissionReply(QString)));
    r->apiRequestPost(scrobblingUrl, post_params);
}

void ApiClient::scrobblerSubmissionReply(QString data)
{
    emit scrobblerSubmissionData(data);
}

void ApiClient::setApiKey(QString apikey)
{
    api_key = apikey;
}

void ApiClient::setApiSecret(QString apisecret)
{
    api_secret = apisecret;
}
