/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>
#include <QtGui/QApplication>
#include <QtGui/QDesktopWidget>
#include <QtCore/QTimer>
#include <QtCore/QDebug>
#include <QMaemo5InformationBox>
#include "mainwindow.h"
#include "mainwindowstack.h"
#include "themeloader.h"
#include "mainmenu.h"
#include "odometer.h"
#include "widgetscreen.h"
#include "poialerts.h"
#include "speedalarm.h"
#include "mediaplayer.h"
#include "orientation.h"
#include "settings.h"
#include <QOrientationSensor>
#include <QSensor>


MainWindow::MainWindow(): QMainWindow(0), menu_(0), themeLoader_(0),
mainScreen_(0), orientation_(0)
{
    setWindowTitle(tr("jSpeed"));
    showFullScreen();
    orientation_ = new Orientation(this);
    orientation_->start();
    addScreens();
    QTimer::singleShot(500, this, SLOT(loadServices()));
}

MainWindow::~MainWindow()
{
    delete themeLoader_;
}

void MainWindow::loadServices()
{
    Odometer::instance().start();
    QApplication::processEvents();
    startBacklight();
    QApplication::processEvents();
    PoiAlerts::instance().start();
    QApplication::processEvents();
    SpeedAlarm::instance().start();
    QApplication::processEvents();
    MediaPlayer::init();
}

void MainWindow::addScreens()
{
    stack_ = new MainWindowStack(this);

    connect(stack_, SIGNAL(minimizePressed()), this, SLOT(minimize()));
    connect(stack_, SIGNAL(settingsPressed()), this, SLOT(openMenu()));
    connect(stack_, SIGNAL(closePressed()), this, SIGNAL(quit()));

    mainScreen_ = new WidgetScreen(this);
    WidgetScreen* detailScreen = new WidgetScreen(this);

    themeLoader_ = new ThemeLoader(mainScreen_, detailScreen);

    if(!loadTheme())
    {
        return;
    }

    stack_->addScreen(mainScreen_);
    stack_->addScreen(detailScreen);

    connect(QApplication::desktop(), SIGNAL(resized(int)), stack_, SLOT(reArrange()));

    setCentralWidget(stack_);
}

bool MainWindow::loadTheme()
{
    if(!themeLoader_->load())
    {
        QMaemo5InformationBox::information(this, tr("Unable to load theme: %1").arg(themeLoader_->error()));
        close();
        return false;
    }

    int orientations = 0;

    if(mainScreen_->orientationEnabled(WidgetScreen::LANDSCAPE))
    {
        orientations |= Orientation::LANDSCAPE;
    }
    if(mainScreen_->orientationEnabled(WidgetScreen::PORTRAIT))
    {
        orientations |= Orientation::PORTRAIT;
    }

    orientation_->setSupportedOrientations(orientations);
    onOrientationChanged();
    orientation_->update();

    return true;
}

void MainWindow::minimize()
{
    QDBusConnection connection = QDBusConnection::sessionBus();
    QDBusMessage msg = QDBusMessage::createSignal("/",
                                                  "com.nokia.hildon_desktop",
                                                  "exit_app_view");
    connection.send(msg);
}

void MainWindow::openMenu()
{
    if(!menu_)
    {
        menu_ = new MainMenu(this);
        connect(menu_, SIGNAL(resetTrip()), &(Odometer::instance()), SLOT(resetTrip()));
        connect(menu_, SIGNAL(resetAll()), &(Odometer::instance()), SLOT(resetAll()));
        connect(menu_, SIGNAL(flip()), stack_, SLOT(flip()));
        connect(menu_, SIGNAL(themeChanged()), this, SLOT(loadTheme()));
        connect(menu_, SIGNAL(unitChanged()), &(Odometer::instance()), SLOT(updateUnit()));
        connect(menu_, SIGNAL(orientationChanged()), this, SLOT(onOrientationChanged()));
    }

    menu_->show();
}

void MainWindow::startBacklight()
{
    QTimer* timer = new QTimer(this);
    timer->setInterval(25000);
    connect(timer, SIGNAL(timeout()), this, SLOT(keepBacklightOn()));
    timer->start();
}

void MainWindow::keepBacklightOn()
{
    QDBusConnection connection = QDBusConnection::systemBus();
    QDBusMessage msg = QDBusMessage::createMethodCall("com.nokia.mce",
                                                      "/com/nokia/mce/request",
                                                      "com.nokia.mce.request",
                                                      "req_display_blanking_pause");

    connection.call(msg);
}

void MainWindow::onOrientationChanged()
{
    QString value = Settings::instance().value("orientation").toString();

    if(value == "auto")
    {
        orientation_->setOrientationType(Orientation::TYPE_AUTO);
    }
    else if(value == "landscape")
    {
        orientation_->setOrientationType(Orientation::TYPE_LANDSCAPE);
    }
    else if(value == "portrait")
    {
        orientation_->setOrientationType(Orientation::TYPE_PORTRAIT);
    }
}
