/**midi_editor_notes.c is part of JamMo.
License: GPLv2, read more from COPYING

This file is for clutter based gui.
This source files consists of note structures and functions to modify notes.

These structures and functions should make it easier to control notes, their properties
and visual representations.
*/

#include "midi_editor_notes.h"

//TODO: Implement a function for writing the note collection contents to a file
//TODO: Implement a function for reading notes from a file. NOTE: This is a bit tricky because
//		only the editor view knows the correct placement based on the times.
//TODO: At the moment any note with no ClutterActor will be added to collection even if they
//		overlap. Maybe discourage using new_note and start using new_note_with_actor (in this case
//		rename the functions, new_note -> new_note_without_actor, new_note_with_actor->new_note)
//TODO: Test set_note_time() thoroughly!
//TODO: Implement a function for removing notes given their times.

//Returns a pointer to a new note collection
//NOTE: Always free collection memory with free_note_collection() or it leaks memory
NoteArray* new_note_collection() {
	return (NoteArray*)g_ptr_array_new();
}

//Adds note to the end of the note collection
void add_note_to_collection(NoteArray *array, Note *note) {
	if(!is_note_overlapping(array, note)) {
		g_ptr_array_add(array, (gpointer)note);
	}

	//The note is overlapping, is not added and is destroyed
	else {
		printf("***DEBUG NOTE***: Tried to add an overlapping note!\n");
		//NOTE: Should memory be released here or where?
		if(note->note_actor != 0) {
			clutter_actor_destroy(note->note_actor);
		}
		free(note);
		note = NULL;
	}
}

//Frees the memory allocated for the note collection
//NOTE: This must be called in the future when closing the midi editor to avoid leaking memory
//TODO: Test if this releases the element data memory as well
void free_note_collection(NoteArray *array) {
	printf("**DEBUG NOTE**: Note array free'd\n");
	int note_index;
	for(note_index = 0; note_index < get_note_collection_size(array); note_index++) {
		//ClutterActors are freed separately
		if(get_note_actor(array, note_index) != NULL) {
			clutter_actor_destroy(get_note_actor(array, note_index));

			//For debugging only
			//printf("ClutterActor destroyed for note: ");
			//note_debug_print(get_note(array, note_index));
		}
	}

	g_ptr_array_free(array, TRUE);
}

//Returns note collection size
guint get_note_collection_size(NoteArray *array) {
	return array->len;
}

//Checks whether a given note is overlapping an already existing note
gboolean is_note_overlapping(NoteArray *array, Note *note) {
	gboolean overlapping = FALSE;

	guint note_index = 0;
	//Check the whole collection for possible note overlaps
	for(note_index = 0; note_index < get_note_collection_size(array); note_index++) {

		Note *existing_note = get_note(array, note_index);

		//printf("Note 1: %p, Note 2: %p\n", note->note_actor, existing_note->note_actor);

		//Don't check against self
		if((note->note_actor != existing_note->note_actor)) {

			//Uncomment for debugging
			//printf("Checking overlaps between note 1 and note 2:\nNote 1:");
			//note_debug_print(note);
			//printf("Note 2:");
			//note_debug_print(existing_note);
			//printf("\n");

			//If the note starts during an existing note
			if(note->start_time > existing_note->start_time &&
			   note->start_time < existing_note->end_time) {
				overlapping = TRUE;
			}
			//If the note ends during an existing note
			if(note->end_time > existing_note->start_time &&
			   note->end_time < existing_note->end_time) {
				overlapping = TRUE;
			}
			//If the note starts before and ends after
			if(note->start_time < existing_note->start_time &&
			   note->end_time > existing_note->end_time) {
				overlapping = TRUE;
			}
			//If the note is exactly the same
			else if(note->start_time == existing_note->start_time &&
					note->end_time == existing_note->end_time) {
				overlapping = TRUE;
			}
		}
	}

	return overlapping;
}

//Used internally as a comparison function to sort notes according to their start times
static gint sort_notes_by_time(gconstpointer a, gconstpointer b) {
	gint return_value = 0;

	const Note *first = *((const Note **)a);
	const Note *second = *((const Note **)b);

	//For debugging
	//printf("First is:  %f Second is: %f\n", first->start_time, second->start_time);

	//If the first note comes earlier
	if(first->start_time < second->start_time) {
		return_value = -1;
	}
	//If the second note comes earlier
	else if(first->start_time > second->start_time) {
		return_value = 1;
	}

	return return_value;
}

//Sorts the note collection according to start times
void sort_note_collection(NoteArray *array) {
	g_ptr_array_sort(array, (GCompareFunc)sort_notes_by_time);
}

//Returns pointer to a new note
Note* new_note(char note, int octave, double start_time, double end_time) {
	Note *new_note = (Note*)malloc(sizeof(Note));
	new_note->note = note;
	new_note->octave = octave;
	new_note->start_time = start_time;
	new_note->end_time = end_time;
	new_note->note_actor = NULL;

	return new_note;
}

//Returns pointer to a new note with a ClutterActor
Note* new_note_with_actor(char note, int octave, double start_time, double end_time, ClutterActor *actor) {
	Note *new_note = (Note*)malloc(sizeof(Note));
	new_note->note = note;
	new_note->octave = octave;
	new_note->start_time = start_time;
	new_note->end_time = end_time;
	new_note->note_actor = actor;

	return new_note;
}

//Get pointer to the note specified by the array index
Note* get_note(NoteArray *array, int note_index) {
	Note *note;

	if(note_index < get_note_collection_size(array)) {
		note = g_ptr_array_index(array, note_index);
	}

	//The index is greater than the number of array elements
	else {
		note = NULL;
		printf("**DEBUG NOTE ERROR**: get_note() out of bounds!\n");
	}

	return note;
}

//Returns the note that points to the passed ClutterActor
Note* get_note_from_actor(NoteArray *array, ClutterActor *actor) {

	Note* note = NULL;

	int note_index=0;
	for(note_index=0; note_index<get_note_collection_size(array); note_index++) {
		if(get_note_actor(array, note_index) == actor) {
			note = get_note(array, note_index);
		}
	}

	if(!note) {
		printf("***DEBUG ERROR***: No matching note found with get_note_from_actor()!");
	}

	return note;
}

//Get pointer to ClutterActor for the note specified by the array index
ClutterActor* get_note_actor(NoteArray *array, int note_index) {
	Note *note = get_note(array, note_index);
	return note->note_actor;
}

//Sets a new note time to the note that is represented by the passed ClutterActor *
//NOTE: The editor view _should_ calculate new start and end times if the ClutterActor
//		is moved and pass them here, so for now, this is probably the best solution.
//NOTE: The passed ClutterActor pointer is only used to find the matching note from the collection.
gboolean set_note_time(NoteArray *array, double start_time, double end_time, ClutterActor *actor) {

	Note *note = NULL;
	gboolean succesfully_set = FALSE;

	int note_index = 0;
	for(note_index = 0; note_index < get_note_collection_size(array); note_index++ ) {
		//Found a note that points to the same ClutterActor (so it's the same note)
		if(actor == get_note_actor(array, note_index)) {
			note = get_note(array, note_index);
		}
	}

	if(note) {
		//Create a dummy note for overlap tests
		Note compare_note = *note;
		compare_note.start_time = start_time;
		compare_note.end_time = end_time;
		gboolean overlaps = FALSE;

		overlaps = is_note_overlapping(array, &compare_note);
		if(overlaps) {
			printf("Unable to set note! Note overlaps with old one!\n");
		}
		//The note won't overlap with anything so it can be set
		if(!overlaps) {
			note->start_time = start_time;
			note->end_time = end_time;
			succesfully_set = TRUE;
		}
	}
	return succesfully_set;
}

//TODO: Test this
//Removes a note pointing to the same ClutterActor as the 2nd parameter
gboolean remove_note_from_collection(NoteArray *array, ClutterActor *actor) {

	gboolean removed = FALSE;
	int note_index;

	//Go through the collection looking for a matching ClutterActor in any note
	for(note_index = 0; note_index < get_note_collection_size(array); note_index++ ) {
		//If note points to the given ClutterActor
		if(actor == get_note_actor(array, note_index)) {

			//Uncomment for debugging
			//printf("Parameter actor address: %p\n", actor);
			//printf("Collection actor address: %p\n", (get_note(array, note_index))->note_actor);

			//TODO:Not sure if g_ptr_array_remove automatically frees memory
			clutter_actor_destroy(get_note_actor(array, note_index));
			free(get_note(array, note_index));
			g_ptr_array_remove_index(array, note_index);
			removed = TRUE;
		}
	}

	return removed;
}

//Prints a debug message for a single note
void note_debug_print(Note *note) {
	printf("**DEBUG NOTE**: Note %c, Octave: %i, Start: %f, End %f\n",
		   note->note, note->octave, note->start_time, note->end_time);
}

//Prints a debug message for the entire note collection
void note_collection_debug_print(NoteArray *array) {
	Note *current_note;
	int i;
	//Print debug prints for each note in collection
	for(i = 0; i < get_note_collection_size(array); i++) {
		current_note = g_ptr_array_index(array, i);
		note_debug_print(current_note);
	}
	printf("-----------------------\n");

}

//Saves the notes to a file
//NOTE: Very preliminary, most likely not the final note format
void save_note_collection_to_file(NoteArray *array, gchar *filename) {
	FILE *note_file;
	note_file = fopen(filename, "w");

	int note_index = 0;
	for(note_index = 0; note_index < get_note_collection_size(array); note_index++) {
		Note *note = get_note(array, note_index);
		fprintf(note_file, "Note:%c %i\nOn:%lf\nOff:%lf\n", note->note, note->octave, note->start_time, note->end_time);
	}

	fclose(note_file);
}

//Loads notes from a file
//TODO: Implement this, dummy function for now
//NOTE: Very preliminary, most likely not the final note format
void load_note_collection_from_file(gchar *filename) {

}

//This function is only for tests related to the note collection
void midi_note_test() {
	NoteArray *note_array = new_note_collection();
	/*
	add_note_to_collection(note_array, new_note('C', 5, 21.543935, 28.342043));
	add_note_to_collection(note_array, new_note('F', 5, 11.224324, 20.421312));
	add_note_to_collection(note_array, new_note('A', 3, 57.000055, 68.493389));
	add_note_to_collection(note_array, new_note('G', 3, 31.234233, 56.050505));
	add_note_to_collection(note_array, new_note('F', 2, 10.40434, 11.00044));
	add_note_to_collection(note_array, new_note('E', 2, 15.00000, 21.0000)); //This should overlap
	*/

	ClutterActor *test1 = clutter_rectangle_new();
	add_note_to_collection(note_array, new_note_with_actor('A', 2, 0.25, 0.5, test1));

	sort_note_collection(note_array);
	note_collection_debug_print(note_array);
	free_note_collection(note_array);

}
