#!/usr/bin/env python
#
# N900 internet radio installer
# Reads eradio.list and updates the media player
#
# The stations are saved into .mafw.db sqlite3 database 
# in /home/user/ directory.
#
# The database has only one table 'iradiobookmarks' with 3 fields 'id', 'key', 'value'. The 
# 'id' is not unique and the key is one of 'title', 'uri', 'mime-type'. The value is a data 
# blob which contains the data which describe the key. One radio has at least 3 entries in 
# the database with the same id but with different key types. So at least 'title', 'uri' and # 'mime-type' should be present.
#
# Actually is less complicated than it sounds. I coded last night a proof of concept python 
# script to edit the database to add greek stations in a automated manner. I will clean up 
# the code and release it as a greek radios package.

import sqlite3
from eradio import getPlaylist
import hildon
from subprocess import Popen
import gtk
import sys
from generator import GeneratorTask


DATABASE = "/home/user/.mafw.db"
ERADIOLIST = "eradio.list"

class GreekInternetRadio(hildon.Program):
    def __init__(self):
        hildon.Program.__init__(self)
        gtk.set_application_name("Greek Internet Radio")
        self.window = hildon.Window()  
        self.window.connect("delete_event", self.quit)  
        self.add_window(self.window)
        self.playlist = {}

        self.createButtons()

        menu = self.create_menu(None)
        self.window.set_app_menu(menu)
        
        self.drawUI()
        
    def createButtons(self):
        self.img = gtk.Image()
        self.img.set_from_file("/opt/greekiradio/img/greekiradio.png")
        self.statusLabel = gtk.Label()
        self.statusLabel.set_text("Ready")
        self.updateButton = hildon.GtkButton(gtk.HILDON_SIZE_THUMB_HEIGHT)
        self.updateButton.set_label("Update Internet Radio List")
        self.updateButton.connect("clicked", self.updateList, None) 
        
    def drawUI(self):
        vbox = gtk.VBox(False, 10)
        vbox.pack_start(self.img, expand=True)
        vbox.pack_start(self.statusLabel, expand=False)
        vbox.pack_start(self.updateButton, expand=False)
        self.window.add(vbox)
        self.window.show_all()

    def display_status(self, msg):
        self.statusLabel.set_text(msg)

    def quit(self, *args):
        gtk.main_quit()
        
    def run(self):     
        self.window.show_all()
        gtk.main()        
        
    def updateList(self, *args):
        # fetch the urls 
        try:
            self.updateButton.set_sensitive(False)
            self.display_status("Fetching internet radio urls")
            GeneratorTask(getPlaylist, complete_callback=self.updateDB).start(playlist=self.playlist, callback=self.display_status)
        except IOError:
            self.display_status("Connection error! Are you connected to the internet?")
            return

    def updateDB(self):
        # connect to the database
        conn = sqlite3.connect(DATABASE)
        cursor = conn.cursor()
        
        # update the database 
        self.display_status("Updating the database")
       
        # stop media play iradio manager
        Popen(["/usr/bin/mafw.sh", "stop", "mafw-iradio-source"])
        self.insertIntoDB(cursor, self.playlist)
       
        # close the database
        conn.commit()
        conn.close()
        
        # update media play list
        self.display_status("Restarting media player")
        Popen(["/usr/bin/mafw.sh", "start", "mafw-iradio-source"])
        
        self.display_status("Done ;)")
        self.updateButton.set_sensitive(True)
        
    # Find the largest ID in the database.
    def findLargestId(self, cursor):
        cursor.execute("SELECT MAX(id) FROM iradiobookmarks")
        row = cursor.fetchone()
        if row:
            return row[0]
        else:
            return -1

    # Convert 'string' to a blob readable from media player
    def buildBlob(self, string):
        return sqlite3.Binary(chr(1) + chr(0) * 3 + '@' + chr(0) * 3 + string + chr(0) )

    # Check if station in already listed and return it's id.
    # Used to do updates        
    def findStationId(self, cursor, station):
        cursor.execute('SELECT id FROM iradiobookmarks WHERE key="title" and value=?', (self.buildBlob(station),))
        row = cursor.fetchone()
        if row:
            return row[0]
        else:
            return -1

    def insertIntoDB(self, cursor, radioList):
        i = 0
        nextID = self.findLargestId(cursor)+1
        for radio in radioList.keys():
            uri = radioList[radio]
            title = radio

            stationId = self.findStationId(cursor, title)
            if stationId == -1:
                # station is not listed. Add it to the list with the next ID
                stationId = nextID
                nextID += 1
                cursor.execute('INSERT INTO iradiobookmarks VALUES (?, "title", ?)', (stationId, self.buildBlob(title),))
                cursor.execute('INSERT INTO iradiobookmarks VALUES (?, "mime-type", ?)', (stationId, self.buildBlob("audio"),))
                cursor.execute('INSERT INTO iradiobookmarks VALUES (?, "uri", ?)', (stationId, self.buildBlob(uri),))
            else:
                # station is listed. Update the uri entry
                cursor.execute('UPDATE iradiobookmarks SET value = ? WHERE id = ? AND key = "uri" ', (self.buildBlob(uri), stationId))
            
            i += 2

    def create_menu(self,label):
        menu = hildon.AppMenu()
        
        # about button
        button = hildon.GtkButton(gtk.HILDON_SIZE_AUTO)
        button.set_label("About")
        button.connect("clicked", self.on_about_button_clicked)
        menu.append(button)

        menu.show_all()
        
        return menu
            
    def on_about_button_clicked(self, widget):
        about = gtk.AboutDialog()
        about.set_name("Greek Internet Radio Playlist")
        about.set_version("0.1")
        about.set_copyright("Giorgos Logiotatidis\nseadog@sealabs.net")
        about.set_website("https://www.sealabs.net/seadog")
        about.set_logo_icon_name("greekiradio")
        about.connect("response", lambda x, y: x.destroy())
        about.show()
        
        
if __name__ == "__main__":
    app = GreekInternetRadio() 
    app.run()      
