/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 * 
 * This file is based upon parts of gpsd/libgps
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef GPS_H
#define GPS_H

#include "gpx.h"

#ifndef NAN
#define NAN (0.0/0.0)
#endif /* !NAN */

#define MAXCHANNELS 20
#define MAXTAGLEN    8       /* maximum length of sentence tag name */
#define MPS_TO_KNOTS 1.9438445       /* Meters per second to knots */

struct gps_fix_t {
    int    mode;	/* Mode of fix */
#define MODE_NOT_SEEN	0	/* mode update not seen yet */
#define MODE_NO_FIX	1	/* none */
#define MODE_2D  	2	/* good for latitude/longitude */
#define MODE_3D  	3	/* good for altitude/climb too */
    pos_t pos;      /* Latitude/Longitude in degrees (valid if mode >= 2) */
    float eph;      /* Horizontal position uncertainty, meters */
    float track;    /* Course made good (relative to true north) */
};

typedef unsigned int gps_mask_t;

struct gps_data_t {
    gps_mask_t set;	/* has field been set since this was last cleared? */
#define LATLON_SET	0x00000008u
#define SPEED_SET	0x00000020u
#define TRACK_SET	0x00000040u
#define STATUS_SET	0x00000100u
#define MODE_SET	0x00000200u
#define HERR_SET	0x00008000u
#define SATELLITE_SET	0x00040000u
    struct gps_fix_t	fix;		/* accumulated PVT data */

    /* GPS status -- always valid */
    int    status;		/* Do we have a fix? */
#define STATUS_NO_FIX	0	/* no */
#define STATUS_FIX	1	/* yes, without DGPS */
#define STATUS_DGPS_FIX	2	/* yes, with DGPS */

    /* precision of fix -- valid if satellites_used > 0 */
    int satellites_used;	/* Number of satellites used in solution */
    int used[MAXCHANNELS];	/* PRNs of satellites used in solution */

    /* satellite status -- valid when satellites > 0 */
    int satellites;		/* # of satellites in view */
    int PRN[MAXCHANNELS];	/* PRNs of satellite */
    int elevation[MAXCHANNELS];	/* elevation of satellite */
    int azimuth[MAXCHANNELS];	/* azimuth */
    int ss[MAXCHANNELS];	/* signal-to-noise ratio (dB) */
};

#ifdef USE_MAEMO
#ifdef ENABLE_GPSBT
#include <gpsbt.h>
#include <gpsmgr.h>
#endif
#include <errno.h>
#endif

#ifdef ENABLE_LIBLOCATION
#include <location/location-gps-device.h>
#include <location/location-gpsd-control.h>
#endif

typedef struct {
  int num;
  int *PRN;
  int *ss;
  int *used;
} gps_sat_t;

typedef void (*gps_cb)(struct gps_state *, gpointer data);
#define	GPS_CB(f) ((gps_cb)(f))

typedef struct {
  gps_cb cb;
  gpointer data;
} gps_cb_t;

typedef struct gps_state {
#ifndef ENABLE_LIBLOCATION
#ifdef ENABLE_GPSBT
  gpsbt_t context;
#endif

  GThread* thread_p;
  GMutex *mutex;
  GnomeVFSInetConnection *iconn;
  GnomeVFSSocket *socket;

  struct gps_data_t gpsdata;
#else
  LocationGPSDevice *device;
  LocationGPSDControl *control;
  gboolean in_use;
  guint idd_changed;
  int fields;
  
  struct gps_fix_t fix;
  gps_sat_t sats;
#endif

  GSList *cb;

} gps_state_t;

void gps_init(appdata_t *appdata);
void gps_release(appdata_t *appdata);
pos_t *gps_get_pos(appdata_t *appdata);
float gps_get_heading(appdata_t *appdata);
float gps_get_eph(appdata_t *appdata);
gps_sat_t *gps_get_sats(appdata_t *appdata);
void gps_change_state(appdata_t *appdata);

void *gps_register_callback(appdata_t *appdata, gps_cb cb, gpointer data);
void gps_unregister_callback(appdata_t *appdata, void *cb);

#endif // GPS_H
