/*
 * Copyright (C) 2009 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

/* 
 * This simple breadcrumbtrail replacement is a simple hbox with 
 * buttons inside 
 */

#include "gpxview.h"

typedef struct {
  GtkWidget *button;
  GDestroyNotify destroy;
  gpointer id;
} crumb_t;

typedef struct {
  GtkWidget *back;

  GSList *crumbs;
} bct_priv_t;

static void bct_free(gpointer data, gpointer user_data) {
  g_free(data);
}

static gint bct_destroy_event(GtkWidget *widget, gpointer data ) {
  bct_priv_t *priv = (bct_priv_t*)data;

  g_slist_foreach(priv->crumbs, bct_free, NULL);
  g_free(priv);

  return FALSE;
}

static void crumb_destroy(bct_priv_t *priv, crumb_t *crumb) {
  /* call destroy callback */
  crumb->destroy(crumb->id);
  
  /* destroy it */
  gtk_widget_destroy(crumb->button);

  /* remove element from list */
  priv->crumbs = g_slist_remove(priv->crumbs, crumb);

  g_free(crumb);
}

static void bct_update(bct_priv_t *priv) {
  /* still something in list? */
  if(g_slist_length(priv->crumbs)) {
    crumb_t *crumb = g_slist_last(priv->crumbs)->data;
    /* disable button now last in chain */
    gtk_widget_set_sensitive(crumb->button, FALSE);
  }

  /* disable back button if crumb trail has 1 item or less */
  if(g_slist_length(priv->crumbs) <= 1)
    gtk_widget_set_sensitive(priv->back, FALSE);
}

static void on_back_clicked(GtkButton *button, bct_priv_t *priv) {
  crumb_destroy(priv, g_slist_last(priv->crumbs)->data);
  bct_update(priv);
}

static void on_crumb_clicked(GtkButton *button, bct_priv_t *priv) {
  /* clear crumb trail until we reach the button that was clicked */
  crumb_t *crumb = g_slist_last(priv->crumbs)->data;
  while(crumb->button != (GtkWidget*)button) {
    crumb_destroy(priv, crumb);
    crumb = g_slist_last(priv->crumbs)->data;
  }
  bct_update(priv);
}

GtkWidget *bct_new(void) {
  bct_priv_t *priv = g_new0(bct_priv_t, 1);

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);
  g_object_set_data(G_OBJECT(hbox), "priv", priv);

  priv->back = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(priv->back), 
	gtk_image_new_from_stock(GTK_STOCK_GO_BACK, GTK_ICON_SIZE_BUTTON));
  gtk_widget_set_sensitive(priv->back, FALSE);
  g_signal_connect(priv->back, "clicked", 
		   G_CALLBACK(on_back_clicked), priv); 

  gtk_box_pack_start(GTK_BOX(hbox), priv->back, FALSE, FALSE, 0);

  g_signal_connect(G_OBJECT(hbox), "destroy", 
		   G_CALLBACK(bct_destroy_event), priv);

  return hbox;
}

void bct_push_text(GtkWidget *bct, const gchar *text, gpointer id,
		   GDestroyNotify destroy) {
  bct_priv_t *priv = g_object_get_data(G_OBJECT(bct), "priv");
  g_return_if_fail (priv);

  /* enable last crumb */
  if(g_slist_length(priv->crumbs)) {
    crumb_t *crumb = g_slist_last(priv->crumbs)->data;
    gtk_widget_set_sensitive(crumb->button, TRUE);
    gtk_widget_set_sensitive(priv->back, TRUE);
  }

  /* create a new crumb */
  crumb_t *crumb = g_new0(crumb_t, 1);

  /* append a new crumb */
  crumb->button = gtk_button_new_with_label(text);
  crumb->destroy = destroy;
  crumb->id = id;

  gtk_box_pack_start(GTK_BOX(bct), crumb->button, FALSE, FALSE, 0);
  gtk_widget_set_sensitive(crumb->button, FALSE);
  gtk_widget_show(crumb->button);

  g_signal_connect(crumb->button, "clicked", 
  		   G_CALLBACK(on_crumb_clicked), priv);

  /* append to list of crumbs */
  priv->crumbs = g_slist_append(priv->crumbs, crumb);

}

void bct_pop(GtkWidget *bct) {
  bct_priv_t *priv = g_object_get_data(G_OBJECT(bct), "priv");
  g_assert(priv);
  crumb_destroy(priv, g_slist_last(priv->crumbs)->data);
}
