/*
 * Copyright 2010 Felipe Crochik <foss@crochik.com>
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "geocoderequester.h"

#include <QUrl>
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QDebug>
#include <QMessageBox>

#include <QDomDocument>

GeocodeRequester::GeocodeRequester(QNetworkAccessManager *pNetAccessMngr, QObject *parent) :
    QObject(parent)
{
    m_pNetAccessMngr = pNetAccessMngr;
    m_pReply = NULL;
}

GeocodeRequester::~GeocodeRequester() {
    if ( m_pReply ) { delete m_pReply; m_pReply = NULL; }
}

bool GeocodeRequester::query(QString address) {
    QUrl url("http://maps.google.com/maps/api/geocode/xml");
    url.addQueryItem("sensor", "false");
    url.addQueryItem("address", address);

    qDebug() << QString(url.toEncoded());

    if ( m_pReply ) { delete m_pReply; m_pReply = NULL; }

    m_address.clear();
    m_pReply = m_pNetAccessMngr->get(QNetworkRequest(url));
    connect(m_pReply, SIGNAL(finished()), this, SLOT(onFinished()));

    qDebug() << "Requested";
    return true;
}

void GeocodeRequester::onFinished() {
    qDebug() << "reply finished";
    QDomDocument doc;
    QString error;
    int line, col;
    if ( !doc.setContent(m_pReply, false, &error, &line, &col) ) {
        qCritical() << "DOM error: " << error << line << col;
        return;
    }
    qDebug() << "parsed xml?";
    qDebug() << doc.toString();

    QDomNodeList list = doc.elementsByTagName("status");
    if ( list.length()!=1 ) {
        qCritical() << "Error";
        return;
    }

    QDomNode node = list.at(0);
    QString status = node.firstChild().nodeValue();
    qDebug()<< "-- status: " << status;

    if ( status == "ZERO_RESULTS")  {
        emit finish(false, "No results found.");
        return;
    }

    if ( status != "OK" ) {
        emit finish(false, "Service returned status: " + status);
        return;
    }

    /*
    * "OK" indicates that no errors occurred; the place was successfully detected and at least one result was returned.
    * "ZERO_RESULTS" indicates that the reference was valid but no longer refers to a valid result. This may occur if the establishment is no longerin business.
    * "OVER_QUERY_LIMIT" indicates that you are over your quota.
    * "REQUEST_DENIED" indicates that your request was denied, generally because of lack of a sensor parameter.
    * "INVALID_REQUEST" generally indicates that the query (reference) is missing or expired.
    */

    list = doc.elementsByTagName("result");
    for ( uint c=0; c<list.length(); c++ ) {
        GeocodeAddress address;
        address.setDescription( list.at(c).firstChildElement("formatted_address").firstChild().nodeValue() );

        QDomNodeList types = list.at(c).toElement().elementsByTagName("type");
        for ( uint d=0; d<types.length(); d++ ) address.m_types.append(types.at(d).firstChild().nodeValue());

        /*
        location_type stores additional data about the specified location. The following values are currently supported:
        * "ROOFTOP" indicates that the returned result is a precise geocode for which we have location information accurate down to street address precision.
        * "RANGE_INTERPOLATED" indicates that the returned result reflects an approximation (usually on a road) interpolated between two precise points (such as intersections). Interpolated results are generally returned when rooftop geocodes are unavailable for a street address.
        * "GEOMETRIC_CENTER" indicates that the returned result is the geometric center of a result such as a polyline (for example, a street) or polygon (region).
        * "APPROXIMATE" indicates that the returned result is approximate.
         */

        node = list.at(c).firstChildElement("geometry");
        address.m_location_type = node.firstChildElement("location_type").firstChild().nodeValue();

        node = node.firstChildElement("location");

        address.setCoordinates(
            node.firstChildElement("lat").firstChild().nodeValue().toDouble(),
            node.firstChildElement("lng").firstChild().nodeValue().toDouble()
        );

        qDebug() << address.description() << address.latitude() << address.longitude() << address.m_location_type << address.m_types;
        m_address.append(address);
    }

    emit finish(true, QString::null);
}
