/****************************************************************************
**
** Copyright (C) 2010  Aimone Davide realsportcars@hotmail.com
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "keyboard.h"
#include <QSet>
#include <QPushButton>
#include <QDebug>
#include <QSettings>

const int Keyboard::MODIFIERS_NONE = 0;
const int Keyboard::MODIFIERS_SHIFT = 1;
const int Keyboard::MODIFIERS_SYM = 3;
const int Keyboard::MODIFIERS_ACCENT = 5;

const QString Keyboard::MOVERIGHT = "MR";
const QString Keyboard::MOVELEFT = "ML";
const QString Keyboard::NEXTWORD = "NW";
const QString Keyboard::PREVWORD = "PW";
const QString Keyboard::DELETE = "\b";
const QString Keyboard::NEWLINE = "\n";
const QString Keyboard::SPACE = " ";

const QString Keyboard::CAPITALIZATION_CHAR = ".!?";

Keyboard::Keyboard(QWidget *parent) :
    QWidget(parent)
{
    m_bgCharacters = new QButtonGroup(this);
    m_nActiveModifiers = 0;
    m_btAccent = 0;
    m_btSymbol = 0;
    m_sLastInputChar = ".";
    m_nActiveStyle = SettingsMenu::AutoCapitalize;
}

void Keyboard::settingsChanged(int nStyle)
{
    if(isVisible())
    {
        emit inputStyleChanged(nStyle);
        if((SettingsMenu::inputStyle)nStyle == SettingsMenu::AutoCapitalize
                ||(SettingsMenu::inputStyle)nStyle == SettingsMenu::LowerCase)
        {
            m_nActiveStyle = (SettingsMenu::inputStyle)nStyle;
        }
    }
}

void Keyboard::updateSettingsMenu(int nActiveKeyboard)
{
    m_settingsMenu->updateMenu(nActiveKeyboard);
}

void Keyboard::prepareForFirstCharacter()
{
    m_nActiveModifiers = 0;

    if(m_nActiveStyle == SettingsMenu::AutoCapitalize)
    {
        m_btShift->setState(ThreeStateButton::singleCheck);
        changeInputModifiers(true,MODIFIERS_SHIFT);
    }
    else
    {
        m_btShift->setState(ThreeStateButton::noCheck);
    }
    if(m_btSymbol)
        m_btSymbol->setState(ThreeStateButton::noCheck);
    if(m_btAccent)
        m_btAccent->setState(ThreeStateButton::noCheck);
}

void Keyboard::charButtonClicked(QString sChar, Core::Keyboards kb)
{
    if(sChar.compare(" ") != 0)
        m_sLastInputChar = sChar;

    bool bUppercase = m_btShift->isChecked();

    if(m_btShift->getState()==ThreeStateButton::singleCheck)
    {
        m_btShift->setState(ThreeStateButton::noCheck);
        changeInputModifiers(false,MODIFIERS_SHIFT);
    }
    if(m_btAccent)
    {
        if(m_btAccent->getState()==ThreeStateButton::singleCheck)
        {
            m_btAccent->setState(ThreeStateButton::noCheck);
            changeInputModifiers(false,MODIFIERS_ACCENT);
        }
    }
    if(m_nActiveStyle == SettingsMenu::AutoCapitalize)
    {
        if(CAPITALIZATION_CHAR.contains(sChar) && (m_btShift->getState() != ThreeStateButton::doubleCheck) && m_btSymbol->getState() == ThreeStateButton::noCheck)
        {
            m_btShift->setState(ThreeStateButton::singleCheck);
            changeInputModifiers(true,MODIFIERS_SHIFT);
        }
    }


    emit input(sChar,kb,bUppercase);
}

void Keyboard::maiuscClicked()
{
    if(m_btSymbol)
    {
        if(m_btSymbol->getState()==ThreeStateButton::singleCheck)
        {
            m_btSymbol->setState(ThreeStateButton::noCheck);
            changeInputModifiers(false,MODIFIERS_SYM);
        }
        else if(m_btSymbol->getState()==ThreeStateButton::doubleCheck)
        {
            m_btSymbol->setState(ThreeStateButton::noCheck);
            changeInputModifiers(false,MODIFIERS_SYM + MODIFIERS_SHIFT);
        }
    }

    ThreeStateButton *clickedButton = qobject_cast<ThreeStateButton *>(sender());

    int nState = clickedButton->getState();

    if(nState == ThreeStateButton::noCheck)
        changeInputModifiers(false,MODIFIERS_SHIFT);
    else if(nState == ThreeStateButton::singleCheck)
        changeInputModifiers(true,MODIFIERS_SHIFT);
    else if(nState == ThreeStateButton::doubleCheck)
    {
        //do nothing
    }

}

void Keyboard::changeInputModifiers(bool bActivated,int nModifiers)
{
    if(bActivated)
        m_nActiveModifiers = m_nActiveModifiers + nModifiers;
    else
        m_nActiveModifiers = m_nActiveModifiers - nModifiers;

    setButtonsLabel();

}

void Keyboard::setButtonsLabel()
{
    if(m_nkbType == Core::Alfanumeric)
    {
        foreach(QAbstractButton *button, m_bgCharacters->buttons())
        {
            if (m_nActiveModifiers == MODIFIERS_NONE)
            {
                button->setText(button->text().toLower());
            }
            else if(m_nActiveModifiers == MODIFIERS_SHIFT)
            {
                //qDebug() << sCharset.at(1);
                button->setText(button->text().toUpper());
            }
        }
    }
    else
    {
        foreach(QAbstractButton *button, m_bgCharacters->buttons())
        {
            QString sCharset = button->property("charset").toString();

            if (m_nActiveModifiers == MODIFIERS_NONE)
            {
                //qDebug() << sCharset.at(0);
                button->setText(sCharset.at(0));
            }
            else if(m_nActiveModifiers == MODIFIERS_SHIFT)
            {
                //qDebug() << sCharset.at(1);
                button->setText(sCharset.at(1));
            }
            else if(m_nActiveModifiers == MODIFIERS_SYM)
            {
                //qDebug() << sCharset.at(2);
                button->setText(sCharset.at(2));
            }
            else if (m_nActiveModifiers == MODIFIERS_SHIFT + MODIFIERS_SYM)
            {
                //qDebug() << sCharset.at(3);
                button->setText(sCharset.at(3));
            }
            else if(m_nActiveModifiers == MODIFIERS_ACCENT)
            {
                //qDebug() << sCharset.at(4);
                button->setText(sCharset.at(4));
            }
            else if(m_nActiveModifiers == MODIFIERS_SHIFT + MODIFIERS_ACCENT)
            {
                //qDebug() << sCharset.at(4);
                if(m_activeLayout == Core::latinLayout)
                    button->setText(sCharset.at(4).toUpper());
                else
                    button->setText(sCharset.at(5));
            }
//            else if(m_nActiveModifiers == MODIFIERS_SHIFT + MODIFIERS_SYM + MODIFIERS_ACCENT)
//            {
//                //qDebug() << sCharset.at(4);
//                button->setText(sCharset.at(4).toUpper());
//            }
//            else if(m_nActiveModifiers == MODIFIERS_SYM + MODIFIERS_ACCENT)
//            {
//                //qDebug() << sCharset.at(4);
//                button->setText(sCharset.at(4));
//            }
        }
    }
}

void Keyboard::setLabelColor(Core::LabelColor nColorIdx)
{
    switch(nColorIdx)
    {
    case Core::themeDefault:
        setStyleSheet("color:");
        break;
    case Core::black:
        setStyleSheet("color:black");
        break;
    case Core::white:
        setStyleSheet("color:white");
        break;
    case Core::grey:
        setStyleSheet("color:grey");
        break;
    case Core::purple:
        setStyleSheet("color:purple");
        break;
    case Core::pink:
        setStyleSheet("color:pink");
        break;
    case Core::red:
        setStyleSheet("color:red");
        break;
    case Core::yellow:
        setStyleSheet("color:yellow");
        break;
    case Core::green:
        setStyleSheet("color:green");
        break;
    case Core::aqua:
        setStyleSheet("color:aqua");
        break;
    case Core::blue:
        setStyleSheet("color:blue");
        break;
    case Core::orange:
        setStyleSheet("color:orange");
        break;
    }
}
