/*
	This file is part of Faster Application Manager.

	Faster Application Manager is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Faster Application Manager is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Faster Application Manager.  If not, see <http://www.gnu.org/licenses/>.

	(C) Heikki Holstila 2010
*/

#ifndef PACKAGE_H
#define PACKAGE_H

#include <QtCore>
#include "blacklistselect.h"

class AAptInterface;
class Repository;

class Package
{
public:
	enum operation { PkgOpNone, PkgOpInstallUpgrade, PkgOpRemove };

	// the order must match status filter strings
	enum packageStatus { PkgStatUnknown, PkgStatNotInstalled, PkgStatUpgradeable, PkgStatInstalled };

	explicit Package(QByteArray name_, AAptInterface* apt_);
	virtual ~Package();

	inline void setName(QByteArray n_) { iName=n_; }
	inline void setMaemoDisplayName(QByteArray n_) { iMaemoDisplayName=n_; }
	inline void setInstalled(bool i_) { iIsInstalled=i_; }
	void setMarkedForOperation(operation op_);
	inline void setVersion(QByteArray v_) { iVersion=v_; }
	inline void setDescShort(QByteArray d_) { iDescriptionShort=d_; }
	inline void appendDescLong(QByteArray d_) { iDescriptionLong.append(d_); }
	inline void setSection(QByteArray s_) { iSection=s_; }
	inline void setSize(int i_) { iSize=i_; }
	inline void setInstalledSize(int i_) { iInstalledSize=i_; }
	inline void appendIconData(QByteArray d_) { iIconData.append(d_); }
	inline void addFullFileName(QString f_) { iFullFileNames.append(f_); }
	void updateStatus();
	inline void addRepository(Repository* r_) { iRepositories.append(r_); }
	inline void setDate(QDateTime d_) { iDate=d_; }
	inline void setBlacklisted(BlacklistSelect::blackList bl_) { iBlacklist=bl_; }
	inline void appendDepends(QByteArray line_) { iDepends << line_.split(','); }
	inline void appendConflicts(QByteArray line_) { iConflicts << line_.split(','); }
	inline void appendPreDepends(QByteArray line_) { iPreDepends << line_.split(','); }
	inline void appendProvides(QByteArray line_) { iProvides << line_.split(','); }
	inline void appendReplaces(QByteArray line_) { iReplaces << line_.split(','); }
	inline void appendBreaks(QByteArray line_) { iBreaks << line_.split(','); }
	inline void appendRecommends(QByteArray line_) { iRecommends << line_.split(','); }
	inline void appendSuggests(QByteArray line_) { iSuggests << line_.split(','); }
	inline void appendUpgradeDescription(QByteArray d_) { iUpgradeDescription.append(d_); }
	inline void setPinned(bool p_) { iPinned=p_; }
	inline void setMaintainer(QByteArray m_) { iMaintainer = m_; }

	void convertIcon();

	inline QString name() const { return iName; }
	inline QString maemoDisplayName() const { return QString::fromUtf8(iMaemoDisplayName); }
	QString displayName() const;
	inline bool isInstalled() const { return iIsInstalled; }
	inline bool isMarkedForOperation() const { return iMarkedForOperation; }
	inline operation markedOperation() const { return iMarkedOperation; }
	inline QString version() const { return iVersion; }
	inline QString descShort() const { return QString::fromUtf8(iDescriptionShort); }
	inline QString descLong() const { return QString::fromUtf8(iDescriptionLong); }
	inline QString section() const { return iSection; }
	inline int size() const { return iSize; }
	inline int installedSize() const { return iInstalledSize; }
	inline QPixmap* icon() const { return iIcon; }
	bool hasIconData() const;
	bool isUpgradeable() const;
	QString upgradeableVersion() const;
	Package* availablePackage() const;
	packageStatus status();
	QString fileName() const;
	inline QStringList fullFileNames() const { return iFullFileNames; }
	inline QDateTime date() const { return iDate; }
	inline QList<const Repository*> repositories() const { return iRepositories; }
	inline BlacklistSelect::blackList blacklisted() const { return iBlacklist; }
	inline bool isBlacklisted() const { if(iBlacklist==BlacklistSelect::BlacklistNone) return false; else return true; }
	inline QString upgradeDescription() const { return QString::fromUtf8(iUpgradeDescription); }
	inline QList<QByteArray> depends() const { return iDepends; }
	inline QList<QByteArray> conflicts() const { return iConflicts; }
	inline QList<QByteArray> preDepends() const { return iPreDepends; }
	inline QList<QByteArray> provides() const { return iProvides; }
	inline QList<QByteArray> replaces() const { return iReplaces; }
	inline QList<QByteArray> breaks() const { return iBreaks; }
	inline QList<QByteArray> recommends() const { return iRecommends; }
	inline QList<QByteArray> suggests() const { return iSuggests; }
	inline bool isPinned() const { return iPinned; }
	QString maintainerRichText() const;

	QStringList checkConflicts_RichText() const;

	static bool versionCompare(QString isNewer, QString compare);
	static QStringList toTrimmedRichTextList(QList<QByteArray> list_in);

private:
	Package(const Package& old);
	Package operator= (const Package& old);

	static bool versionConflicts(QString conflictVer, QString operVer);

	AAptInterface* iAptInterface;

	QByteArray iName;
	QByteArray iMaemoDisplayName;
	bool iIsInstalled;
	bool iMarkedForOperation;
	packageStatus iPkgStatus;
	QByteArray iVersion;
	QByteArray iDescriptionShort;
	QByteArray iDescriptionLong;
	QByteArray iSection;
	int iSize;
	int iInstalledSize;
	operation iMarkedOperation;
	QStringList iFullFileNames;
	QDateTime iDate;
	QList<const Repository*> iRepositories;
	BlacklistSelect::blackList iBlacklist;
	QByteArray iUpgradeDescription;
	QByteArray iMaintainer;

	QByteArray iIconData;
	QPixmap* iIcon;

	QList<QByteArray> iDepends;
	QList<QByteArray> iConflicts;
	QList<QByteArray> iPreDepends;
	QList<QByteArray> iProvides;
	QList<QByteArray> iReplaces;
	QList<QByteArray> iBreaks;
	QList<QByteArray> iRecommends;
	QList<QByteArray> iSuggests;

	bool iPinned;
};

#endif // PACKAGE_H
