#include "accountsdialog.h"
#include "accountdelegate.h"
#include "youtubeauthdialog.h"
#include "dailymotionauthdialog.h"
#include "vimeoauthdialog.h"
#include "facebookauthdialog.h"
#include "twitterauthdialog.h"
#include "videoserviceselector.h"
#include "../base/dailymotionaccountslistmodel.h"
#include "../base/youtubeaccountslistmodel.h"
#include "../base/vimeoaccountslistmodel.h"
#include "listview.h"
#include "../base/sharing.h"
#include "../base/youtube.h"
#include "../base/dailymotion.h"
#include "../base/vimeo.h"
#include "separatorlabel.h"
#include <QPushButton>
#include <QLineEdit>
#include <QGridLayout>
#include <QScrollArea>
#include <QMenu>
#include <QAction>

AccountsDialog::AccountsDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_nameEdit(new QLineEdit(this)),
    m_serviceSelector(new VideoServiceSelector(tr("Service"), this)),
    m_loginButton(new QPushButton(tr("Sign in"), this)),
    m_dailymotionModel(new DailymotionAccountsListModel(this)),
    m_youtubeModel(new YouTubeAccountsListModel(this)),
    m_vimeoModel(new VimeoAccountsListModel(this)),
    m_dailymotionView(new ListView(this)),
    m_youtubeView(new ListView(this)),
    m_vimeoView(new ListView(this)),
    m_facebookLabel(new QLabel(Sharing::instance()->userSignedInToFacebook() ? tr("You are signed in to a facebook account") : tr("Sign in to a facebook account"), this)),
    m_twitterLabel(new QLabel(Sharing::instance()->userSignedInToTwitter() ? tr("You are signed in to a twitter account") : tr("Sign in to a twitter account"), this)),
    m_facebookButton(new QPushButton(Sharing::instance()->userSignedInToFacebook() ? tr("Sign out") : tr("Sign in"), this)),
    m_twitterButton(new QPushButton(Sharing::instance()->userSignedInToTwitter() ? tr("Sign out") : tr("Sign in"), this)),
    m_scrollArea(new QScrollArea(this)),
    m_contextMenu(new QMenu(this)),
    m_dailymotionDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteDailymotionAccount()))),
    m_youtubeDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteYouTubeAccount()))),
    m_vimeoDeleteAction(m_contextMenu->addAction(tr("Delete"), this, SLOT(deleteVimeoAccount()))),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Accounts"));

    m_nameEdit->setPlaceholderText(tr("Display name"));
    m_nameEdit->setFocusPolicy(Qt::ClickFocus);
    m_loginButton->setEnabled(false);
    m_loginButton->setFixedWidth(150);
    m_serviceSelector->setValue(Services::YouTube);
    m_dailymotionView->setModel(m_dailymotionModel);
    m_dailymotionView->setItemDelegate(new AccountDelegate(m_dailymotionView));
    m_youtubeView->setModel(m_youtubeModel);
    m_youtubeView->setItemDelegate(new AccountDelegate(m_youtubeView));
    m_vimeoView->setModel(m_vimeoModel);
    m_vimeoView->setItemDelegate(new AccountDelegate(m_vimeoView));
    m_facebookLabel->setWordWrap(true);
    m_facebookLabel->setAlignment(Qt::AlignCenter);
    m_twitterLabel->setWordWrap(true);
    m_twitterLabel->setAlignment(Qt::AlignCenter);

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(new QLabel(tr("Add an account:"), this), 0, 0, 1, 2);
    grid->addWidget(m_nameEdit, 1, 0, 1, 2);
    grid->addWidget(m_serviceSelector, 2, 0);
    grid->addWidget(m_loginButton, 2, 1);
    grid->addWidget(new SeparatorLabel("YouTube", this), 3, 0, 1, 2);
    grid->addWidget(m_youtubeView, 4, 0, 1, 2);
    grid->addWidget(new SeparatorLabel("Dailymotion", this), 5, 0, 1, 2);
    grid->addWidget(m_dailymotionView, 6, 0, 1, 2);
    grid->addWidget(new SeparatorLabel("Vimeo", this), 7, 0, 1, 2);
    grid->addWidget(m_vimeoView, 8, 0, 1, 2);
    grid->addWidget(new SeparatorLabel("facebook", this), 9, 0, 1, 2);
    grid->addWidget(m_facebookLabel, 10, 0, 1, 2);
    grid->addWidget(m_facebookButton, 11, 0, 1, 2, Qt::AlignHCenter);
    grid->addWidget(new SeparatorLabel("twitter", this), 12, 0, 1, 2);
    grid->addWidget(m_twitterLabel, 13, 0, 1, 2);
    grid->addWidget(m_twitterButton, 14, 0, 1, 2, Qt::AlignHCenter);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);

    m_grid->addWidget(m_scrollArea, 0, 0);

    this->connect(m_nameEdit, SIGNAL(textChanged(QString)), this, SLOT(onNameTextChanged(QString)));
    this->connect(m_nameEdit, SIGNAL(returnPressed()), this, SLOT(showAuthDialog()));
    this->connect(m_loginButton, SIGNAL(clicked()), this, SLOT(showAuthDialog()));
    this->connect(m_youtubeView, SIGNAL(clicked(QModelIndex)), this, SLOT(switchYouTubeAccount(QModelIndex)));
    this->connect(m_youtubeView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showYouTubeContextMenu(QPoint)));
    this->connect(m_dailymotionView, SIGNAL(clicked(QModelIndex)), this, SLOT(switchDailymotionAccount(QModelIndex)));
    this->connect(m_dailymotionView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showDailymotionContextMenu(QPoint)));
    this->connect(m_vimeoView, SIGNAL(clicked(QModelIndex)), this, SLOT(switchVimeoAccount(QModelIndex)));
    this->connect(m_vimeoView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showVimeoContextMenu(QPoint)));
    this->connect(m_facebookButton, SIGNAL(clicked()), this, SLOT(onFacebookButtonClicked()));
    this->connect(m_twitterButton, SIGNAL(clicked()), this, SLOT(onTwitterButtonClicked()));
    this->connect(Sharing::instance(), SIGNAL(userSignedInToFacebookChanged()), this, SLOT(onSignedInOrOutOfFacebook()));
    this->connect(Sharing::instance(), SIGNAL(userSignedInToTwitterChanged()), this, SLOT(onSignedInOrOutOfTwitter()));

    this->onOrientationChanged();
}

void AccountsDialog::onNameTextChanged(const QString &text) {
    m_loginButton->setEnabled(!text.isEmpty());
}

void AccountsDialog::showAuthDialog() {
    switch (m_serviceSelector->currentValue()) {
    case Services::YouTube:
        this->showYouTubeAuthDialog();
        break;
    case Services::Dailymotion:
        this->showDailymotionAuthDialog();
        break;
    case Services::Vimeo:
        this->showVimeoAuthDialog();
        break;
    default:
        this->showYouTubeAuthDialog();
    }
}

void AccountsDialog::showYouTubeAuthDialog() {
    YouTubeAuthDialog *dialog = new YouTubeAuthDialog(m_nameEdit->text(), this);
    dialog->open();

    m_nameEdit->clear();
}

void AccountsDialog::showDailymotionAuthDialog() {
    DailymotionAuthDialog *dialog = new DailymotionAuthDialog(m_nameEdit->text(), this);
    dialog->open();

    m_nameEdit->clear();
}

void AccountsDialog::showVimeoAuthDialog() {
    VimeoAuthDialog *dialog = new VimeoAuthDialog(m_nameEdit->text(), this);
    dialog->open();

    m_nameEdit->clear();
}

void AccountsDialog::showFacebookAuthDialog() {
    FacebookAuthDialog *dialog = new FacebookAuthDialog(this);
    dialog->open();
}

void AccountsDialog::showTwitterAuthDialog() {
    TwitterAuthDialog *dialog = new TwitterAuthDialog(this);
    dialog->open();
}

void AccountsDialog::onFacebookButtonClicked() {
    if (Sharing::instance()->userSignedInToFacebook()) {
        Sharing::instance()->signOutOfFacebook();
    }
    else {
        this->showFacebookAuthDialog();
    }
}

void AccountsDialog::onTwitterButtonClicked() {
    if (Sharing::instance()->userSignedInToTwitter()) {
        Sharing::instance()->signOutOfTwitter();
    }
    else {
        this->showTwitterAuthDialog();
    }
}

void AccountsDialog::onSignedInOrOutOfFacebook() {
    bool signedIn = Sharing::instance()->userSignedInToFacebook();
    m_facebookButton->setText(signedIn ? tr("Sign out") : tr("Sign in"));
    m_facebookLabel->setText(signedIn ? tr("You are signed in to a facebook account") : tr("Sign in to a facebook account"));
}

void AccountsDialog::onSignedInOrOutOfTwitter() {
    bool signedIn = Sharing::instance()->userSignedInToTwitter();
    m_twitterButton->setText(signedIn ? tr("Sign out") : tr("Sign in"));
    m_twitterLabel->setText(signedIn ? tr("You are signed in to a twitter account") : tr("Sign in to a twitter account"));
}

void AccountsDialog::switchYouTubeAccount(const QModelIndex &index) {
    if (index.data(YouTubeAccountsListModel::UsernameRole).toString() != YouTube::instance()->username()) {
        m_youtubeModel->switchAccount(index.row());
    }
}

void AccountsDialog::switchDailymotionAccount(const QModelIndex &index) {
    if (index.data(DailymotionAccountsListModel::UsernameRole).toString() != Dailymotion::instance()->username()) {
        m_dailymotionModel->switchAccount(index.row());
    }
}

void AccountsDialog::switchVimeoAccount(const QModelIndex &index) {
    if (index.data(VimeoAccountsListModel::UsernameRole).toString() != Vimeo::instance()->username()) {
        m_vimeoModel->switchAccount(index.row());
    }
}

void AccountsDialog::showYouTubeContextMenu(const QPoint &pos) {
    if (m_youtubeView->currentIndex().isValid()) {
        m_youtubeDeleteAction->setVisible(true);
        m_dailymotionDeleteAction->setVisible(false);
        m_contextMenu->popup(pos, m_youtubeDeleteAction);
    }
}

void AccountsDialog::showDailymotionContextMenu(const QPoint &pos) {
    if (m_dailymotionView->currentIndex().isValid()) {
        m_dailymotionDeleteAction->setVisible(true);
        m_youtubeDeleteAction->setVisible(false);
        m_contextMenu->popup(pos, m_dailymotionDeleteAction);
    }
}

void AccountsDialog::showVimeoContextMenu(const QPoint &pos) {
    if (m_vimeoView->currentIndex().isValid()) {
        m_vimeoDeleteAction->setVisible(true);
        m_vimeoDeleteAction->setVisible(false);
        m_contextMenu->popup(pos, m_vimeoDeleteAction);
    }
}

void AccountsDialog::deleteYouTubeAccount() {
    m_youtubeModel->deleteAccount(m_youtubeView->currentIndex().row());
}

void AccountsDialog::deleteDailymotionAccount() {
    m_dailymotionModel->deleteAccount(m_dailymotionView->currentIndex().row());
}

void AccountsDialog::deleteVimeoAccount() {
    m_vimeoModel->deleteAccount(m_vimeoView->currentIndex().row());
}
