#include "settingsdialog.h"
#include "separatorlabel.h"
#include "../base/settings.h"
#include "../base/youtube.h"
#include "../base/dailymotion.h"
#include "../base/vimeo.h"
#include "mediaplayerselector.h"
#include "concurrenttransfersselector.h"
#include "screenorientationselector.h"
#include "languageselector.h"
#include "playbackqualitydialog.h"
#include "downloadqualitydialog.h"
#include "searchoptionsdialog.h"
#include "networkproxydialog.h"
#include <QCheckBox>
#include <QLineEdit>
#include <QPushButton>
#include <QScrollArea>
#include <QDialogButtonBox>
#include <QGridLayout>
#include <QFileDialog>
#include <QDBusInterface>

SettingsDialog::SettingsDialog(QWidget *parent) :
    AbstractDialog(parent),
    m_transferStatusCheckbox(new QCheckBox(tr("Start transfers automatically"), this)),
    m_safeSearchCheckbox(new QCheckBox(tr("Safe search"), this)),
    m_eventFeedCheckbox(new QCheckBox(tr("Publish latest videos to event feed"))),
    m_clipboardCheckbox(new QCheckBox(tr("Monitor clipboard for video URLs"))),
    m_searchButton(new QPushButton(tr("Search dialog options"), this)),
    m_playbackButton(new QPushButton(tr("Playback quality"), this)),
    m_downloadButton(new QPushButton(tr("Download quality"), this)),
    m_proxyButton(new QPushButton(tr("Network proxy"), this)),
    m_downloadFolderSelector(new QMaemo5ValueButton(tr("Download folder"), this)),
    m_mediaPlayerSelector(new MediaPlayerSelector(tr("Media player"), this)),
    m_concurrentSelector(new ConcurrentTransfersSelector(tr("Concurrent transfers"), this)),
    m_orientationSelector(new ScreenOrientationSelector(tr("Screen orientation"), this)),
    m_languageSelector(new LanguageSelector(tr("Language"), true, this)),
    m_mediaPlayerCommandEdit(new QLineEdit(Settings::instance()->mediaPlayerCommand(), this)),
    m_buttonBox(new QDialogButtonBox(QDialogButtonBox::Cancel | QDialogButtonBox::Save, Qt::Vertical, this)),
    m_scrollArea(new QScrollArea(this)),
    m_playbackDialog(0),
    m_downloadDialog(0),
    m_searchDialog(0),
    m_proxyDialog(0),
    m_grid(new QGridLayout(this))
{
    this->setWindowTitle(tr("Settings"));

    bool eventFeedInstalled = Settings::instance()->eventFeedInstalled();
    bool signedIn = (YouTube::instance()->userSignedIn()) || (Dailymotion::instance()->userSignedIn()) || (Vimeo::instance()->userSignedIn());
    m_eventFeedCheckbox->setEnabled((eventFeedInstalled) && (signedIn));
    m_mediaPlayerCommandEdit->setPlaceholderText(tr("Custom media player command"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(new SeparatorLabel(tr("Media/content"), this));
    vbox->addWidget(m_downloadFolderSelector);
    vbox->addWidget(m_mediaPlayerSelector);
    vbox->addWidget(m_mediaPlayerCommandEdit);
    vbox->addWidget(m_playbackButton);
    vbox->addWidget(m_downloadButton);
    vbox->addWidget(m_clipboardCheckbox);
    vbox->addWidget(new SeparatorLabel(tr("Search"), this));
    vbox->addWidget(m_safeSearchCheckbox);
    vbox->addWidget(m_searchButton);
    vbox->addWidget(new SeparatorLabel(tr("Transfers"), this));
    vbox->addWidget(m_concurrentSelector);
    vbox->addWidget(m_transferStatusCheckbox);
    vbox->addWidget(new SeparatorLabel(tr("Event feed"), this));
    vbox->addWidget(m_eventFeedCheckbox);

    if ((!eventFeedInstalled) || (!signedIn)) {
        QLabel *feedLabel = new QLabel(this);
        feedLabel->setWordWrap(true);
        feedLabel->setAlignment(Qt::AlignHCenter);

        if (!eventFeedInstalled) {
            feedLabel->setText(QString("<p style='color:red;'>%1</p>").arg(tr("Requires installation of Hildon Event Feed (package: hildonevents)")));
        }
        else {
            feedLabel->setText(QString("<p style='color:red;'>%1</p>").arg(tr("Requires you to be signed in to an account")));
        }

        vbox->addWidget(feedLabel);
    }

    vbox->addWidget(new SeparatorLabel(tr("Other"), this));
    vbox->addWidget(m_orientationSelector);
    vbox->addWidget(m_languageSelector);
    vbox->addWidget(m_proxyButton);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    this->connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(saveSettings()));
    this->connect(m_buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
    this->connect(m_downloadFolderSelector, SIGNAL(clicked()), this, SLOT(showFileDialog()));
    this->connect(m_searchButton, SIGNAL(clicked()), this, SLOT(showSearchOptionsDialog()));
    this->connect(m_playbackButton, SIGNAL(clicked()), this, SLOT(showPlaybackFormatDialog()));
    this->connect(m_downloadButton, SIGNAL(clicked()), this, SLOT(showDownloadFormatDialog()));
    this->connect(m_mediaPlayerSelector, SIGNAL(valueChanged(QString)), this, SLOT(onMediaPlayerChanged(QString)));
    this->connect(m_proxyButton, SIGNAL(clicked()), this, SLOT(showNetworkProxyDialog()));

    this->loadSettings();
    this->onOrientationChanged();
}

void SettingsDialog::setLandscapeLayout() {
    this->setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void SettingsDialog::setPortraitLayout() {
    this->setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void SettingsDialog::loadSettings() {
    m_transferStatusCheckbox->setChecked(Settings::instance()->startTransfersAutomatically());
    m_safeSearchCheckbox->setChecked(Settings::instance()->safeSearch());
    m_eventFeedCheckbox->setChecked(m_eventFeedEnabled = Settings::instance()->eventFeedEnabled());
    m_clipboardCheckbox->setChecked(Settings::instance()->monitorClipboard());
    m_downloadPath = Settings::instance()->downloadPath();
    m_downloadFolderSelector->setValueText(m_downloadPath);
    m_mediaPlayerSelector->setValue(Settings::instance()->mediaPlayer());
    m_concurrentSelector->setValue(Settings::instance()->maximumConcurrentTransfers());
    m_orientationSelector->setValue(Settings::instance()->screenOrientation());
    m_languageSelector->setValue(Settings::instance()->language());
    m_mediaPlayerCommandEdit->setEnabled(Settings::instance()->mediaPlayer() == "other");
}

void SettingsDialog::saveSettings() {
    Settings::instance()->setStartTransfersAutomatically(m_transferStatusCheckbox->isChecked());
    Settings::instance()->setSafeSearch(m_safeSearchCheckbox->isChecked());
    Settings::instance()->setEventFeedEnabled(m_eventFeedCheckbox->isChecked());
    Settings::instance()->setMonitorClipboard(m_clipboardCheckbox->isChecked());
    Settings::instance()->setDownloadPath(m_downloadPath);
    Settings::instance()->setMediaPlayer((m_mediaPlayerSelector->currentValue() == "other") && (m_mediaPlayerCommandEdit->text().isEmpty()) ? "mediaplayer" : m_mediaPlayerSelector->currentValue());
    Settings::instance()->setMediaPlayerCommand(m_mediaPlayerCommandEdit->text());
    Settings::instance()->setMaximumConcurrentTransfers(m_concurrentSelector->currentValue());
    Settings::instance()->setScreenOrientation(m_orientationSelector->currentValue());
    Settings::instance()->setLanguage(m_languageSelector->currentValue());

    bool eventFeedEnabled = Settings::instance()->eventFeedEnabled();

    if (m_eventFeedEnabled != eventFeedEnabled) {
        if (eventFeedEnabled) {
            this->scheduleEvents();
        }
        else {
            this->clearEvents();
        }
    }

    this->accept();
}

void SettingsDialog::showFileDialog() {
    m_downloadPath = QFileDialog::getExistingDirectory(this, tr("Download folder"), m_downloadPath);
    m_downloadFolderSelector->setValueText(m_downloadPath);
}

void SettingsDialog::showSearchOptionsDialog() {
    if (!m_searchDialog) {
        m_searchDialog = new SearchOptionsDialog(this);
    }

    m_searchDialog->open();
}

void SettingsDialog::showPlaybackFormatDialog() {
    if (!m_playbackDialog) {
        m_playbackDialog = new PlaybackFormatDialog(this);
    }

    m_playbackDialog->open();
}

void SettingsDialog::showDownloadFormatDialog() {
    if (!m_downloadDialog) {
        m_downloadDialog = new DownloadFormatDialog(this);
    }

    m_downloadDialog->open();
}

void SettingsDialog::showNetworkProxyDialog() {
    if (!m_proxyDialog) {
        m_proxyDialog = new NetworkProxyDialog(this);
    }

    m_proxyDialog->open();
}

void SettingsDialog::onMediaPlayerChanged(const QString &player) {
    m_mediaPlayerCommandEdit->setEnabled(player == "other");
}

void SettingsDialog::scheduleEvents() {
    QDBusInterface iface("com.maemo.cuteTubeEvents", "/", "com.maemo.cuteTubeEvents");
    iface.call(QDBus::NoBlock, "enableSync");
}

void SettingsDialog::clearEvents() {
    QDBusInterface iface("com.maemo.cuteTubeEvents", "/", "com.maemo.cuteTubeEvents");
    iface.call(QDBus::NoBlock, "disableSync");
}
