#include "youtube.h"
#include "json.h"
#include "utils.h"
#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>
#include <QRegExp>
#include <QMap>
#include <QStringList>
#include <QDebug>
#include <QTimer>
#ifdef QML_USER_INTERFACE
#include <QDeclarativeEngine>
#endif

using namespace QtJson;

YouTube* youtubeInstance = 0;

const QByteArray CLIENT_ID("446197852855.apps.googleusercontent.com");
const QByteArray CLIENT_SECRET("YzuBGnTyRdz9cK5J3Pg5Xlte");
const QByteArray DEVELOPER_KEY("AI39si6x9O1gQ1Z_BJqo9j2n_SdVsHu1pk2uqvoI3tVq8d6alyc1og785IPCkbVY3Q5MFuyt-IFYerMYun0MnLdQX5mo2BueSw");

YouTube::YouTube(QObject *parent) :
    QObject(parent),
    m_nam(0),
    m_safeSearch(false),
    m_actionsProcessed(0),
    m_playlistCache(new QList< QSharedPointer<PlaylistItem> >),
    m_subscriptionCache(new QList< QSharedPointer<UserItem> >),
    m_playlistCacheLoaded(false),
    m_subscriptionCacheLoaded(false),
    m_busy(false),
    m_cancelled(false)
{
    if (!youtubeInstance) {
        youtubeInstance = this;
    }

    m_queryOrders[Queries::Relevance] = "relevance";
    m_queryOrders[Queries::Date] = "published";
    m_queryOrders[Queries::Views] = "viewCount";
    m_queryOrders[Queries::Rating] = "rating";

    m_timeFilters[Queries::AllTime] = "all_time";
    m_timeFilters[Queries::ThisWeek] = "this_week";
    m_timeFilters[Queries::ThisMonth] = "this_month";

    m_durationFilters[Queries::Any] = "";
    m_durationFilters[Queries::Short] = "short";
    m_durationFilters[Queries::Medium] = "medium";
    m_durationFilters[Queries::Long] = "long";
}

YouTube::~YouTube() {
    this->clearCache();
}

YouTube* YouTube::instance() {
    return youtubeInstance;
}

void YouTube::setBusy(bool isBusy, const QString &message, int numberOfOperations) {
    if (isBusy != this->busy()) {
        m_busy = isBusy;

        if (isBusy) {
            this->setCancelled(false);
            emit busy(message, numberOfOperations);
        }
        else if (!this->cancelled()) {
            emit busyProgressChanged(numberOfOperations);
        }

        emit busyChanged(isBusy);
    }
}

void YouTube::cancelCurrentOperation() {
    this->setCancelled(true);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, 0);
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, 0);
#ifdef QML_USER_INTERFACE
    this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
    this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    emit currentOperationCancelled();
}

QNetworkReply* YouTube::createReply(QUrl feed, int offset) {
    if (this->userSignedIn()) {
        feed.addQueryItem("access_token", this->accessToken());
    }

    if (offset) {
        feed.addQueryItem("start-index", QString::number(offset));
    }

    qDebug() << feed;

    QNetworkRequest request(feed);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);

    return this->networkAccessManager()->get(request);
}

QNetworkReply* YouTube::createSearchReply(int queryType, const QString &query, int offset, int order, int time, int duration, const QString &language) {
    QUrl url;

    switch (queryType) {
    case Queries::Videos:
        url.setUrl(YOUTUBE_VIDEOS_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_VIDEO_FIELDS);
        url.addQueryItem("orderby", m_queryOrders.value(order, "relevance"));
        url.addQueryItem("time", m_timeFilters.value(time, "all_time"));

        if (duration != Queries::Any) {
            url.addQueryItem("duration", m_durationFilters.value(duration, "medium"));
        }
        if ((!language.isEmpty()) && (language != "all")) {
            url.addQueryItem("lr", language);
        }
        break;
    case Queries::Playlists:
        url.setUrl(YOUTUBE_PLAYLISTS_SEARCH_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_PLAYLIST_FIELDS);
        url.addQueryItem("orderby", "relevance");
        break;
    case Queries::Users:
        url.setUrl(YOUTUBE_CHANNELS_BASE_URL);
        url.addQueryItem("fields", YOUTUBE_USER_FIELDS);
        url.addQueryItem("orderby", "relevance");
        break;
    default:
        qWarning() << "YouTube::createSearchReply(): No/invalid query type.";
        break;
    }

    url.addQueryItem("q", QString("\"%1\"|%2").arg(query).arg(query.simplified().replace(' ', '+')));
    url.addQueryItem("v", "2.1");
    url.addQueryItem("safeSearch", this->safeSearch() ? "strict" : "none");
    url.addQueryItem("max-results", "30");

    if (offset) {
        url.addQueryItem("start-index", QString::number(offset));
    }

    qDebug() << url;

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());

    return this->networkAccessManager()->get(request);
}

QNetworkReply* YouTube::createUploadReply(const QVariantMap &metadata) {
    QUrl url("http://uploads.gdata.youtube.com/resumable/feeds/api/users/default/uploads");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + metadata.value("title").toString().toUtf8() + "</media:title>\n" \
                   "<media:description>\n" + metadata.value("description").toString().toUtf8() + "\n\n" + metadata.value("uploadAttribute").toString().toUtf8() + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">\n" + metadata.value("category").toString().toUtf8() + "\n</media:category>\n" \
                   "<media:keywords>" + metadata.value("tags").toString().toUtf8() + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "</entry>");

    if (metadata.value("isPrivate").toBool()) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "uploads.gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml; charset=UTF-8");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + this->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2");
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    request.setRawHeader("Slug", metadata.value("filePath").toString().section('/', -1).toUtf8());
    return this->networkAccessManager()->post(request, xml);
}

QUrl YouTube::authUrl() const {
    QUrl url("https://accounts.google.com/o/oauth2/auth");
    url.addQueryItem("client_id", CLIENT_ID);
    url.addQueryItem("redirect_uri", "urn:ietf:wg:oauth:2.0:oob");
    url.addQueryItem("response_type", "code");
    url.addQueryItem("scope", "https://gdata.youtube.com");
    url.addQueryItem("access_type", "offline");
    url.addQueryItem("display", "popup");

    return url;
}

void YouTube::clearLoginInfo() {
    m_login.displayName = QString();
    m_login.deviceCode = QString();
    m_login.deviceInterval = 0;
}

void YouTube::signIn(const QString &displayName, const QString &code) {
    this->setBusy(true, tr("Signing in"));
    m_login.displayName = displayName;
    QUrl url("https://accounts.google.com/o/oauth2/token");

    QByteArray body("code=" + code.toUtf8() +
                    "&client_id=" + CLIENT_ID +
                    "&client_secret=" + CLIENT_SECRET +
                    "&redirect_uri=urn:ietf:wg:oauth:2.0:oob&grant_type=authorization_code");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedIn()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::checkIfSignedIn() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit error(tr("Cannot parse server response"));
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if ((token.isEmpty()) || (refresh.isEmpty())) {
            QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
            emit error(statusText);
        }
        else {
            emit alert(tr("You are signed in to your YouTube account"));
            emit signedIn(m_login.displayName, token, refresh);
        }
    }

    reply->deleteLater();
}

void YouTube::signInFromDevice(const QString &displayName) {
    this->setBusy(true, tr("Requesting authorisation code"));
    m_login.displayName = displayName;
    QUrl url("https://accounts.google.com/o/oauth2/device/code");
    QByteArray body("scope=https://gdata.youtube.com&client_id=" + CLIENT_ID);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDeviceSignInInfo()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::checkDeviceSignInInfo() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    qDebug() << response;
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        this->clearLoginInfo();
        emit error(tr("Cannot parse server response"));
    }
    else {
        QString deviceCode = result.value("device_code").toString();
        QString userCode = result.value("user_code").toString();
        QString verificationUrl = result.value("verification_url").toString();
        int interval = result.value("interval").toInt();

        if ((!deviceCode.isEmpty()) && (!userCode.isEmpty()) && (!verificationUrl.isEmpty()) && (interval > 0)) {
            m_login.deviceCode = deviceCode;
            m_login.deviceInterval = interval * 1100;
            this->setBusy(true, QString("%1\n\n%2\n\n%3:\n\n%4").arg(tr("Please visit")).arg(verificationUrl).arg(tr("using your web browser and enter the following")).arg(userCode));
            QTimer::singleShot(m_login.deviceInterval, this, SLOT(pollServerForDeviceToken()));
        }
        else {
            this->clearLoginInfo();
            QString errorString = result.value("error").toString();
            emit error(errorString.isEmpty() ? tr("Unable to obtain authorisation code") : errorString);
        }
    }

    reply->deleteLater();
}

void YouTube::pollServerForDeviceToken() {
    if (m_login.deviceCode.isEmpty()) {
        this->setBusy(false);
        emit error(tr("No device code for authorisation"));
        return;
    }

    QUrl url("https://accounts.google.com/o/oauth2/token");

    QByteArray body("client_id=" + CLIENT_ID +
                    "&client_secret=" + CLIENT_SECRET +
                    "&grant_type=http://oauth.net/grant_type/device/1.0" +
                    "&code=" + m_login.deviceCode.toUtf8());

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkDeviceToken()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::checkDeviceToken() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        this->setBusy(false);
        this->clearLoginInfo();
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    qDebug() << response;
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        this->setBusy(false);
        this->clearLoginInfo();
        emit error(tr("Cannot parse server response"));
    }
    else {
        QString token = result.value("access_token").toString();
        QString refresh = result.value("refresh_token").toString();

        if ((token.isEmpty()) || (refresh.isEmpty())) {
            QString errorString = result.value("error").toString();

            if (errorString == "authorization_pending") {
                QTimer::singleShot(m_login.deviceInterval, this, SLOT(pollServerForDeviceToken()));
            }
            else if (errorString == "slow_down") {
                m_login.deviceInterval = m_login.deviceInterval * 2;
                QTimer::singleShot(m_login.deviceInterval, this, SLOT(pollServerForDeviceToken()));
            }
            else {
                this->setBusy(false);
                this->clearLoginInfo();
                emit error(errorString.isEmpty() ? tr("Unable to obtain access token") : errorString);
            }
        }
        else {
            this->setBusy(false);
            emit alert(tr("You are signed in to your YouTube account"));
            emit signedIn(m_login.displayName, token, refresh);
        }
    }

    reply->deleteLater();
}

void YouTube::signOut() {
    this->setBusy(true, tr("Signing out"));
    QUrl url("https://accounts.google.com/o/oauth2/revoke");
    url.addQueryItem("token", this->accessToken());
    QNetworkRequest request(url);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkIfSignedOut()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::checkIfSignedOut() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        this->setAccount();
        emit info(tr("You have signed out of your YouTube account"));
    }
    else {
        emit error(tr("Unable to sign out of your YouTube account. Please visit the YouTube website to revoke access"));
    }

    reply->deleteLater();
}

void YouTube::refreshAccessToken() {
    QUrl url("https://accounts.google.com/o/oauth2/token");

    QByteArray body("client_id=" + CLIENT_ID +
                    "&client_secret=" + CLIENT_SECRET +
                    "&refresh_token=" + refreshToken().toUtf8() +
                    "&grant_type=refresh_token");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");
    QNetworkReply *reply = this->networkAccessManager()->post(request, body);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkTokenRefresh()));
}

void YouTube::checkTokenRefresh() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QString response(reply->readAll());
    bool ok;
    QVariantMap result = Json::parse(response, ok).toMap();

    if (!ok) {
        emit postFailed(tr("Cannot parse server response"));
        emit refreshError();
    }
    else {
        QString token = result.value("access_token").toString();

        if (token.isEmpty()) {
            QString error = result.value("error").toString();

            if (error == "invalid_request") {
                emit postFailed(tr("Error refreshing access token. Token is no longer valid"));
                emit refreshError();
                this->setAccount();
            }
            else {
                QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit postFailed(statusText);
                emit refreshError();
            }

            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, 0);
        }
        else {
            this->setAccessToken(token);
            emit accessTokenRefreshed(token);
        }
    }

    reply->deleteLater();
}

void YouTube::linkGoogleAccount(QString username) {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default?v=2.1");
    QByteArray xml("<entry xmlns='http://www.w3.org/2005/Atom'\n" \
                   "xmlns:yt='http://gdata.youtube.com/schemas/2007'>\n" \
                   "<yt:username>" + username.toUtf8() + "</yt:username>\n" \
                   "</entry>");

    this->putRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onGoogleAccountLinked(QString)));
}

void YouTube::onGoogleAccountLinked(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    QString username = entry.firstChildElement("yt:username").attribute("display");
    QString userId = entry.firstChildElement("yt:userId").text();
    this->setUsername(userId);

    emit googleAccountLinked();
    emit info(tr("Your Google account is now linked to your YouTube account '%1'").arg(username));
}

void YouTube::checkUsernameAvailability(QString username) {
    QUrl url("https://gdata.youtube.com/feeds/api/suggest/username");
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", "entry/title");
    url.addQueryItem("hint", username);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkSuggestedUsernames()));
}

void YouTube::checkSuggestedUsernames() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");
    QStringList usernames;

    for (int i = 0; i < entries.size(); i++) {
        usernames.append(entries.at(i).firstChildElement("title").text());
    }

    if (usernames.isEmpty()) {
        emit usernameUnavailable();
    }
    else if (usernames.contains(reply->request().url().queryItemValue("hint"))) {
        emit usernameAvailable();
        this->linkGoogleAccount(reply->request().url().queryItemValue("hint"));
    }
    else {
        emit gotSuggestedUsernames(usernames);
    }

    reply->deleteLater();
}

void YouTube::cancelLinkGoogleAccount() {
    this->disconnect(this, SIGNAL(googleAccountLinked()), this, 0);
}

void YouTube::setAccount(const QString &user, const QString &token, const QString &refresh) {
    if (user != this->username()) {
        this->setUsername(user);
    }

    this->setAccessToken(token);
    this->setRefreshToken(refresh);
    this->clearCache();

    if ((user.isEmpty()) && (!token.isEmpty())) {
        emit newAccountSet();
    }
}

void YouTube::clearCache() {
    m_playlistCache->clear();
    m_subscriptionCache->clear();

    this->setPlaylistsLoaded(false);
    this->setSubscriptionsLoaded(false);
}

void YouTube::postRequest(const QUrl &url, const QByteArray &xml) {
    qDebug() << url;
    qDebug() << xml;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + this->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    QNetworkReply* reply = this->networkAccessManager()->post(request, xml);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::putRequest(const QUrl &url, const QByteArray &xml) {
    qDebug() << url;
    qDebug() << xml;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    request.setRawHeader("Authorization", "Bearer " + this->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    QNetworkReply* reply = this->networkAccessManager()->put(request, xml);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::deleteRequest(const QUrl &url) {
    qDebug() << url;
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setRawHeader("Host", "gdata.youtube.com");
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setRawHeader("Authorization", "Bearer " + this->accessToken().toUtf8());
    request.setRawHeader("GData-Version", "2.1");
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    QNetworkReply* reply = this->networkAccessManager()->deleteResource(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(postFinished()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::postFinished() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if ((statusCode == 401) && (this->userSignedIn())) {
        this->refreshAccessToken();
    }
    else {
        if ((statusCode == 200) || (statusCode == 201)) {
            QString response(reply->readAll());
            emit postSuccessful(response);
        }
        else {
            QDomDocument doc;
            doc.setContent(reply->readAll());
            QDomNode errorNode = doc.namedItem("errors").namedItem("error");

            if (!errorNode.isNull()) {
                QString errorString = errorNode.firstChildElement("internalReason").text();
                QString errorCode = errorNode.firstChildElement("code").text();

                if (errorString.contains(QRegExp("(already in favorite|already in playlist)"))) {
                    emit postSuccessful(QString());
                }
                else if (errorCode == "youtube_signup_required") {
                    emit requestToLinkGoogleAccount();
                }
                else if (!errorString.isEmpty()) {
                    emit postFailed(errorString);
                }
            }
            else {
                QString statusText = reply->attribute(QNetworkRequest::HttpReasonPhraseAttribute).toString();
                emit postFailed(statusText);
            }
        }
    }

    reply->deleteLater();
}

void YouTube::getPlaylists(int offset) {
    this->setPlaylistsLoaded(false);
    QUrl url(YOUTUBE_PLAYLISTS_FEED);
    url.addQueryItem("start-index", QString::number(offset));
    url.addQueryItem("fields", YOUTUBE_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addPlaylists()));
}

void YouTube::addPlaylists() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            PlaylistItem *playlist = new PlaylistItem;
            playlist->loadYouTubePlaylist(entries.at(i));
            m_playlistCache->append(QSharedPointer<PlaylistItem>(playlist));
        }

        if ((!doc.namedItem("feed").namedItem("link").isNull()) && (!entries.isEmpty())) {
            this->getPlaylists(m_playlistCache->size() + 1);
        }
        else {
            this->setPlaylistsLoaded(true);
            emit allPlaylistsLoaded();
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getPlaylists()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void YouTube::addNewPlaylistToCache(QSharedPointer<PlaylistItem> playlist) {
    m_playlistCache->insert(0, playlist);
    emit playlistAddedToCache(0);
}

QSharedPointer<PlaylistItem> YouTube::removePlaylistFromCache(const QString &id) {
    int i = 0;
    bool removed = false;
    QSharedPointer<PlaylistItem> playlist;

    while ((!removed) && (i < m_playlistCache->size())) {
        removed = (m_playlistCache->at(i).data()->playlistId() == id);
        i++;
    }

    if (removed) {
        playlist = m_playlistCache->takeAt(i - 1);
        emit playlistRemovedFromCache(i - 1);
    }

    return playlist;
}

void YouTube::updatePlaylistVideoCount(const QString &id, int change) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i)->playlistId() == id);
        i++;
    }

    if (updated) {
        m_playlistCache->at(i - 1).data()->setVideoCount(m_playlistCache->at(i - 1).data()->videoCount() + change);
        emit playlistUpdated(i - 1);
    }
}

void YouTube::updatePlaylistThumbnail(const QString &id, const QUrl &thumbnailUrl) {
    int i = 0;
    bool updated = false;

    while ((!updated) && (i < m_playlistCache->size())) {
        updated = (m_playlistCache->at(i).data()->playlistId() == id);
        i++;
    }

    if ((updated) && (m_playlistCache->at(i - 1).data()->thumbnailUrl().isEmpty())) {
        m_playlistCache->at(i - 1).data()->setThumbnailUrl(thumbnailUrl);
        emit playlistUpdated(i - 1);
    }
}

void YouTube::getSubscriptions(int offset) {
    this->setSubscriptionsLoaded(false);
    QUrl url(YOUTUBE_SUBSCRIPTIONS_FEED);
    url.addQueryItem("start-index", QString::number(offset));
    url.addQueryItem("fields", YOUTUBE_SUBSCRIPTION_FIELDS);
    url.addQueryItem("access_token", this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(addSubscriptions()));
}

void YouTube::addSubscriptions() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getSubscriptions()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNodeList entries = doc.elementsByTagName("entry");

        for (int i = 0; i < entries.size(); i++) {
            UserItem *user = new UserItem;
            user->loadYouTubeUser(entries.at(i), true, true);
            m_subscriptionCache->append(QSharedPointer<UserItem>(user));
        }

        if ((!doc.namedItem("feed").namedItem("link").isNull()) && (!entries.isEmpty())) {
            this->getSubscriptions(m_subscriptionCache->size() + 1);
        }
        else {
            this->setSubscriptionsLoaded(true);
            emit allSubscriptionsLoaded();
        }

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getSubscriptions()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void YouTube::addNewSubscriptionToCache(QSharedPointer<UserItem> user) {
    m_subscriptionCache->insert(0, user);
    emit subscriptionAddedToCache(0);
    emit subscriptionChanged(user.data()->id(), true, user.data()->subscriptionId());
}

QSharedPointer<UserItem> YouTube::removeSubscriptionFromCache(const QString &id) {
    int i = 0;
    bool removed = false;

    QSharedPointer<UserItem> user;

    while ((!removed) && (i < m_subscriptionCache->size())) {
        removed = (m_subscriptionCache->at(i).data()->subscriptionId() == id);
        i++;
    }

    if (removed) {
        user = m_subscriptionCache->takeAt(i - 1);
        emit subscriptionRemovedFromCache(i - 1);
        emit subscriptionChanged(id, false, QString());
    }

    return user;
}

bool YouTube::subscribedToChannel(const QString &userId) {
    if (!userId.isEmpty()) {
        m_actionId = userId;
    }

    int i = 0;
    bool subscribed = false;

    while ((i < m_subscriptionCache->size()) && (!subscribed)) {
        subscribed = (m_subscriptionCache->at(i).data()->id() == m_actionId);
        i++;
    }

    this->disconnect(this, 0, this, SLOT(subscribedToChannel()));

    emit subscriptionChanged(m_actionId, subscribed, subscribed ? m_subscriptionCache->at(i - 1).data()->subscriptionId() : QString());

    return subscribed;
}

void YouTube::getFullVideo(QString id) {
    this->setBusy(true, tr("Retrieving video details"));

    if (id.size() > 11) {
        id = id.section(QRegExp("(v=|/)"), -1).left(11);
    }

    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_VIDEO_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideo()));
    this->connect(this, SIGNAL(currentOperationCancelled()), reply, SLOT(deleteLater()));
}

void YouTube::checkFullVideo() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    if (this->cancelled()) {
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        VideoItem *video = new VideoItem;
        video->loadYouTubeVideo(doc.namedItem("entry"));
#ifdef QML_USER_INTERFACE
        QDeclarativeEngine::setObjectOwnership(video, QDeclarativeEngine::JavaScriptOwnership);
        emit gotVideo(video);
#else
        emit gotVideo(QSharedPointer<VideoItem>(video));
#endif
    }
    else {
        emit error(tr("Video could not be retrieved"));
#ifdef QML_USER_INTERFACE
        this->disconnect(this, SIGNAL(gotVideo(VideoItem*)), 0, 0);
#else
        this->disconnect(this, SIGNAL(gotVideo(QSharedPointer<VideoItem>)), 0, 0);
#endif
    }

    reply->deleteLater();
}

void YouTube::getVideoMetadata(const QString &id) {
    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("access_token", this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkVideoMetadata()));
}

void YouTube::checkVideoMetadata() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideoMetadata()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        QDomNode entry = doc.namedItem("entry");
        QDomElement mediaElement = entry.firstChildElement("media:group");
        QVariantMap metadata;
        metadata.insert("title", mediaElement.firstChildElement("media:title").text());
        metadata.insert("description", mediaElement.firstChildElement("media:description").text());
        metadata.insert("tags", mediaElement.firstChildElement("media:keywords").text());
        metadata.insert("category", mediaElement.firstChildElement("media:category").text());

        QDomElement permissionsElement = entry.firstChildElement("yt:accessControl");
        metadata.insert("commentsPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("commentVotingPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("responsesPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("ratingsPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("embedPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("listingsPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("autoPlayPermission", permissionsElement.attribute("permission"));

        permissionsElement = permissionsElement.nextSiblingElement("yt:accessControl");
        metadata.insert("syndicationPermission", permissionsElement.attribute("permission"));

        metadata.insert("isPrivate", !mediaElement.firstChildElement("yt:private").isNull());

        emit gotVideoMetadata(metadata);

        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getVideoMetadata()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void YouTube::getFullComment(const QString &videoId, const QString &commentId) {
    QUrl url(YOUTUBE_VIDEOS_BASE_URL + QString("/%1/comments/%2").arg(videoId).arg(commentId));
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_COMMENT_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullComment()));
}

void YouTube::checkFullComment() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        CommentItem *comment = new CommentItem;
        comment->loadYouTubeComment(doc.namedItem("entry"));
        emit gotComment(QSharedPointer<CommentItem>(comment));
    }
    else {
        emit error(tr("Comment could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::getCurrentUserProfile() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default");
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_USER_PROFILE_FIELDS);
    url.addQueryItem("access_token", this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCurrentUserProfile()));
}

void YouTube::checkCurrentUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 401) {
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getCurrentUserProfile()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
        this->refreshAccessToken();
    }
    else {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        UserItem *user = new UserItem;
        user->loadYouTubeUser(doc.namedItem("entry"));

        if (this->username() != user->id()) {
            this->setUsername(user->id());
        }

#ifdef QML_USER_INTERFACE
        QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
        emit gotUser(user);
#else
        emit gotUser(QSharedPointer<UserItem>(user));
#endif
        this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(getCurrentUserProfile()));
        this->disconnect(this, SIGNAL(postFailed(QString)), this, SIGNAL(error(QString)));
    }

    reply->deleteLater();
}

void YouTube::getUserProfile(const QString &id) {
    QUrl url(YOUTUBE_USERS_BASE_URL + QString("/") + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_USER_PROFILE_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkUserProfile()));
}

void YouTube::checkUserProfile() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        UserItem *user = new UserItem;
        user->loadYouTubeUser(doc.namedItem("entry"));

        if (this->subscriptionsLoaded()) {
            this->subscribedToChannel(user->id());
        }
        else if (this->userSignedIn()) {
            m_actionId = user->id();
            this->getSubscriptions();
            this->connect(this, SIGNAL(allSubscriptionsLoaded()), this, SLOT(subscribedToChannel()));
        }
#ifdef QML_USER_INTERFACE
        QDeclarativeEngine::setObjectOwnership(user, QDeclarativeEngine::JavaScriptOwnership);
        emit gotUser(user);
#else
        emit gotUser(QSharedPointer<UserItem>(user));
#endif
    }
    else {
        emit error(tr("Profile could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::updateVideoMetadata(const QVariantMap &metadata) {
    m_metadataAction = metadata;
    this->updateVideoMetadata();
}

void YouTube::updateVideoMetadata() {
    QUrl url(QString("https://gdata.youtube.com/feeds/api/users/default/uploads/%1?v=2.1").arg(m_metadataAction.value("videoId").toString()));
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<media:group>\n" \
                   "<media:title>" + m_metadataAction.value("title").toString().toUtf8() + "</media:title>\n" \
                   "<media:description>\n" + m_metadataAction.value("description").toString().toUtf8() + "\n</media:description>\n" \
                   "<media:category scheme=\"http://gdata.youtube.com/schemas/2007/categories.cat\">" + m_metadataAction.value("category").toString().toUtf8() + "</media:category>\n" \
                   "<media:keywords>" + m_metadataAction.value("tags").toString().toUtf8() + "</media:keywords>\n" \
                   "</media:group>\n" \
                   "<yt:accessControl action=\"comment\" permission=\"" + m_metadataAction.value("commentsPermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"commentVote\" permission=\"" + m_metadataAction.value("commentVotePermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"rate\" permission=\"" + m_metadataAction.value("ratingsPermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"syndicate\" permission=\"" + m_metadataAction.value("syndicationPermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"list\" permission=\"" + m_metadataAction.value("listingsPermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"embed\" permission=\"" + m_metadataAction.value("embedPermission").toString().toUtf8() + "\"/>\n" \
                   "<yt:accessControl action=\"videoRespond\" permission=\"" + m_metadataAction.value("responsesPermission").toString().toUtf8() + "\"/>\n" \
                   "</entry>");

    if (m_metadataAction.value("isPrivate").toBool()) {
        xml.insert(xml.indexOf("</media:group>"), "<yt:private/>\n");
    }

    this->putRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoMetadataUpdated()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(updateVideoMetadata()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoMetadataUpdated() {
    emit videoMetadataUpdated(m_metadataAction.value("videoId").toString(), m_metadataAction);
    emit alert(tr("Video metadata updated"));
    m_metadataAction.clear();

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoMetadataUpdated()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(updateVideoMetadata()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::deleteFromUploads(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromUploads();
        this->setBusy(true, tr("Deleting video(s) from uploads"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::deleteFromUploads() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/users/default/uploads/" + m_actionIdList.first());
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromUploads()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onVideoDeleted() {
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromUploads(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromUploads();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from uploads"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDeleted()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromUploads()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::addToFavourites(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->addToFavourites();
        this->setBusy(true, tr("Adding video(s) to favourites"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::addToFavourites() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/users/default/favorites");
        QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                       "<entry xmlns=\"http://www.w3.org/2005/Atom\">\n" \
                       "<id>" + m_actionIdList.first().toUtf8() + "</id>\n" \
                       "</entry>");

        this->postRequest(url, xml);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites(QString)));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onAddedToFavourites(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");

    if (!m_actionIdList.isEmpty()) {
        emit favouriteChanged(m_actionIdList.takeFirst(), true, entry.firstChildElement("id").text().section(':', -1));
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToFavourites();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to favourites"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToFavourites(QString)));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToFavourites()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::deleteFromFavourites(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromFavourites();
        this->setBusy(true, tr("Deleting video(s) from favourites"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::deleteFromFavourites() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/users/default/favorites/" + m_actionIdList.first());
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromFavourites()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onDeletedFromFavourites() {
    if (!m_actionIdList.isEmpty()) {
        emit favouriteChanged(m_actionIdList.takeFirst(), false, QString());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromFavourites();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from favourites"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromFavourites()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromFavourites()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::addToPlaylist(const QStringList &videoIds, const QString &playlistId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = playlistId;
        m_actionsProcessed = 0;
        this->addToPlaylist();
        this->setBusy(true, tr("Adding video(s) to playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::addToPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/playlists/" + m_actionId);
        QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                       "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                       "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                       "<id>" + m_actionIdList.first().toUtf8() + "</id>\n" \
                       "</entry>");

        this->postRequest(url, xml);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToPlaylist(QString)));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        this->disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }
}

void YouTube::onAddedToPlaylist(const QString &response) {
    if (this->playlistsLoaded()) {
        QDomDocument doc;
        doc.setContent(response);
        QDomNode entry = doc.firstChildElement("entry");
        QUrl thumbnailUrl(entry.firstChildElement("media:group").firstChildElement("media:thumbnail").attribute("url"));
        this->updatePlaylistThumbnail(m_actionId, thumbnailUrl);
        this->updatePlaylistVideoCount(m_actionId, 1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit addedToPlaylist(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToPlaylist(QString)));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::deleteFromPlaylist(const QStringList &videoIds, const QString &playlistId) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionId = playlistId;
        m_actionsProcessed = 0;
        this->deleteFromPlaylist();
        this->setBusy(true, tr("Deleting video(s) from playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::deleteFromPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url(QString("https://gdata.youtube.com/feeds/api/playlists/%1/%2").arg(m_actionId).arg(m_actionIdList.first()));
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromPlaylist()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onDeletedFromPlaylist() {
    if (this->playlistsLoaded()) {
        this->updatePlaylistVideoCount(m_actionId, -1);
    }
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromPlaylist(m_actionIdList.takeFirst(), m_actionId);
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromPlaylist()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::addToWatchLaterPlaylist(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->addToWatchLaterPlaylist();
        this->setBusy(true, tr("Adding video(s) to playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::addToWatchLaterPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/users/default/watch_later");
        QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                       "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                       "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                       "<id>" + m_actionIdList.first().toUtf8() + "</id>\n" \
                       "</entry>");

        this->postRequest(url, xml);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToWatchLaterPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onAddedToWatchLaterPlaylist(const QString &response) {
    Q_UNUSED(response)

    if (!m_actionIdList.isEmpty()) {
        emit addedToWatchLaterPlaylist(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->addToWatchLaterPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) added to 'Watch Later' playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onAddedToWatchLaterPlaylist(QString)));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addToWatchLaterPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::deleteFromWatchLaterPlaylist(const QStringList &videoIds) {
    if (!videoIds.isEmpty()) {
        m_actionIdList = videoIds;
        m_actionsProcessed = 0;
        this->deleteFromWatchLaterPlaylist();
        this->setBusy(true, tr("Deleting video(s) to playlist"), m_actionIdList.size());
    }
    else {
        emit error(tr("No videos specified"));
    }
}

void YouTube::deleteFromWatchLaterPlaylist() {
    if (!m_actionIdList.isEmpty()) {
        QUrl url("https://gdata.youtube.com/feeds/api/playlists/watch_later/" + m_actionIdList.first());
        this->deleteRequest(url);
        this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist()));
        this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromWatchLaterPlaylist()));
        this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    }
}

void YouTube::onDeletedFromWatchLaterPlaylist() {    
    if (!m_actionIdList.isEmpty()) {
        emit deletedFromWatchLaterPlaylist(m_actionIdList.takeFirst());
    }
    if (!this->cancelled()) {
        m_actionsProcessed++;
        emit busyProgressChanged(m_actionsProcessed);

        if (!m_actionIdList.isEmpty()) {
            this->deleteFromWatchLaterPlaylist();
        }
        else {
            this->setBusy(false);
            emit alert(tr("Video(s) deleted from 'Watch Later' playlist"));

            this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onDeletedFromWatchLaterPlaylist()));
            this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deleteFromWatchLaterPlaylist()));
            this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
        }
    }
}

void YouTube::createPlaylist(const QVariantMap &playlist, const QStringList &videoIds) {
    m_metadataAction = playlist;
    m_actionIdList = videoIds;
    this->createPlaylist();
}

void YouTube::createPlaylist() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<title>" + m_metadataAction.value("title").toString().toUtf8() + "</title>\n" \
                   "<summary>" + m_metadataAction.value("description").toString().toUtf8() + "</summary>\n" \
                   "</entry>");

    if (m_metadataAction.value("isPrivate").toBool()) {
        int index = xml.lastIndexOf("<");
        xml.insert(index, "<yt:private/>\n");
    }

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistCreated(QString)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(createPlaylist()));
    this->connect(this, SIGNAL(googleAccountLinked()), this, SLOT(createPlaylist()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::onPlaylistCreated(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    m_actionId = entry.firstChildElement("yt:playlistId").text();

    if (!m_actionIdList.isEmpty()) {
        m_actionsProcessed = 0;

        if (this->playlistsLoaded()) {
            this->getPlaylistForCache(m_actionId);
            this->connect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
        }
        else {
            this->addToPlaylist();
        }
    }
    else {
        if (this->playlistsLoaded()) {
            this->getPlaylistForCache(m_actionId);
        }

        emit alert(tr("Playlist created"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistCreated(QString)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(createPlaylist()));
    this->disconnect(this, SIGNAL(googleAccountLinked()), this, SLOT(createPlaylist()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::getPlaylistForCache(const QString &id) {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists/" + id);
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_PLAYLIST_FIELDS);
    url.addQueryItem("access_token", this->accessToken());
    QNetworkRequest request(url);
    request.setRawHeader("X-Gdata-Key", "key=" + DEVELOPER_KEY);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkCachePlaylist()));
}

void YouTube::checkCachePlaylist() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        PlaylistItem *playlist = new PlaylistItem;
        playlist->loadYouTubePlaylist(doc.namedItem("entry"));
        this->addNewPlaylistToCache(QSharedPointer<PlaylistItem>(playlist));
    }
    else {
        emit error(tr("Unable to retrieve playlist details"));
        this->disconnect(this, SIGNAL(playlistAddedToCache(int)), this, SLOT(addToPlaylist()));
    }

    reply->deleteLater();
}

void YouTube::deletePlaylist(const QString &playlistId) {
    m_actionId = playlistId;
    this->deletePlaylist();
}

void YouTube::deletePlaylist() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/playlists/" + m_actionId);
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistDeleted()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deletePlaylist()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::onPlaylistDeleted() {
    emit alert(tr("Playlist deleted"));

    if (this->playlistsLoaded()) {
        this->removePlaylistFromCache(m_actionId);
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onPlaylistDeleted()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(deletePlaylist()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
}

void YouTube::subscribe(const QString &userId) {
    m_actionId = userId;
    this->subscribe();
}

void YouTube::subscribe() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/subscriptions");
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<category scheme=\"http://gdata.youtube.com/schemas/2007/subscriptiontypes.cat\"\n" \
                   "term=\"channel\"/>\n" \
                   "<yt:channelId>UC" + m_actionId.toUtf8() + "</yt:channelId>\n" \
                   "</entry>");

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(subscribe()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::onSubscribed(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.namedItem("entry");

    if (this->subscriptionsLoaded()) {
        UserItem *user = new UserItem;
        user->loadYouTubeUser(entry, true, true);
        this->addNewSubscriptionToCache(QSharedPointer<UserItem>(user));
    }
    else {
        emit subscriptionChanged(m_actionId, true, entry.firstChildElement("id").text().section(':', -1));
    }

    emit alert(tr("You have subscribed to '%1'").arg(entry.firstChildElement("yt:username").attribute("display")));

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onSubscribed(QString)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(subscribe()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::unsubscribe(const QString &userId) {
    m_actionId = userId;
    this->unsubscribe();
}

void YouTube::unsubscribe() {
    QUrl url("https://gdata.youtube.com/feeds/api/users/default/subscriptions/" + m_actionId);
    this->deleteRequest(url);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(unsubscribe()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::onUnsubscribed() {
    if (this->subscriptionsLoaded()) {
        QSharedPointer<UserItem> user = this->removeSubscriptionFromCache(m_actionId);

        if (!user.isNull()) {
            emit alert(tr("You have unsubscribed to '%1'").arg(user.data()->username()));
        }
        else {
            emit subscriptionChanged(m_actionId, false, QString());
            emit alert(tr("You have unsubscribed to this channel"));
        }
    }
    else {
        emit subscriptionChanged(m_actionId, false, QString());
        emit alert(tr("You have unsubscribed to this channel"));
    }

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onUnsubscribed()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(unsubscribe()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
}

void YouTube::likeVideo(const QString &videoId) {
    m_actionId = videoId;
    this->likeVideo();
}

void YouTube::likeVideo() {
    QUrl url(QString("https://gdata.youtube.com/feeds/api/videos/%1/ratings").arg(m_actionId));
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<yt:rating value=\"like\"/>\n" \
                   "</entry>");

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoLiked()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(likeVideo()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoLiked() {
    emit alert(tr("You liked this video"));
    emit videoLiked(m_actionId);

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoLiked()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(likeVideo()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::dislikeVideo(const QString &videoId) {
    m_actionId = videoId;
    this->dislikeVideo();
}

void YouTube::dislikeVideo() {
    QUrl url(QString("https://gdata.youtube.com/feeds/api/videos/%1/ratings").arg(m_actionId));
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<yt:rating value=\"dislike\"/>\n" \
                   "</entry>");

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDisliked()));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(dislikeVideo()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::onVideoDisliked() {
    emit alert(tr("You disliked this video"));
    emit videoDisliked(m_actionId);

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onVideoDisliked()));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(dislikeVideo()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
}

void YouTube::addComment(const QVariantMap &comment) {
    m_metadataAction = comment;
    this->addComment();
}

void YouTube::addComment() {
    QUrl url(QString("https://gdata.youtube.com/feeds/api/videos/%1/comments").arg(m_metadataAction.value("videoId").toString()));
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<content>" + m_metadataAction.value("body").toString().toUtf8() + "</content>\n" \
                   "</entry>");

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addComment()));
    this->connect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::replyToComment(const QVariantMap &comment) {
    m_metadataAction = comment;
    this->replyToComment();
}

void YouTube::replyToComment() {
    QUrl url(QString("https://gdata.youtube.com/feeds/api/videos/%1/comments").arg(m_metadataAction.value("videoId").toString()));
    QByteArray xml("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n" \
                   "<entry xmlns=\"http://www.w3.org/2005/Atom\"\n" \
                   "xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n" \
                   "<link rel=\"http://gdata.youtube.com/schemas/2007#in-reply-to\"\n" \
                   "type=\"application/atom+xml\"\n" \
                   "href=\"https://gdata.youtube.com/feeds/api/videos/" + m_metadataAction.value("videoId").toString().toUtf8() + "/comments/" + m_metadataAction.value("replyId").toString().toUtf8() + "\"/>\n" \
                   "<content>" + m_metadataAction.value("body").toString().toUtf8() + "</content>\n" \
                   "</entry>");

    this->postRequest(url, xml);
    this->connect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->connect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(replyToComment()));
    this->connect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    this->connect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::onCommentAdded(const QString &response) {
    QDomDocument doc;
    doc.setContent(response);
    QDomNode entry = doc.firstChildElement("entry");
    QString videoId = entry.firstChildElement("yt:videoid").text();
    QString commentId = entry.firstChildElement("id").text().section(':', -1);
    this->getAddedComment(videoId, commentId);

    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, SLOT(onCommentAdded(QString)));
    this->disconnect(this, SIGNAL(accessTokenRefreshed(QString)), this, SLOT(addComment()));
    this->disconnect(this, SIGNAL(googleAccountLinked()), this, SLOT(addComment()));
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
}

void YouTube::getAddedComment(const QString &videoId, const QString &commentId) {
    QUrl url(QString("%1/%2/comments/%3").arg(QString(YOUTUBE_VIDEOS_BASE_URL)).arg(videoId).arg(commentId));
    url.addQueryItem("v", "2.1");
    url.addQueryItem("fields", YOUTUBE_SINGLE_COMMENT_FIELDS);
    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    QNetworkReply *reply = this->networkAccessManager()->get(request);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkAddedComment()));
}

void YouTube::checkAddedComment() {
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    int statusCode = reply->attribute(QNetworkRequest::HttpStatusCodeAttribute).toInt();

    if (statusCode == 200) {
        QDomDocument doc;
        doc.setContent(reply->readAll());
        CommentItem *comment = new CommentItem;
        comment->loadYouTubeComment(doc.namedItem("entry"));
        emit commentAdded(QSharedPointer<CommentItem>(comment));
        emit alert(tr("Your comment has been added"));
    }
    else {
        emit error(tr("New comment could not be retrieved"));
    }

    reply->deleteLater();
}

void YouTube::onVideoActionError(const QString &errorString) {
    m_actionIdList.clear();
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onVideoActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void YouTube::onPlaylistActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onPlaylistActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void YouTube::onUserActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onUserActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void YouTube::onCommentActionError(const QString &errorString) {
    emit error(errorString);
    this->setBusy(false);
    this->disconnect(this, SIGNAL(postFailed(QString)), this, SLOT(onCommentActionError(QString)));
    this->disconnect(this, SIGNAL(postSuccessful(QString)), this, 0);
}

void YouTube::getVideosFromIds(QStringList ids) {
    this->setBusy(true, QString(), 0);
    QUrl url("https://gdata.youtube.com/feeds/api/videos/batch");
    url.addQueryItem("v", "2.1");

    QByteArray xml("<feed xmlns=\"http://www.w3.org/2005/Atom\" xmlns:media=\"http://search.yahoo.com/mrss/\"\n" \
                   "xmlns:batch=\"http://schemas.google.com/gdata/batch\" xmlns:yt=\"http://gdata.youtube.com/schemas/2007\">\n"
                   "<batch:operation type=\"query\" />\n");

    int i = 0;

    while ((!ids.isEmpty()) && (i < 50)) {
        xml.append("<entry>\n" \
                   "<id>https://gdata.youtube.com/feeds/api/videos/" + ids.takeFirst().toUtf8() + "</id>\n" \
                   "</entry>\n");
        i++;
    }

    xml.append("</feed>");

    QNetworkRequest request(url);
    request.setRawHeader("User-Agent", QString("cuteTube/%1 (Qt)").arg(Utils::versionNumberString()).toUtf8());
    request.setHeader(QNetworkRequest::ContentTypeHeader, "application/atom+xml");
    request.setHeader(QNetworkRequest::ContentLengthHeader, xml.length());
    QNetworkReply *reply = this->networkAccessManager()->post(request, xml);
    this->connect(reply, SIGNAL(finished()), this, SLOT(checkFullVideos()));
}

void YouTube::checkFullVideos() {
    this->setBusy(false);
    QNetworkReply* reply = qobject_cast<QNetworkReply*>(this->sender());

    if (!reply) {
        emit error(tr("Network error"));
        return;
    }

    QDomDocument doc;
    doc.setContent(reply->readAll());
    QDomNodeList entries = doc.elementsByTagName("entry");

    if (!entries.isEmpty()) {
        QList< QSharedPointer<VideoItem> > videos;

        for (int i = 0; i < entries.size(); i++) {
            if (entries.at(i).firstChildElement("batch:status").attribute("code").toInt() == 200) {
                VideoItem *video = new VideoItem;
                video->loadYouTubeVideo(entries.at(i));
                videos.append(QSharedPointer<VideoItem>(video));
            }
        }

        emit gotVideosFromIds(videos);
    }

    reply->deleteLater();
}
