#include "videocontrols.h"
#include "utils.h"
#include "playbacklistdelegate.h"

VideoControls::VideoControls(Session *session, QMediaPlayer *player, QList< QSharedPointer<VideoItem> > playlist, QWidget *parent) :
    QWidget(parent),
    m_session(session),
    m_player(player),
    m_playlist(new PlaybackListModel(m_session->newThumbnailCache(), this)),
    m_listView(new QListView(this)),
    m_previousButton(new VideoPlayerButton(VideoPlayerButton::Previous, this)),
    m_playButton(new VideoPlayerButton(VideoPlayerButton::PlayPause, this)),
    m_nextButton(new VideoPlayerButton(VideoPlayerButton::Next, this)),
    m_backButton(new VideoPlayerButton(VideoPlayerButton::Back, this)),
    m_progressSlider(new QSlider(Qt::Horizontal, this)),
    m_bufferProgressBar(new QProgressBar(this)),
    m_positionLabel(new QLabel("00:00", this)),
    m_durationLabel(new QLabel("00:00", this)),
    m_hbox(new QHBoxLayout(this)),
    m_listTimer(new QTimer(this)),
    m_currentIndex(-1)
{
    setFixedSize(800, 70);
    setAttribute(Qt::WA_TranslucentBackground, true);
    setStyleSheet("background:transparent");

    m_listTimer->setSingleShot(true);
    m_listTimer->setInterval(3000);

    m_progressSlider->setFocusPolicy(Qt::NoFocus);
    m_previousButton->setFocusPolicy(Qt::NoFocus);
    m_playButton->setFocusPolicy(Qt::NoFocus);
    m_nextButton->setFocusPolicy(Qt::NoFocus);
    m_backButton->setFocusPolicy(Qt::NoFocus);
    m_listView->setFocusPolicy(Qt::NoFocus);

    m_playlist->addVideos(playlist);

    m_listView->setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_listView->setVerticalScrollMode(QListView::ScrollPerItem);
    m_listView->setModel(m_playlist);
    m_listView->setItemDelegate(new PlaybackListDelegate(m_listView));

    m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
    m_backButton->setIcon(QIcon::fromTheme("general_backspace"));

    m_positionLabel->setFixedSize(70, 70);
    m_durationLabel->setFixedSize(70, 70);
    m_positionLabel->setAlignment(Qt::AlignCenter);
    m_durationLabel->setAlignment(Qt::AlignCenter);

    QFont font;
    font.setPixelSize(18);

    m_positionLabel->setFont(font);
    m_durationLabel->setFont(font);

    m_bufferProgressBar->setTextVisible(true);
    m_bufferProgressBar->setFormat("Buffering");
    m_bufferProgressBar->setMinimum(0);
    m_bufferProgressBar->setMaximum(100);

    m_hbox->setContentsMargins(0, 0, 0, 0);
    m_hbox->setSpacing(0);
    m_hbox->addWidget(m_positionLabel, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_progressSlider, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_bufferProgressBar, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_listView, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_durationLabel, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_previousButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_playButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_nextButton, 0, Qt::AlignVCenter);
    m_hbox->addWidget(m_backButton, 0, Qt::AlignVCenter);

    m_bufferProgressBar->hide();
    m_listView->hide();

    connect(m_previousButton, SIGNAL(clicked()), this, SLOT(previous()));
    connect(m_playButton, SIGNAL(clicked()), this, SLOT(togglePlayPause()));
    connect(m_nextButton, SIGNAL(clicked()), this, SLOT(next()));
    connect(m_backButton, SIGNAL(clicked()), this, SIGNAL(back()));
    connect(m_progressSlider, SIGNAL(sliderReleased()), this, SLOT(setPosition()));
    connect(m_player, SIGNAL(stateChanged(QMediaPlayer::State)), this, SLOT(onStateChanged(QMediaPlayer::State)));
    connect(m_player, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(onMediaStatusChanged(QMediaPlayer::MediaStatus)));
    connect(m_player, SIGNAL(bufferStatusChanged(int)), this, SLOT(onBufferStatusChanged(int)));
    connect(m_player, SIGNAL(bufferStatusChanged(int)), m_bufferProgressBar, SLOT(setValue(int)));
    connect(m_listTimer, SIGNAL(timeout()), this, SLOT(hideListView()));
    connect(m_listView, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(m_session->urlGrabber(), SIGNAL(gotVideoUrl(QString)), this, SLOT(onUrlReady(QString)));
    connect(m_session->urlGrabber(), SIGNAL(busyProgressChanged(int)), m_bufferProgressBar, SLOT(setValue(int)));
    connect(m_session->urlGrabber(), SIGNAL(busy(QString)), this, SLOT(onVideoLoading(QString)));

    next();
}

void VideoControls::onPositionChanged(qint64 position) {
    m_positionLabel->setText(Utils::durationFromMSecs(position));
    m_progressSlider->setValue(position / 1000);
}

void VideoControls::onDurationChanged(qint64 duration) {
    m_durationLabel->setText(Utils::durationFromMSecs(duration));
    m_progressSlider->setMaximum(duration / 1000);
}

void VideoControls::onStateChanged(QMediaPlayer::State state) {
    if (state == QMediaPlayer::PlayingState) {
        m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Pause.png"));
    }
    else {
        m_playButton->setIcon(QIcon("/etc/hildon/theme/mediaplayer/Play.png"));
    }
}

void VideoControls::onMediaStatusChanged(QMediaPlayer::MediaStatus status) {
    switch (status) {
    case QMediaPlayer::InvalidMedia:
        m_session->showNoteBanner(tr("Video cannot be played"));
        break;
    case QMediaPlayer::EndOfMedia:
        next();
        break;
    default:
        return;
    }
}

void VideoControls::onVideoLoading(const QString &message) {
    m_bufferProgressBar->setFormat(message);
    m_bufferProgressBar->show();
    m_progressSlider->hide();
    m_listView->hide();
    show();
}

void VideoControls::onUrlReady(const QString &url) {
    m_bufferProgressBar->setFormat(tr("Buffering"));
    m_player->setMedia(QMediaContent(QUrl(url)));
    m_player->play();
}

void VideoControls::onBufferStatusChanged(int progress) {
    setVisible(progress < 100);
    m_bufferProgressBar->setVisible(progress < 100);
    m_progressSlider->setVisible(progress == 100);
    m_listView->hide();
}

void VideoControls::onItemClicked(const QModelIndex &index) {
    m_currentIndex = index.row() - 1;
    next();
}

void VideoControls::togglePlayPause() {
    if (m_player->state() == QMediaPlayer::PlayingState) {
        m_player->pause();
    }
    else {
        m_player->play();
    }
}

void VideoControls::previous() {
    if (m_currentIndex > 0) {
        m_currentIndex--;

        if (m_currentIndex < m_playlist->rowCount()) {
            m_listView->setCurrentIndex(m_playlist->index(m_currentIndex));
            QSharedPointer<VideoItem> video = m_playlist->get(m_currentIndex);

            if (video.data()->service() == Services::NoService) {
                m_player->setMedia(video.data()->url());
                m_player->play();
            }
            else {
                m_session->urlGrabber()->getVideoUrl(video.data()->service(), video.data()->videoId());
            }
        }
    }
    else {
        m_player->setPosition(0);
        m_player->play();
    }
}

void VideoControls::next() {
    if (m_currentIndex < m_playlist->rowCount() - 1) {
        m_currentIndex++;
        m_listView->setCurrentIndex(m_playlist->index(m_currentIndex));
        QSharedPointer<VideoItem> video = m_playlist->get(m_currentIndex);

        if (video.data()->service() == Services::NoService) {
            m_player->setMedia(video.data()->url());
            m_player->play();
        }
        else {
            m_session->urlGrabber()->getVideoUrl(video.data()->service(), video.data()->videoId());
        }
    }
    else {
        emit back();
    }
}

void VideoControls::keyPressEvent(QKeyEvent *event) {
    switch(event->key()) {
    case Qt::Key_Space:
        togglePlayPause();
        break;
    case Qt::Key_Left:
        previous();
        break;
    case Qt::Key_Right:
        next();
        break;
    case Qt::Key_Up:
        listViewUp();
        break;
    case Qt::Key_Down:
        listViewDown();
        break;
    case Qt::Key_Return:
        setPlaylistIndex();
        break;
    case Qt::Key_Enter:
        setPlaylistIndex();
        break;
    }
}

void VideoControls::setPosition() {
    qint64 pos = m_progressSlider->value() * 1000;
    m_player->setPosition(pos);
}

void VideoControls::hideListView() {
    m_listView->hide();
    m_listView->scrollTo(m_playlist->index(m_currentIndex));

    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->show();
    }
}

void VideoControls::listViewUp() {
    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->hide();
        m_listView->show();
        m_listView->scrollTo(m_listView->indexAt(QPoint(0, -5)));
        m_listTimer->start();
        show();
    }
}

void VideoControls::listViewDown() {
    if (!m_bufferProgressBar->isVisible()) {
        m_progressSlider->hide();
        m_listView->show();
        m_listView->scrollTo(m_listView->indexAt(QPoint(0, 75)));
        m_listTimer->start();
        show();
    }
}

void VideoControls::setPlaylistIndex() {
    if (m_listView->isVisible()) {
        m_currentIndex = m_listView->indexAt(QPoint(0, 0)).row() - 1;
        next();
    }
}

void VideoControls::showEvent(QShowEvent *event) {
    Q_UNUSED(event);

    onPositionChanged(m_player->position());
    onDurationChanged(m_player->duration());

    connect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    connect(m_player, SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
}

void VideoControls::hideEvent(QHideEvent *event) {
    Q_UNUSED(event);

    disconnect(m_player, SIGNAL(positionChanged(qint64)), this, SLOT(onPositionChanged(qint64)));
    disconnect(m_player, SIGNAL(durationChanged(qint64)), this, SLOT(onDurationChanged(qint64)));
}
