#include "socialvideolistwindow.h"
#include "youtubevideoinfowindow.h"
#include "dailymotionvideoinfowindow.h"
#include "videolistdelegate.h"
#include "videoselectionwindow.h"
#include "videoplaybackwindow.h"
#include <QMenuBar>

SocialVideoListWindow::SocialVideoListWindow(Session *session, QWidget *parent) :
    StackedWindow(session, parent),
    m_listView(new ListView(this)),
    m_model(new SocialVideoListModel(m_session, this)),
    m_refreshAction(menuBar()->addAction(tr("Refresh"), this, SLOT(refresh()))),
    m_addToQueueAction(menuBar()->addAction(tr("Add to playback queue"), this, SLOT(queueVideos()))),
    m_videoDownloadAction(menuBar()->addAction(tr("Download"), this, SLOT(addVideoDownloads()))),
    m_audioDownloadAction(menuBar()->addAction(tr("Download (audio)"), this, SLOT(addAudioDownloads()))),
    m_contextMenu(new QMenu(this)),
    m_addToQueueContextAction(m_contextMenu->addAction(tr("Add to playback queue"), this, SLOT(queueVideo()))),
    m_videoDownloadContextAction(m_contextMenu->addAction(tr("Download"), this, SLOT(addVideoDownload()))),
    m_audioDownloadContextAction(m_contextMenu->addAction(tr("Download (audio)"), this, SLOT(addAudioDownload()))),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this))
{
    setWindowTitle(tr("Social"));
    setCentralWidget(m_listView);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);
    m_noResultsLabel->hide();

    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_session->sharing(), SIGNAL(videosAdded()), m_listView, SLOT(positionAtBeginning()));
    connect(m_session->youtube(), SIGNAL(busyChanged()), this, SLOT(onLoadingChanged()));
    connect(m_session->dailymotion(), SIGNAL(busyChanged()), this, SLOT(onLoadingChanged()));
    connect(m_session->sharing(), SIGNAL(busyChanged()), this, SLOT(onLoadingChanged()));

    onOrientationChanged();
}

void SocialVideoListWindow::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void SocialVideoListWindow::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void SocialVideoListWindow::onLoadingChanged() {
    bool busy = (m_session->youtube()->busy()) || (m_session->dailymotion()->busy()) || (m_session->sharing()->busy());
    toggleBusy(busy);
    m_noResultsLabel->setVisible((!busy) && (m_model->rowCount() == 0));
}

void SocialVideoListWindow::refresh() {
    m_session->sharing()->getVideos(true);
}

void SocialVideoListWindow::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::NoService, tr("Download"), SLOT(addVideoDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::NoService, tr("Download"), SLOT(addAudioDownloads()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::NoService, tr("Add to queue"), SLOT(queueVideos()), this);
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void SocialVideoListWindow::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToQueueContextAction);
}

void SocialVideoListWindow::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void SocialVideoListWindow::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void SocialVideoListWindow::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}

void SocialVideoListWindow::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void SocialVideoListWindow::showVideoInfo(QSharedPointer<VideoItem> video) {
    if (video->service() == Services::YouTube) {
        YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, this);
        window->show();
    }
    else {
        DailymotionVideoInfoWindow *window = new DailymotionVideoInfoWindow(m_session, video, this);
        window->show();
    }
}

void SocialVideoListWindow::playVideo(const QModelIndex &index) {
    if (m_session->settings()->mediaPlayer() == "cutetube") {
        VideoPlaybackWindow *window = new VideoPlaybackWindow(m_session, QList< QSharedPointer<VideoItem> >() << m_model->get(index.row()), this);
        window->show();
    }
    else {
        m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
    }
}
