#include "youtuberelatedvideostab.h"
#include "youtubeplaylistdialog.h"
#include "videolistdelegate.h"
#include "youtubevideoinfowindow.h"
#include "videoselectionwindow.h"

YouTubeRelatedVideosTab::YouTubeRelatedVideosTab(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractTab(parent),
    m_session(session),
    m_video(video),
    m_model(new YouTubeVideoListModel(m_session, this)),
    m_listView(new ListView(this)),
    m_addToFavouritesAction(new QAction(tr("Add to favourites"), this)),
    m_addToPlaylistAction(new QAction(tr("Add to playlist"), this)),
    m_watchLaterAction(new QAction(tr("Watch later"), this)),
    m_addToQueueAction(new QAction(tr("Add to playback queue"), this)),
    m_videoDownloadAction(new QAction(tr("Download"), this)),
    m_audioDownloadAction(new QAction(tr("Download (audio)"), this)),
    m_noResultsLabel(new QLabel(QString("<p style='color:#4d4d4d;font-size:40px;'><center><b>%1</b></center></p>").arg(tr("No videos found")), this)),
    m_grid(new QGridLayout(this))
{
    addAction(m_addToFavouritesAction);
    addAction(m_addToPlaylistAction);
    addAction(m_watchLaterAction);
    addAction(m_addToQueueAction);
    addAction(m_videoDownloadAction);
    addAction(m_audioDownloadAction);

    m_noResultsLabel->hide();
    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->addWidget(m_listView, 0, 0);

    VideoListDelegate *delegate = new VideoListDelegate(m_listView);

    m_listView->setModel(m_model);
    m_listView->setItemDelegate(delegate);

    connect(m_model, SIGNAL(loadingChanged()), this, SLOT(onLoadingChanged()));
    connect(delegate, SIGNAL(clicked(QModelIndex)), this, SLOT(onItemClicked(QModelIndex)));
    connect(delegate, SIGNAL(play(QModelIndex)), this, SLOT(playVideo(QModelIndex)));
    connect(m_listView, SIGNAL(atEnd()), m_model, SLOT(getMoreVideos()));
    connect(m_listView, SIGNAL(customContextMenuRequested(QPoint)), this, SLOT(showContextMenu(QPoint)));
    connect(m_addToFavouritesAction, SIGNAL(triggered()), this, SLOT(addVideosToFavourites()));
    connect(m_addToPlaylistAction, SIGNAL(triggered()), this, SLOT(addVideosToPlaylist()));
    connect(m_watchLaterAction, SIGNAL(triggered()), this, SLOT(addVideosToWatchLaterPlaylist()));
    connect(m_addToQueueAction, SIGNAL(triggered()), this, SLOT(queueVideos()));
    connect(m_videoDownloadAction, SIGNAL(triggered()), this, SLOT(addVideoDownloads()));
    connect(m_audioDownloadAction, SIGNAL(triggered()), this, SLOT(addAudioDownloads()));

    m_model->getRelatedVideos(m_video.data()->videoId());

    setMenuActions();
    onOrientationChanged();
}

void YouTubeRelatedVideosTab::setLandscapeLayout() {
    m_noResultsLabel->setFixedSize(800, 400);
}

void YouTubeRelatedVideosTab::setPortraitLayout() {
    m_noResultsLabel->setFixedSize(480, 700);
}

void YouTubeRelatedVideosTab::setMenuActions() {
    bool signedIn = m_session->youtube()->userSignedIn();

    m_addToFavouritesAction->setEnabled(signedIn);
    m_addToPlaylistAction->setEnabled(signedIn);
    m_watchLaterAction->setEnabled(signedIn);
    m_addToFavouritesContextAction->setVisible(signedIn);
    m_addToPlaylistContextAction->setVisible(signedIn);
    m_watchLaterContextAction->setVisible(signedIn);
}

void YouTubeRelatedVideosTab::onLoadingChanged() {
    m_session->toggleBusy(m_model->loading());
    m_noResultsLabel->setVisible((!m_model->loading()) && (m_model->rowCount() == 0));
}

void YouTubeRelatedVideosTab::onItemClicked(const QModelIndex &index) {
    if (index.isValid()) {
        showVideoInfo(m_model->get(index.row()));
    }
}

void YouTubeRelatedVideosTab::showVideoInfo(QSharedPointer<VideoItem> video) {
    YouTubeVideoInfoWindow *window = new YouTubeVideoInfoWindow(m_session, video, m_session->currentWindow());
    window->show();
}

void YouTubeRelatedVideosTab::playVideo(const QModelIndex &index) {
    m_session->urlGrabber()->getYouTubeVideoUrl(index.data(AbstractVideoListModel::VideoIdRole).toString());
}

void YouTubeRelatedVideosTab::addVideosToFavourites() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to favourites"), SLOT(addVideosToFavourites()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideosToPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to playlist"), SLOT(showPlaylistDialog()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideosToWatchLaterPlaylist() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Watch later"), SLOT(addVideosToWatchLaterPlaylist()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addVideoDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addVideoDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::addAudioDownloads() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Download"), SLOT(addAudioDownloads()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::queueVideos() {
    VideoSelectionWindow *window = new VideoSelectionWindow(m_session, m_model, Services::YouTube, tr("Add to queue"), SLOT(queueVideos()), m_session->currentWindow());
    window->positionAtIndex(m_listView->indexAt(QPoint(0, 0)));
    window->show();
}

void YouTubeRelatedVideosTab::showContextMenu(const QPoint &pos) {
    m_contextMenu->popup(pos, m_addToFavouritesContextAction);
}

void YouTubeRelatedVideosTab::addVideoToFavourites() {
    m_session->youtube()->addToFavourites(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeRelatedVideosTab::showPlaylistDialog() {
    YouTubePlaylistDialog *dialog = new YouTubePlaylistDialog(m_session, this);
    dialog->open();

    connect(dialog, SIGNAL(playlistChosen(QString)), this, SLOT(addVideoToPlaylist(QString)));
    connect(dialog, SIGNAL(addToNewPlaylist(NewPlaylist)), this, SLOT(addVideoToNewPlaylist(NewPlaylist)));
}

void YouTubeRelatedVideosTab::addVideoToPlaylist(const QString &id) {
    m_session->youtube()->addToPlaylist(m_model->get(m_listView->currentIndex().row()), id);
}

void YouTubeRelatedVideosTab::addVideoToNewPlaylist(const NewPlaylist &playlist) {
    m_session->youtube()->createPlaylist(playlist, m_model->get(m_listView->currentIndex().row()));
}

void YouTubeRelatedVideosTab::addVideoToWatchLaterPlaylist() {
    m_session->youtube()->addToWatchLaterPlaylist(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeRelatedVideosTab::addVideoDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()));
}

void YouTubeRelatedVideosTab::addAudioDownload() {
    m_session->transferManager()->addDownloadTransfer(m_model->get(m_listView->currentIndex().row()), true);
}

void YouTubeRelatedVideosTab::queueVideo() {
    m_session->playbackQueue()->addVideo(m_model->get(m_listView->currentIndex().row()));
}
