#include "youtubemetadatadialog.h"
#include <QLabel>

YouTubeMetadataDialog::YouTubeMetadataDialog(Session *session, QSharedPointer<VideoItem> video, QWidget *parent) :
    AbstractDialog(parent),
    m_session(session),
    m_video(video),
    m_titleEdit(new QLineEdit(this)),
    m_descriptionEdit(new QTextEdit(this)),
    m_tagsEdit(new QLineEdit(this)),
    m_categorySelector(new YouTubeUploadCategorySelector(tr("Category"), this)),
    m_responsesSelector(new YouTubePermissionSelector(tr("Allow video repsonses?"), this)),
    m_commentsSelector(new YouTubePermissionSelector(tr("Allow comments?"), this)),
    m_commentVoteCheckbox(new QCheckBox(tr("Allow voting on comments?"), this)),
    m_ratingsCheckbox(new QCheckBox(tr("Allow ratings?"), this)),
    m_privateCheckbox(new QCheckBox(tr("Private?"), this)),
    m_listingsCheckbox(new QCheckBox(tr("Appear in listings?"), this)),
    m_syndicationCheckbox(new QCheckBox(tr("Syndicate to mobile devices?"), this)),
    m_autoPlayCheckbox(new QCheckBox(tr("Allow automatic playback?"), this)),
    m_embedCheckbox(new QCheckBox(tr("Allow embedding?"), this)),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_doneButton(m_buttonBox->addButton(tr("Done"), QDialogButtonBox::AcceptRole)),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Edit metadata"));

    QLabel *titleLabel = new QLabel(tr("Title"), this);
    QLabel *descriptionLabel = new QLabel(tr("Description"), this);
    QLabel *tagsLabel = new QLabel(tr("Tags"), this);
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(titleLabel);
    vbox->addWidget(m_titleEdit);
    vbox->addWidget(descriptionLabel);
    vbox->addWidget(m_descriptionEdit);
    vbox->addWidget(tagsLabel);
    vbox->addWidget(m_tagsEdit);
    vbox->addWidget(m_categorySelector);
    vbox->addWidget(m_responsesSelector);
    vbox->addWidget(m_commentsSelector);
    vbox->addWidget(m_commentVoteCheckbox);
    vbox->addWidget(m_ratingsCheckbox);
    vbox->addWidget(m_autoPlayCheckbox);
    vbox->addWidget(m_privateCheckbox);
    vbox->addWidget(m_listingsCheckbox);
    vbox->addWidget(m_syndicationCheckbox);
    vbox->addWidget(m_embedCheckbox);

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_doneButton->setEnabled(false);
    m_descriptionEdit->setAcceptRichText(false);

    connect(m_titleEdit, SIGNAL(textChanged(QString)), this, SLOT(onTitleTextChanged(QString)));
    connect(m_privateCheckbox, SIGNAL(toggled(bool)), this, SLOT(onPrivateToggled(bool)));
    connect(m_buttonBox, SIGNAL(accepted()), this, SLOT(submitMetadata()));
    connect(m_session->youtube(), SIGNAL(gotVideoMetadata(VideoMetadata)), this, SLOT(setMetadata(VideoMetadata)));

    m_session->youtube()->getVideoMetadata(m_video.data()->videoId());

    toggleBusy(true);
    onOrientationChanged();
}

void YouTubeMetadataDialog::setLandscapeLayout() {
    setFixedHeight(340);
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
}

void YouTubeMetadataDialog::setPortraitLayout() {
    setFixedHeight(600);
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
}

void YouTubeMetadataDialog::onTitleTextChanged(const QString &text) {
    m_doneButton->setEnabled(!text.isEmpty());
}

void YouTubeMetadataDialog::onPrivateToggled(bool isPrivate) {
    m_listingsCheckbox->setEnabled(!isPrivate);
    m_syndicationCheckbox->setEnabled(!isPrivate);
    m_embedCheckbox->setEnabled(!isPrivate);

    if (isPrivate) {
        m_listingsCheckbox->setChecked(false);
        m_syndicationCheckbox->setChecked(false);
        m_embedCheckbox->setChecked(false);
    }
}

void YouTubeMetadataDialog::setMetadata(const VideoMetadata &metadata) {
    toggleBusy(false);

    m_titleEdit->setText(metadata.title());
    m_descriptionEdit->setText(metadata.description());
    m_tagsEdit->setText(metadata.tags());
    m_categorySelector->setValue(metadata.category());
    m_responsesSelector->setValue(metadata.responsesPermission());
    m_commentsSelector->setValue(metadata.commentsPermission());
    m_commentVoteCheckbox->setEnabled(metadata.commentsPermission() != "denied");
    m_commentVoteCheckbox->setChecked(metadata.commentVotingPermission() == "allowed");
    m_ratingsCheckbox->setChecked(metadata.ratingsPermission() == "allowed");
    m_privateCheckbox->setChecked(metadata.isPrivate());
    m_listingsCheckbox->setEnabled(!metadata.isPrivate());
    m_listingsCheckbox->setChecked(metadata.listingsPermission() == "allowed");
    m_syndicationCheckbox->setEnabled(!metadata.isPrivate());
    m_syndicationCheckbox->setChecked(metadata.syndicationPermission() == "allowed");
    m_autoPlayCheckbox->setChecked(metadata.autoPlayPermission() == "allowed");
    m_embedCheckbox->setEnabled(!metadata.isPrivate());
    m_embedCheckbox->setChecked(metadata.syndicationPermission() == "allowed");

    disconnect(m_session->youtube(), SIGNAL(gotVideoMetadata(VideoMetadata)), this, SLOT(setMetadata(VideoMetadata)));
}

void YouTubeMetadataDialog::submitMetadata() {
    VideoMetadata metadata;
    metadata.setTitle(m_titleEdit->text());
    metadata.setDescription(m_descriptionEdit->toPlainText());
    metadata.setTags(m_tagsEdit->text());
    metadata.setCategory(m_categorySelector->currentValue());
    metadata.setResponsesPermission(m_responsesSelector->currentValue());
    metadata.setCommentsPermission(m_commentsSelector->currentValue());
    metadata.setCommentVotingPermission(m_commentVoteCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setRatingsPermission(m_ratingsCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setListingsPermission(m_listingsCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setSyndicationPermission(m_syndicationCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setAutoPlayPermission(m_autoPlayCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setEmbedPermission(m_embedCheckbox->isChecked() ? "allowed" : "denied");
    metadata.setPrivate(m_privateCheckbox->isChecked());

    m_session->youtube()->updateVideoMetadata(m_video, metadata);
    accept();
}
