#include "searchdialog.h"
#include "searchhistorydialog.h"
#include <QClipboard>
#include <QRegExp>

SearchDialog::SearchDialog(Settings *settings, QWidget *parent) :
    AbstractDialog(parent),
    m_settings(settings),
    m_searchEdit(new QLineEdit(this)),
    m_searchServiceSelector(m_settings->showSearchSite() ? new VideoServiceSelector(tr("Service"), this) : 0),
    m_searchTypeSelector(m_settings->showSearchType() ? new SearchTypeSelector(tr("Search for"), this) : 0),
    m_searchOrderSelector(m_settings->showSearchOrder() ? new SearchOrderSelector(tr("Order by"), this) : 0),
    m_searchTimeSelector(m_settings->showSearchTime() ? new SearchTimeSelector(tr("Show results from"), this) : 0),
    m_searchDurationSelector(m_settings->showSearchDuration() ? new SearchDurationSelector(tr("Show videos of duration"), this) : 0),
    m_searchLanguageSelector(m_settings->showSearchLanguage() ? new LanguageSelector(tr("Show videos with language"), false, this) : 0),
    m_buttonBox(new QDialogButtonBox(Qt::Vertical, this)),
    m_historyButton(m_buttonBox->addButton(tr("History"), QDialogButtonBox::ActionRole)),
    m_searchButton(m_buttonBox->addButton(clipboardUrl().isEmpty() ? tr("Search") : tr("From URL"), QDialogButtonBox::ActionRole)),
    m_scrollArea(new QScrollArea(this)),
    m_model(new SearchListModel(this)),
    m_historyDialog(0),
    m_grid(new QGridLayout(this))
{
    setWindowTitle(tr("Search"));
    m_searchEdit->setPlaceholderText(tr("Search query"));

    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QVBoxLayout *vbox = new QVBoxLayout(scrollWidget);

    vbox->addWidget(m_searchEdit);

    if (m_searchServiceSelector) {
        connect(m_searchServiceSelector, SIGNAL(valueChanged(Services::VideoService)), m_settings, SLOT(setSearchSite(Services::VideoService)));
        connect(m_settings, SIGNAL(searchSiteChanged(Services::VideoService)), this, SLOT(onServiceChanged(Services::VideoService)));
        m_searchServiceSelector->showExtras(true);
        m_searchServiceSelector->showXTube(m_settings->xtubeEnabled());
        m_searchServiceSelector->setValue(m_settings->searchSite());
        vbox->addWidget(m_searchServiceSelector);
    }
    if (m_searchTypeSelector) {
        connect(m_searchTypeSelector, SIGNAL(valueChanged(Queries::QueryType)), m_settings, SLOT(setSearchType(Queries::QueryType)));
        connect(m_settings, SIGNAL(searchTypeChanged(Queries::QueryType)), this, SLOT(onSearchTypeChanged(Queries::QueryType)));
        m_searchTypeSelector->setValue(m_settings->searchType());
        vbox->addWidget(m_searchTypeSelector);
    }
    if (m_searchOrderSelector) {
        connect(m_searchOrderSelector, SIGNAL(valueChanged(Queries::QueryOrder)), m_settings, SLOT(setSearchOrder(Queries::QueryOrder)));
        m_searchOrderSelector->setValue(m_settings->searchOrder());
        vbox->addWidget(m_searchOrderSelector);
    }
    if (m_searchTimeSelector) {
        connect(m_searchTimeSelector, SIGNAL(valueChanged(Queries::TimeFilter)), m_settings, SLOT(setSearchTime(Queries::TimeFilter)));
        m_searchTimeSelector->setValue(m_settings->searchTime());
        vbox->addWidget(m_searchTimeSelector);
    }
    if (m_searchDurationSelector) {
        connect(m_searchDurationSelector, SIGNAL(valueChanged(Queries::DurationFilter)), m_settings, SLOT(setSearchDuration(Queries::DurationFilter)));
        m_searchDurationSelector->setValue(m_settings->searchDuration());
        vbox->addWidget(m_searchDurationSelector);
    }    
    if (m_searchLanguageSelector) {
        connect(m_searchLanguageSelector, SIGNAL(valueChanged(QString)), m_settings, SLOT(setLanguageFilter(QString)));
        m_searchLanguageSelector->setValue(m_settings->languageFilter());
        vbox->addWidget(m_searchLanguageSelector);
    }

    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);

    connect(m_searchEdit, SIGNAL(returnPressed()), this, SLOT(parseSearchQuery()));
    connect(m_searchEdit, SIGNAL(textChanged(QString)), this, SLOT(onSearchTextChanged(QString)));
    connect(m_historyButton, SIGNAL(clicked()), this, SLOT(showSearchHistory()));
    connect(m_searchButton, SIGNAL(clicked()), this, SLOT(parseSearchQuery()));

    onOrientationChanged();
}

void SearchDialog::onServiceChanged(Services::VideoService service) {
    if (m_searchTypeSelector) {
        m_searchTypeSelector->showGroups(service == Services::Dailymotion);
        m_searchTypeSelector->setEnabled((service == Services::YouTube) || (service == Services::Dailymotion));

        if (((m_settings->searchType() == Queries::Groups) && (service != Services::Dailymotion)) || ((service != Services::YouTube) && (service != Services::Dailymotion))) {
            m_searchTypeSelector->setValue(Queries::Videos);
        }
    }
    if (m_searchTimeSelector) {
        m_searchTimeSelector->setEnabled((service == Services::YouTube) && (m_settings->searchType() == Queries::Videos));
    }
    if (m_searchDurationSelector) {
        m_searchDurationSelector->setEnabled((service == Services::YouTube) && (m_settings->searchType() == Queries::Videos));
    }
    if (m_searchLanguageSelector) {
        m_searchLanguageSelector->setEnabled(((service == Services::YouTube) || (service == Services::Dailymotion)) && (m_settings->searchType() == Queries::Videos));
    }
}

void SearchDialog::onSearchTypeChanged(Queries::QueryType type) {
    if (m_searchOrderSelector) {
        m_searchOrderSelector->setEnabled(type == Queries::Videos);
    }
    if (m_searchTimeSelector) {
        m_searchTimeSelector->setEnabled((m_settings->searchSite() == Services::YouTube) && (type == Queries::Videos));
    }
    if (m_searchDurationSelector) {
        m_searchDurationSelector->setEnabled((m_settings->searchSite() == Services::YouTube) && (type == Queries::Videos));
    }
    if (m_searchLanguageSelector) {
        m_searchLanguageSelector->setEnabled(type == Queries::Videos);
    }
}

void SearchDialog::setLandscapeLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 0, 1, Qt::AlignBottom);
    }

    setFixedHeight(340);
}

void SearchDialog::setPortraitLayout() {
    m_buttonBox->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);

    if (m_grid->isEmpty()) {
        m_grid->addWidget(m_scrollArea, 0, 0);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }
    else {
        m_grid->removeWidget(m_buttonBox);
        m_grid->addWidget(m_buttonBox, 1, 0, Qt::AlignBottom);
    }

    setFixedHeight(qMin(600, m_scrollArea->widget()->height() + 160));
}

void SearchDialog::showSearchHistory() {
    if (!m_historyDialog) {
        m_historyDialog = new SearchHistoryDialog(m_model, this);
        connect(m_historyDialog, SIGNAL(queryChosen(QString)), m_searchEdit, SLOT(setText(QString)));
    }

    m_historyDialog->open();
}

void SearchDialog::onSearchTextChanged(const QString &text) {
    if ((text.isEmpty()) && (!clipboardUrl().isEmpty())) {
        m_searchButton->setText(tr("From URL"));
    }
    else {
        m_searchButton->setText(tr("Search"));
    }
}

void SearchDialog::parseSearchQuery() {
    if ((!clipboardUrl().isEmpty()) && (m_searchEdit->text().isEmpty())) {
        emit goToVideo(clipboardUrl());
    }
    else {
        m_model->addSearch(m_searchEdit->text());
        emit search(m_searchEdit->text());
    }

    accept();
}

QString SearchDialog::clipboardUrl() const {
    return QApplication::clipboard()->text().contains(QRegExp("(youtube.com/watch|youtu.be|dailymotion.com/video|dai.ly|vimeo.com)")) ? QApplication::clipboard()->text() : QString();
}
