#include "abstractvideolistmodel.h"

AbstractVideoListModel::AbstractVideoListModel(ThumbnailCache *cache, QObject *parent) :
    QAbstractListModel(parent),
    m_cache(cache)
{
    connect(m_cache, SIGNAL(thumbnailReady()), this, SLOT(onThumbnailReady()));
}

AbstractVideoListModel::~AbstractVideoListModel() {
    m_list.clear();
    delete m_cache;
}

int AbstractVideoListModel::rowCount(const QModelIndex &parent) const {
    Q_UNUSED(parent);

    return m_list.size();
}

QVariant AbstractVideoListModel::data(const QModelIndex &index, int role) const {
    switch (role) {
    case IdRole:
        return m_list.at(index.row()).data()->id();
    case VideoIdRole:
        return m_list.at(index.row()).data()->videoId();
    case TitleRole:
        return m_list.at(index.row()).data()->title();
    case ArtistRole:
        return m_list.at(index.row()).data()->artist();
    case DateRole:
        return m_list.at(index.row()).data()->date();
    case DurationRole:
        return m_list.at(index.row()).data()->duration();
    case ViewCountRole:
        return m_list.at(index.row()).data()->viewCount();
    case RatingCountRole:
        return m_list.at(index.row()).data()->ratingCount();
    case RatingRole:
        return m_list.at(index.row()).data()->rating();
    case ThumbnailRole:
        return m_cache->thumbnail(m_list.at(index.row()).data()->thumbnailUrl(), QSize(160, 120));
    case ServiceRole:
        return m_list.at(index.row()).data()->service();
    case FavouriteRole:
        return m_list.at(index.row()).data()->favourite();
    case CheckedRole:
        return m_list.at(index.row()).data()->checked();
    default:
        return QVariant();
    }
}

QSharedPointer<VideoItem> AbstractVideoListModel::get(int row) const {
    if ((row >= 0) && (row < m_list.size())) {
        return QSharedPointer<VideoItem>(m_list.at(row));
    }

    return QSharedPointer<VideoItem>();
}

void AbstractVideoListModel::insertVideo(int row, QSharedPointer<VideoItem> video) {
    beginInsertRows(QModelIndex(), row, row);
    m_list.insert(row, video);
    endInsertRows();
}

void AbstractVideoListModel::appendVideo(QSharedPointer<VideoItem> video) {
    beginInsertRows(QModelIndex(), rowCount(), rowCount());
    m_list.append(video);
    endInsertRows();
}

bool AbstractVideoListModel::removeVideo(const QString &id) {
    bool removed = false;
    int i = 0;

    while ((!removed) && (i < m_list.size())) {
        removed = (get(i).data()->id() == id);

        if (removed) {
            removeRow(i);
        }

        i++;
    }

    return removed;
}

bool AbstractVideoListModel::removeRow(int row, const QModelIndex &parent)
{
    Q_UNUSED(parent);

    if ((row >= 0) && (row < m_list.size())) {
        beginRemoveRows(QModelIndex(), row, row);
        m_list.takeAt(row).clear();
        endRemoveRows();

        return true;
    }

    return false;
}

void AbstractVideoListModel::toggleChecked(int row) {
    QSharedPointer<VideoItem> video = get(row);

    if (!video.isNull()) {

        video.data()->toggleChecked();

        if (video.data()->checked()) {
            m_checkedRows.append(row);
        }
        else {
            m_checkedRows.removeOne(row);
        }

        emit dataChanged(index(row), index(row));
    }
}

void AbstractVideoListModel::checkAll() {
    for (int i = 0; i < m_list.size(); i++) {
        get(i).data()->setChecked(true);
        m_checkedRows.append(i);
    }

    emit dataChanged(index(0), index(m_list.size() - 1));
}

void AbstractVideoListModel::checkNone() {
    for (int i = 0; i < m_list.size(); i++) {
        get(i).data()->setChecked(false);
    }

    m_checkedRows.clear();

    if (!m_list.isEmpty()) {
        emit dataChanged(index(0), index(m_list.size() - 1));
    }
}

QList< QSharedPointer<VideoItem> > AbstractVideoListModel::checkedItems() const {
    QList< QSharedPointer<VideoItem> > checked;

    for (int i = 0; i < m_checkedRows.size(); i++) {
        checked.append(get(m_checkedRows.at(i)));
    }

    return checked;
}

void AbstractVideoListModel::onThumbnailReady() {
    emit dataChanged(index(0), index(rowCount() - 1));
}
