#include "connection.h"
#include <QNetworkAccessManager>
#include <QNetworkReply>
#include <QDir>

static const qint64 BUFFER_SIZE = 1024 * 512;

Connection::Connection(QNetworkAccessManager *manager, QObject *parent) :
    QObject(parent),
    m_nam(manager),
    m_reply(0),
    m_uploadFile(0),
    m_transferType(Transfers::Download),
    m_start(0),
    m_end(0),
    m_transferredBytes(0),
    m_status(Transfers::Paused)
{
}

Connection::~Connection() {
    if (m_reply) {
        delete m_reply;
        m_reply = 0;
    }
}

Transfers::TransferType Connection::transferType() const {
    return m_transferType;
}

void Connection::setTransferType(Transfers::TransferType type) {
    m_transferType = type;
}

QString Connection::uploadFile() const {
    return !m_uploadFile ? QString() : m_uploadFile->fileName();
}

void Connection::setUploadFile(const QString &fileName) {
    if (!m_uploadFile) {
        m_uploadFile = new QFile(fileName, this);
    }
    else if (!m_uploadFile->isOpen()) {
        m_uploadFile->setFileName(fileName);
    }
}

QNetworkRequest Connection::request() const {
    return m_request;
}

void Connection::setRequest(const QNetworkRequest &request) {
    m_request = request;
}

QByteArray Connection::header(const QByteArray &headerName) const {
    return m_request.rawHeader(headerName);
}

void Connection::setHeader(const QByteArray &headerName, const QByteArray &value) {
    m_request.setRawHeader(headerName, value);
}

qint64 Connection::contentRangeStart() const {
    return m_start;
}

void Connection::setContentRangeStart(qint64 start) {
    m_start = start;
}

qint64 Connection::contentRangeEnd() const {
    return m_end;
}

void Connection::setContentRangeEnd(qint64 end) {
    m_end = end;
}

void Connection::setContentRange(qint64 start, qint64 end) {
    this->setContentRangeStart(start);
    this->setContentRangeEnd(end);
}

qint64 Connection::position() const {
    return m_start + m_transferredBytes;
}

Transfers::Status Connection::status() const {
    return m_status;
}

void Connection::setStatus(Transfers::Status status) {
    if (status != this->status()) {
        m_status = status;
        emit statusChanged(status);
    }
}

QString Connection::errorString() const {
    return m_errorString;
}

void Connection::setErrorString(const QString &errorString) {
    m_errorString = errorString;
}

void Connection::start() {
    switch (this->transferType()) {
    case Transfers::Upload:
        this->performUpload();
        return;
    default:
        this->performDownload();
        return;
    }
}

void Connection::performDownload(const QUrl &url) {
    if (this->position() > 0) {
        this->setHeader("Range", "bytes=" + QByteArray::number(this->position()) + "-");
    }

    this->setStatus(Transfers::Downloading);
    QNetworkRequest request = this->request();

    if (!url.isEmpty()) {
        request.setUrl(url);
    }

    qDebug() << "Downloading:" << request.url();

    m_reply = m_nam->get(request);
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onFinished()));

    if (this->contentRangeEnd() <= 0) {
        this->connect(m_reply, SIGNAL(metaDataChanged()), this, SLOT(onMetaDataChanged()));
    }
    else {
        this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onReadyRead()));
    }
}

void Connection::performUpload(const QUrl &url) {
    if ((!m_uploadFile) || (!m_uploadFile->open(QIODevice::ReadOnly))) {
        this->setErrorString(tr("Cannot open file"));
        this->setStatus(Transfers::Failed);
        return;
    }

    if (this->position() > 0) {
        this->setHeader("Range", "bytes=" + QByteArray::number(this->position()) + "-");
    }

    this->setHeader("Content-Length", QByteArray::number(m_uploadFile->size() - this->position()));
    this->setStatus(Transfers::Uploading);
    QNetworkRequest request = this->request();

    if (!url.isEmpty()) {
        request.setUrl(url);
    }

    qDebug() << "Uploading:" << m_uploadFile->fileName();

    MetaInfo info;
    info.size = m_uploadFile->size();
    info.bytesRemaining = m_uploadFile->size();
    info.name = m_uploadFile->fileName();

    emit metaInfoReady(info);

    m_reply = m_nam->post(request, m_uploadFile);
    this->connect(m_reply, SIGNAL(uploadProgress(qint64,qint64)), this, SLOT(onUploadProgressChanged(qint64)));
    this->connect(m_reply, SIGNAL(finished()), this, SLOT(onFinished()));
}

void Connection::pause() {
    if (m_reply) {
        m_reply->abort();
    }

    this->setStatus(Transfers::Paused);
}

void Connection::cancel() {
    if (m_reply) {
        m_reply->abort();
    }

    this->setStatus(Transfers::Cancelled);
}

void Connection::processData() {
    if (m_reply) {
        this->connect(m_reply, SIGNAL(readyRead()), this, SLOT(onReadyRead()));
    }
}

void Connection::onMetaDataChanged() {
    if (m_reply) {
        qint64 size = m_reply->header(QNetworkRequest::ContentLengthHeader).toLongLong();

        if (size <= 0) {
            size = m_reply->rawHeader("Content-Length").toLongLong();
        }

        if (size <= 0) {
            QString redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toString();

            if (!redirect.isEmpty()) {
                return;
            }

            redirect = m_reply->header(QNetworkRequest::LocationHeader).toString();

            if (!redirect.isEmpty()) {
                return;
            }
        }

        qDebug() << "Reported size:" << size;
        this->setContentRangeEnd(size);

        QString fileName = QString(m_reply->rawHeader("Content-Disposition")).section("filename=", -1).section(';', 0, 0).remove(QRegExp("\'|\""));

        MetaInfo info;
        info.size = size;
        info.bytesRemaining = size;
        info.name = fileName;

        emit metaInfoReady(info);
    }
}

void Connection::onReadyRead() {
    if (m_reply) {
        if (this->contentRangeEnd() > 0) {
            qint64 maxBytes = qMin<qint64>(this->contentRangeEnd() - (this->position() + m_buffer.size()), m_reply->bytesAvailable());

            m_buffer += m_reply->read(maxBytes);
            qint64 bufferSize = qint64(m_buffer.size());

            if (bufferSize > BUFFER_SIZE) {
                emit bytesTransferred(bufferSize);
                emit dataAvailable(this->position(), m_buffer);

                m_transferredBytes += bufferSize;
                m_buffer.clear();
            }

            if (this->position() >= this->contentRangeEnd()) {
                m_reply->abort();
            }
        }
        else {
            qint64 maxBytes = m_reply->bytesAvailable();

            m_buffer += m_reply->read(maxBytes);
            qint64 bufferSize = qint64(m_buffer.size());

            if (bufferSize > BUFFER_SIZE) {
                emit bytesTransferred(bufferSize);
                emit dataAvailable(this->position(), m_buffer);

                m_transferredBytes += bufferSize;
                m_buffer.clear();
            }
        }
    }
}

void Connection::onUploadProgressChanged(qint64 bytes) {
    emit bytesTransferred(bytes - m_transferredBytes);
    m_transferredBytes = bytes;
}

void Connection::onFinished() {
    if (!m_reply) {
        this->setErrorString(tr("Network error"));
        this->setStatus(Transfers::Failed);
        return;
    }

    switch (this->transferType()) {
    case Transfers::Download:
    {
        QUrl redirect = m_reply->attribute(QNetworkRequest::RedirectionTargetAttribute).toUrl();

        if (redirect.isEmpty()) {
            redirect = m_reply->header(QNetworkRequest::LocationHeader).toUrl();
        }

        if (!redirect.isEmpty()) {
            m_reply->deleteLater();
            m_reply = 0;
            this->performDownload(redirect);
            return;
        }

        if (!m_buffer.isEmpty()) {
            qint64 bytes = qint64(m_buffer.size());

            emit bytesTransferred(bytes);
            emit dataAvailable(this->position(), m_buffer);

            m_transferredBytes += bytes;
            m_buffer.clear();
        }

        break;
    }
    default:
        break;
    }

    switch (m_reply->error()) {
    case QNetworkReply::NoError:
        m_reply->deleteLater();
        m_reply = 0;
        this->setStatus(Transfers::Completed);
        break;
    case QNetworkReply::OperationCanceledError:
        m_reply->deleteLater();
        m_reply = 0;

        if ((this->contentRangeEnd() > 0) && (this->position() >= this->contentRangeEnd())) {
            this->setStatus(Transfers::Completed);
        }

        break;
    default:
        qDebug() << m_reply->errorString();
        this->setErrorString(m_reply->errorString());
        m_reply->deleteLater();
        m_reply = 0;
        this->setStatus(Transfers::Failed);
        break;
    }
}
