#include "dailymotionprofiletab.h"
#include "../base/dailymotion.h"
#include "../base/notifications.h"
#include "../base/thumbnailcache.h"
#include "../base/utils.h"
#include "separatorlabel.h"
#include "avatarlabel.h"
#include "resourceloader.h"
#include <QLabel>
#include <QPushButton>
#include <QScrollArea>
#include <QGridLayout>
#include <QDesktopServices>
#include <QMaemo5Style>

DailymotionProfileTab::DailymotionProfileTab(const QString &userId, QWidget *parent) :
    AbstractTab(parent),
    m_avatar(new AvatarLabel(this)),
    m_channelLabel(new QLabel(this)),
    m_viewsLabel(new QLabel(this)),
    m_locationLabel(new QLabel(this)),
    m_descriptionLabel(new QLabel(this)),
    m_noProfileLabel(new QLabel(QString("<p style='color:%1;font-size:40px;'><center><b>%2</b></center></p>").arg(QMaemo5Style::standardColor("SecondaryTextColor").name()).arg(tr("No profile found")), this)),
    m_subscriptionButton(new QPushButton(tr("Subscribe"))),
    m_scrollArea(new QScrollArea(this)),
    m_grid(new QGridLayout(this))
{
    QWidget *scrollWidget = new QWidget(m_scrollArea);
    QGridLayout *grid = new QGridLayout(scrollWidget);

    grid->addWidget(m_channelLabel, 0, 0, 1, 3);
    grid->addWidget(m_avatar, 1, 0, Qt::AlignTop);
    grid->addWidget(m_viewsLabel, 1, 1, 1, 2, Qt::AlignTop);
    grid->addWidget(m_subscriptionButton, 2, 2);
    grid->addWidget(new SeparatorLabel(tr("Profile"), this), 3, 0, 1, 3);
    grid->addWidget(m_locationLabel, 4, 0, 1, 3, Qt::AlignTop);
    grid->addWidget(new SeparatorLabel(tr("Description"), this), 5, 0, 1, 3);
    grid->addWidget(m_descriptionLabel, 6, 0, 1, 3);

    m_grid->setContentsMargins(0, 0 ,0, 0);
    m_grid->addWidget(m_scrollArea);
    m_scrollArea->setWidgetResizable(true);
    m_scrollArea->setWidget(scrollWidget);
    m_scrollArea->setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
    m_descriptionLabel->setWordWrap(true);
    m_scrollArea->hide();
    m_noProfileLabel->hide();
    m_viewsLabel->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    m_subscriptionButton->setFixedWidth(150);
    m_subscriptionButton->setVisible(Dailymotion::instance()->userSignedIn());

    this->connect(Dailymotion::instance(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->connect(Dailymotion::instance(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
    this->connect(m_subscriptionButton, SIGNAL(clicked()), this, SLOT(toggleSubscription()));
    this->connect(m_descriptionLabel, SIGNAL(linkActivated(QString)), this, SLOT(parseActivatedLink(QString)));

    if ((userId == "me") || (userId == Dailymotion::instance()->username())) {
        Dailymotion::instance()->getCurrentUserProfile();
        m_subscriptionButton->hide();
    }
    else {
        Dailymotion::instance()->getUserProfile(userId);
    }

    Notifications::toggleBusy(true);
    this->onOrientationChanged();
}

void DailymotionProfileTab::setLandscapeLayout() {
    m_noProfileLabel->setFixedSize(800, 400);
}

void DailymotionProfileTab::setPortraitLayout() {
    m_noProfileLabel->setFixedSize(480, 700);
}

void DailymotionProfileTab::setUser(QSharedPointer<UserItem> user) {
    m_user = user;
    m_avatar->setName(m_user.data()->username());
    m_channelLabel->setText(QString("<p style='font-size:32px;'>%1</p>").arg(m_user.data()->channelName()));
    m_viewsLabel->setText(!m_user.data()->viewCount() ? tr("No views") : QString("%1 %2").arg(m_user.data()->viewCount()).arg(tr("views")));
    m_locationLabel->setText(QString("%1: %2").arg(tr("Location")).arg(m_user.data()->location().isEmpty() ? tr("Unknown") : m_user.data()->location()));
    m_descriptionLabel->setText(m_user.data()->description().isEmpty() ? tr("No description") : Utils::toRichText(m_user.data()->description()));
    m_subscriptionButton->setText(m_user.data()->subscribed() ? tr("Unsubscribe") : tr("Subscribe"));

    m_scrollArea->show();

    this->setAvatar();
    Notifications::toggleBusy(false);

    this->connect(m_user.data(), SIGNAL(subscribedChanged(bool)), this, SLOT(onSubscribedChanged(bool)));
    this->disconnect(Dailymotion::instance(), SIGNAL(gotUser(QSharedPointer<UserItem>)), this, SLOT(setUser(QSharedPointer<UserItem>)));
    this->disconnect(Dailymotion::instance(), SIGNAL(error(QString)), this, SLOT(onProfileError()));
}

void DailymotionProfileTab::setAvatar() {
    QImage avatar = ThumbnailCache::instance()->thumbnail(m_user.data()->avatarUrl(), QSize(128, 128));

    if (!avatar.isNull()) {
        m_avatar->setImage(avatar);
        this->disconnect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
    else {
        this->connect(ThumbnailCache::instance(), SIGNAL(thumbnailReady()), this, SLOT(setAvatar()));
    }
}

void DailymotionProfileTab::onProfileError() {
    m_noProfileLabel->show();
}

void DailymotionProfileTab::toggleSubscription() {
    if (m_user.data()->subscribed()) {
        Dailymotion::instance()->unsubscribe(m_user.data()->id());
    }
    else {
        Dailymotion::instance()->subscribe(m_user.data()->id());
    }
}

void DailymotionProfileTab::parseActivatedLink(const QString &link) {
    if (!ResourceLoader::instance()->display(link, false)) {
        QDesktopServices::openUrl(link);
    }
}

void DailymotionProfileTab::onSubscribedChanged(bool subscribed) {
    m_subscriptionButton->setText(subscribed ? tr("Unsubscribe") : tr("Subscribe"));
}
