#include "storage.h"
#include "connection.h"
#include <QXmlStreamReader>
#include <QXmlStreamWriter>
#include <QDir>
#include <QFile>
#include <QMapIterator>
#include <qplatformdefs.h>
#if QT_VERSION >= 0x050000
#include <QStandardPaths>
#else
#include <QDesktopServices>
#endif

#if (defined Q_WS_MAEMO_5) || (defined MEEGO_EDITION_HARMATTAN)
static const QString DIRECTORY("/home/user/cuteTube/");
#elif (defined Q_OS_SYMBIAN)
static const QString DIRECTORY("C:/.config/cuteTube/");
#elif QT_VERSION >= 0x050000
static const QString DIRECTORY(QStandardPaths::writableLocation(QStandardPaths::HomeLocation) + "/cuteTube/");
#else
static const QString DIRECTORY(QDesktopServices::storageLocation(QDesktopServices::HomeLocation) + "/cuteTube/");
#endif

Storage::Storage(QObject *parent) :
    QObject(parent)
{
}

Storage::~Storage() {}

bool Storage::storeTransfers(QList< QSharedPointer<TransferItem> > transfers) {
    if (transfers.isEmpty()) {
        return false;
    }

    QDir dir;

    if (!dir.mkpath(DIRECTORY)) {
        return false;
    }

    QFile file(DIRECTORY + "transfers.xml");

    if (!file.open(QIODevice::WriteOnly)) {
        return false;
    }

    QXmlStreamWriter writer(&file);
    writer.setAutoFormatting(true);
    writer.writeStartDocument();
    writer.writeStartElement("transfers");

    foreach (QSharedPointer<TransferItem> transfer, transfers) {
        writer.writeStartElement("transfer");
        writer.writeTextElement("transferType", QString::number(Transfers::TransferType(transfer.data()->transferType())));
        writer.writeTextElement("id", transfer.data()->id());
        writer.writeTextElement("service", QString::number(Services::VideoService(transfer.data()->service())));
        writer.writeTextElement("title", transfer.data()->title());
        writer.writeTextElement("fileName", transfer.data()->fileName());
        writer.writeTextElement("size", QString::number(transfer.data()->size()));
        writer.writeTextElement("priority", QString::number(Transfers::Priority(transfer.data()->priority())));

        switch (transfer.data()->transferType()) {
        case Transfers::Upload:
        {
            writer.writeStartElement("uploadMetadata");

            QMapIterator<QString, QVariant> iterator(transfer.data()->uploadMetadata());
            
            while (iterator.hasNext()) {
                iterator.next();
                writer.writeTextElement(iterator.key(), iterator.value().toString());
            }

            writer.writeEndElement();        
            break;
        }
        default:
            writer.writeTextElement("downloadPath", transfer.data()->downloadPath());
            writer.writeTextElement("videoId", transfer.data()->videoId());
            writer.writeTextElement("thumbnailUrl", transfer.data()->thumbnailUrl().toString());
            writer.writeTextElement("videoFormat", QString::number(Videos::VideoFormat(transfer.data()->videoFormat())));
            writer.writeTextElement("downloadSubtitles", transfer.data()->downloadSubtitles() ? "1" : "0");
            writer.writeTextElement("subtitlesLanguage", transfer.data()->subtitlesLanguage());
            writer.writeTextElement("convertToAudio", transfer.data()->convertToAudio() ? "1" : "0");
            writer.writeTextElement("position", QString::number(transfer.data()->position()));
            writer.writeTextElement("preferredConnections", QString::number(transfer.data()->preferredConnections()));

            foreach (Connection* connection, transfer.data()->connections()) {
                writer.writeStartElement("connection");
                writer.writeAttribute("position", QString::number(connection->position()));
                writer.writeAttribute("end", QString::number(connection->contentRangeEnd()));
                writer.writeEndElement();
            }

            break;
        }

        writer.writeEndElement();
    }

    writer.writeEndElement();
    writer.writeEndDocument();

    return true;
}

QList< QSharedPointer<TransferItem> > Storage::restoreTransfers() {
    QList< QSharedPointer<TransferItem> > transfers;

    if (!QFile::exists(DIRECTORY + "transfers.xml")) {
        return transfers;
    }

    QFile file(DIRECTORY + "transfers.xml");

    if (!file.open(QIODevice::ReadOnly)) {
        return transfers;
    }

    QXmlStreamReader reader(&file);

    while ((!reader.atEnd()) && (!reader.hasError())) {
        while ((!reader.atEnd()) && (!reader.hasError()) && (reader.name() != "transfer")) {
            while (reader.readNext() != QXmlStreamReader::Invalid) {
                if ((reader.isStartElement()) || (reader.isEndElement())) {
                    break;
                }
            }
        }

        if (reader.name() == "transfer") {
            QSharedPointer<TransferItem> transfer = QSharedPointer<TransferItem>(new TransferItem);

            while (reader.readNext() != QXmlStreamReader::Invalid) {
                if ((reader.isStartElement()) || (reader.isEndElement())) {
                    break;
                }
            }

            while ((!reader.atEnd()) && (!reader.hasError()) && (reader.name() != "transfer")) {
                if (reader.name() == "transferType") {
                    transfer.data()->setTransferType(static_cast<Transfers::TransferType>(reader.readElementText().toInt()));
                }
                else if (reader.name() == "id") {
                    transfer.data()->setId(reader.readElementText());
                }
                else if (reader.name() == "videoId") {
                    transfer.data()->setVideoId(reader.readElementText());
                }
                else if (reader.name() == "service") {
                    transfer.data()->setService(static_cast<Services::VideoService>(reader.readElementText().toInt()));
                }
                else if (reader.name() == "size") {
                    transfer.data()->setSize(reader.readElementText().toLongLong());
                }
                else if (reader.name() == "fileName") {
                    transfer.data()->setFileName(reader.readElementText());
                }
                else if (reader.name() == "downloadPath") {
                    transfer.data()->setDownloadPath(reader.readElementText());
                }
                else if (reader.name() == "title") {
                    transfer.data()->setTitle(reader.readElementText());
                }
                else if (reader.name() == "thumbnailUrl") {
                    transfer.data()->setThumbnailUrl(reader.readElementText());
                }
                else if (reader.name() == "videoFormat") {
                    transfer.data()->setVideoFormat(static_cast<Videos::VideoFormat>(reader.readElementText().toInt()));
                }
                else if (reader.name() == "downloadSubtitles") {
                    transfer.data()->setDownloadSubtitles(reader.readElementText().toInt());
                }
                else if (reader.name() == "subtitlesLanguage") {
                    transfer.data()->setSubtitlesLanguage(reader.readElementText());
                }
                else if (reader.name() == "convertToAudio") {
                    transfer.data()->setConvertToAudio(reader.readElementText().toInt());
                }
                else if (reader.name() == "position") {
                    transfer.data()->setResumePosition(reader.readElementText().toLongLong());
                }
                else if (reader.name() == "preferredConnections") {
                    transfer.data()->setPreferredConnections(reader.readElementText().toInt());
                }
                else if (reader.name() == "priority") {
                    transfer.data()->setPriority(static_cast<Transfers::Priority>(reader.readElementText().toInt()));
                }
                else if (reader.name() == "uploadMetadata") {
                    while (reader.readNext() != QXmlStreamReader::Invalid) {
                        if ((reader.isStartElement()) || (reader.isEndElement())) {
                            break;
                        }
                    }

                    QVariantMap metadata;

                    while ((!reader.atEnd()) && (!reader.hasError()) && (reader.name() != "uploadMetadata")) {
                        metadata[reader.name().toString()] = reader.readElementText();

                        while (reader.readNext() != QXmlStreamReader::Invalid) {
                            if ((reader.isStartElement()) || (reader.isEndElement())) {
                                break;
                            }
                        }
                    }

                    transfer.data()->setUploadMetadata(metadata);
                }
                else if (reader.name() == "connection") {
                    transfer.data()->restoreConnection(reader.attributes().value("position").toString().toLongLong(),
                                                       reader.attributes().value("end").toString().toLongLong());

                    while (reader.readNext() != QXmlStreamReader::Invalid) {
                        if ((reader.isStartElement()) || (reader.isEndElement())) {
                            break;
                        }
                    }
                }

                while (reader.readNext() != QXmlStreamReader::Invalid) {
                    if ((reader.isStartElement()) || (reader.isEndElement())) {
                        break;
                    }
                }
            }

            while (reader.readNext() != QXmlStreamReader::Invalid) {
                if ((reader.isStartElement()) || (reader.isEndElement())) {
                    break;
                }
            }

            transfers.append(transfer);
        }
    }

    return transfers;
}

bool Storage::clearStoredTransfers() {
    return QFile::remove(DIRECTORY + "transfers.xml");
}
