# -*- coding: utf-8 -*-

"""
Module implementing VideoInfoWindow.
"""
from PyQt4.QtGui import *
from PyQt4.QtCore import *
import time
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass
from Ui_videoinfowindow import Ui_VideoInfoWindow

import cutetubedialogs
import cutetubewindows
import youtubeservice
import customwidgets
from cutetubeconfig import VideoSharingSettings, GeneralSettings
from videoplaybackwindow import VideoPlaybackWindow

class VideoInfoWindow(QMainWindow, Ui_VideoInfoWindow):
    """
    This window displays all video information and provides options 
    for the user to play/download the video and add comments etc.
    """
    def __init__(self, parent, video, playlistId = None, position = None, count = None):
        """
        Constructor
        """
        QMainWindow.__init__(self, parent)
        self.setAttribute(Qt.WA_Maemo5StackedWindow)
        self.setAttribute(Qt.WA_Maemo5AutoOrientation, GeneralSettings.autoOrientation)
        self.setupUi(self)
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.orientationChanged()
        self.descriptionTextBox.viewport().setAutoFillBackground(False)
        self.titleLabel.viewport().setAutoFillBackground(False)
        self.video = video
        if playlistId != None:
            self.playlistId = playlistId
            self.playlistPosition = position
            self.playlistLength = count
            self.actionEditVideo.setVisible(True)
            self.actionEditVideo.setText("Edit custom metadata")
        if self.video.media.credit.text == youtubeservice.YouTubeDataService.currentUser:
            self.actionRateVideo.setVisible(False)
            self.actionUserVideos.setVisible(False)
            self.actionEditVideo.setVisible(True)
        if youtubeservice.YouTubeDataService.currentUser == "":
            self.actionRateVideo.setVisible(False)
            self.actionAddComment.setVisible(False)
            self.actionAddToFavourites.setVisible(False)
            self.actionAddToPlaylist.setVisible(False)
        self.show()
        self.loadData()

        
    def orientationChanged(self):
        """
        Re-orders the widgets when the application 
        is in portrait mode.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.likeLabel.setGeometry(QRect(260, 15, 45, 45))
            self.likeCountLabel.setGeometry(QRect(310, 15, 150, 45))
            self.dislikeLabel.setGeometry(QRect(260, 65, 45, 45))
            self.dislikeCountLabel.setGeometry(QRect(310, 65, 150, 45))
            self.viewsLabel.setGeometry(QRect(260, 120, 50, 30))
            self.viewsNumberLabel.setGeometry(QRect(315, 120, 180, 25))
            self.commentsLabel.setGeometry(QRect(260, 162, 50, 40))
            self.commentNumberLabel.setGeometry(QRect(315, 162, 180, 25))
            self.titleLabel.setGeometry(QRect(10, 205, 460, 70))
            self.descriptionTextBox.setGeometry(QRect(10, 275, 460, 295))
            self.clipboardButton.setGeometry(QRect(10, 580, 70, 70))
            self.facebookButton.setGeometry(QRect(95, 580, 70, 70))
            self.twitterButton.setGeometry(QRect(180, 580, 70, 70))
            self.downloadButton.setGeometry(QRect(10, 660, 225, 70))
            self.playButton.setGeometry(QRect(245, 660, 225, 70))
        else:
            self.titleLabel.setGeometry(QRect(260, 20, 530, 70))
            self.likeLabel.setGeometry(QRect(10, 210, 45, 45))
            self.likeCountLabel.setGeometry(QRect(70, 210, 70, 45))
            self.dislikeLabel.setGeometry(QRect(125, 210, 45, 45))
            self.dislikeCountLabel.setGeometry(QRect(185, 210, 70, 45))
            self.commentsLabel.setGeometry(QRect(10, 290, 50, 40))
            self.viewsLabel.setGeometry(QRect(10, 260, 50, 30))
            self.viewsNumberLabel.setGeometry(QRect(70, 260, 180, 25))
            self.commentNumberLabel.setGeometry(QRect(70, 295, 180, 25))
            self.descriptionTextBox.setGeometry(QRect(260, 90, 530, 227))
            self.clipboardButton.setGeometry(QRect(10, 340, 70, 70))
            self.facebookButton.setGeometry(QRect(95, 340, 70, 70))
            self.twitterButton.setGeometry(QRect(180, 340, 70, 70))
            self.downloadButton.setGeometry(QRect(260, 340, 260, 70))
            self.playButton.setGeometry(QRect(530, 340, 260, 70))

    def loadData(self):
        """
        Loads the video data into the appropriate widgets.
        """
        self.toggleBusy(True)
        self.loadTitleAndDescription()
        self.loadViewsCommentsRating()
        self.loadThumbnail()
        self.toggleBusy()

    def loadTitleAndDescription(self):
        """
        Loads the title and description into the QTextEdit widgets.
        """
        # Load the title and description. Some videos do not have a description, in which case None is returned.
        self.actionUserVideos.setText("%s\'s Videos" % unicode(self.video.media.credit.text, "utf-8"))
        self.titleLabel.setText(unicode(self.video.title.text, "utf-8"))       
        try:
            if "playlist" in self.video.id.text:
                self.descriptionTextBox.setText(unicode(self.video.content.text, "utf-8"))
            else:
                self.descriptionTextBox.setText(unicode(self.video.media.description.text, "utf-8"))
        except:
            self.descriptionTextBox.setText("No description")

    def loadViewsCommentsRating(self):
        """
        Loads the view count, comment count and rating 
        into the appropriate widgets.
        """        
        # Load view count and comment count. Some videos do not have a view count, in which case None is returned.
        try:
            self.viewsNumberLabel.setText(self.video.statistics.view_count)
        except:
            self.viewsNumberLabel.setText("--")
        try: 
            self.commentNumberLabel.setText(self.video.comments.feed_link[0].count_hint)
            if self.video.comments.feed_link[0].count_hint == "0":
                self.actionViewComments.setVisible(False)
        except:
            self.commentNumberLabel.setText("0")
            self.actionViewComments.setVisible(False)
        # Load video's average rating. Some videos may not have a rating, in which case it is set to '3 stars'.
        try:
            self.likeCountLabel.setText(self.video.likes_and_dislikes.num_likes)
            self.dislikeCountLabel.setText(self.video.likes_and_dislikes.num_dislikes)
        except:
            pass
        
                
    def loadThumbnail(self):
        """
        Loads the duration and thumbnail into the durationLabel and thumbLabel.
        """
        # Load video duration, converting seconds to MM:SS format.
        self.durationLabel.setText(unicode(time.strftime('%M:%S', time.gmtime(float(self.video.media.duration.seconds)))))
        
        thumbUrl = self.video.media.thumbnail[4].url
        thumbnail = youtubeservice.YouTubeDataService.getVideoThumbnail(thumbUrl)
        try:
            self.thumbLabel.setPixmap(thumbnail) 
        except:
            self.thumbLabel.setPixmap(QPixmap(":/images/ui-images/thumbnail.png"))
    
    def playVideo(self, event = None):
        """
        Plays the video.
        """
        videoTitle = self.video.title.text
        try:
            self.toggleBusy(True)
            videoPlaybackWindow = VideoPlaybackWindow(self, video = self.video)
        except:
            QMaemo5InformationBox.information(self, "Unable to play video")
        self.toggleBusy()

    def downloadVideo(self):
        videoTitle = self.video.media.title.text
        try:
            youtubeservice.YouTubeVideoDownloader.addTask(self.video)
            QMaemo5InformationBox.information(self, "\'%s\' has been added to your download queue" % unicode(videoTitle, "utf-8"))
            self.emit(SIGNAL("downloadsChanged(int)"), len(youtubeservice.YouTubeVideoDownloader.taskQueue))
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to your download queue")
    
    def toggleBusy(self, isBusy = False):
        """
        Shows the progress indicator and disables the 
        play button if argument is True, and hides it otherwise.
        """
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, isBusy)
        
    @pyqtSignature("")
    def on_clipboardButton_clicked(self):
        """
        Copies the video URL to the clipboard.
        """
        videoUrl = self.video.media.player.url.split("&")[0]
        clipboard = QApplication.clipboard()
        clipboard.setText(videoUrl)
        QMaemo5InformationBox.information(self, "Video URL copied")
        
    @pyqtSignature("")
    def on_facebookButton_clicked(self):
        """
        Raises a ShareViaFacebookDialog.
        """
        self.toggleBusy(True)
        videoUrl = self.video.media.player.url.split("&")[0]
        if not VideoSharingSettings.raiseFacebookAuthorisationDialog:
            shareViaFacebookDialog = cutetubedialogs.ShareViaFacebookDialog(self, videoUrl)
        else:
            facebookAuthorisationDialog = cutetubedialogs.FacebookAuthorisationDialog(self)
            self.connect(facebookAuthorisationDialog, SIGNAL("accepted()"), self.on_facebookButton_clicked)
        self.toggleBusy()
        
    @pyqtSignature("")
    def on_twitterButton_clicked(self):
        """
        Raises a ShareViaTwitterDialog.
        """
        self.toggleBusy(True)
        videoUrl = self.video.media.player.url.split("&")[0]
        if not VideoSharingSettings.raiseTwitterAuthorisationDialog:
            shareViaTwitterDialog = cutetubedialogs.ShareViaTwitterDialog(self, videoUrl)
        else:
            twitterAuthorisationDialog = cutetubedialogs.TwitterAuthorisationDialog(self)
            self.connect(twitterAuthorisationDialog, SIGNAL("accepted()"), self.on_twitterButton_clicked)
        self.toggleBusy()
    
    @pyqtSignature("")
    def on_playButton_clicked(self):
        self.playVideo()
    
    @pyqtSignature("")
    def on_downloadButton_clicked(self):
        """
        Adds the current video to the download queue.
        """
        self.downloadVideo()

    @pyqtSignature("")
    def on_actionAddToQueue_triggered(self):
        """
        Adds the video to the user's playback queue.
        """
        videoTitle = self.video.media.title.text
        try:
            youtubeservice.YouTubeQueueFeed.playlist.append(self.video)
            self.emit(SIGNAL("queueChanged(int)"), len(youtubeservice.YouTubeQueueFeed.playlist))
            QMaemo5InformationBox.information(self, "\'%s\' added to playback queue" % unicode(videoTitle, "utf-8"))            
        except:
            QMaemo5InformationBox.information(self, "Unable to add video to playback queue")
       
    
    @pyqtSignature("")
    def on_actionRateVideo_triggered(self):
        """
        Raises a dialog enabling the user to 'like' or 'dislike' the video.
        """
        videoId = self.video.media.video_id.text
        rateVideoDialog = cutetubedialogs.RateVideoDialog(self, videoId)

    @pyqtSignature("")
    def on_actionAddComment_triggered(self):
        """
        Raises a dialog asking the user to enter and 
        submit a comment for the current video.
        """
        addCommentDialog = cutetubedialogs.AddCommentDialog(self, self.video)
        self.connect(addCommentDialog, SIGNAL("accepted()"), self.on_actionViewComments_triggered)

    @pyqtSignature("")
    def on_actionViewComments_triggered(self):
        """
        Raises a dialog asking the user to enter and 
        submit a comment for the current video.
        """
        commentsWindow = cutetubewindows.CommentsWindow(self, self.video)
    
    @pyqtSignature("")
    def on_actionUserVideos_triggered(self):
        """
        Creates a ChannelWindow displaying all videos 
        from the uploader of the current video.
        """
        userName = self.video.media.credit.text
        userWindow = cutetubewindows.UserWindow(self, userName)
        self.connect(userWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(userWindow, SIGNAL("queueChanged(int)"), self.queueChanged)

    @pyqtSignature("")
    def on_actionRelatedVideos_triggered(self):
        """
        Creates a ChannelWindow displaying a 
        related videos feed.
        """
        relatedVideosWindow = cutetubewindows.RelatedVideosWindow(self, self.video)
        self.connect(relatedVideosWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
        self.connect(relatedVideosWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
    
    @pyqtSignature("")
    def on_actionAddToFavourites_triggered(self):
        """
        Adds the video to the user's favourites and 
        displays an information box to confirm.
        """
        try:
            videoId = self.video.media.video_id.text
            addedToFavourites = youtubeservice.YouTubeDataService.addVideoToFavourites(videoId)
            if addedToFavourites == True:
                QMaemo5InformationBox.information(self, "\'%s\' has been added to your favourites" % unicode(self.video.media.title.text, "utf-8"))
            else:
                QMaemo5InformationBox.information(self,  addedToFavourites)
        except:
            QMaemo5InformationBox.information(self,  "Unable to add video to your favourites")

    @pyqtSignature("")
    def on_actionAddToPlaylist_triggered(self):
        """
        Raises a dialog asking the user to select a 
        playlist to which to add the current video.
        """
        playlistDialog = cutetubedialogs.AddToPlaylistDialog(self, [self.video])

    @pyqtSignature("")
    def on_actionEditVideo_triggered(self):
        """
        Raises an EditVideoDialog.
        """
        if "playlist" in self.video.id.text:
            editVideoDialog = cutetubedialogs.EditPlaylistVideoDialog(self, self.playlistId, self.video, self.playlistPosition, self.playlistLength)
        else:
            editVideoDialog = cutetubedialogs.EditVideoDialog(self, self.video)
        self.connect(editVideoDialog, SIGNAL("accepted()"), self.loadData)

    @pyqtSignature("")
    def on_ratingButton_clicked(self):
        """
        Raises a dialog asking the user to select a 
        rating from 1-5 stars.
        """
        rateVideoDialog = cutetubedialogs.RateVideoDialog(self, self.video.id.text)
        
    def queueChanged(self, queueLength):
        """
        Connects the the queueChanged signal to the parent window.
        """
        self.emit(SIGNAL("queueChanged(int)"), queueLength)
        
    def downloadsChanged(self,  taskQueueLength):
        """
        Connects the downloadsChanged signal to the parent window.
        """
        self.emit(SIGNAL("downloadsChanged(int)"), taskQueueLength)
        

