# -*- coding: utf-8 -*-

"""
Module implementing HomeWindow.
"""
import os
from PyQt4.QtGui import *
from PyQt4.QtCore import *
try:
    from PyQt4.QtMaemo5 import QMaemo5InformationBox
except:
    pass
from Ui_homewindow import Ui_HomeWindow

import myaccountwindow
import cutetubedialogs
import aboutdialog
import cutetubewindows
from youtubeservice import YouTubeDataService, YouTubeVideoDownloader
from cutetubeconfig import *

class HomeWindow(QMainWindow, Ui_HomeWindow):
    """
    Class documentation goes here.
    """
    def __init__(self, parent = None):
        """
        Constructor
        """
        QMainWindow.__init__(self, parent)
        self.setupUi(self)
        try:
            self.setAttribute(Qt.WA_Maemo5StackedWindow)  
            self.setAttribute(Qt.WA_Maemo5AutoOrientation, True)
        except:
            pass
        self.youtubeLabel = QLabel(self)
        self.youtubeLabel.setGeometry(QRect(10, 10, 150, 90))
        self.youtubeLabel.setPixmap(QPixmap(":/images/ui-images/youtube.png"))
        self.youtubeLabel.setScaledContents(True)
        self.orientationChanged()
        self.connect(QApplication.desktop(), SIGNAL("resized(int)"), self.orientationChanged)
        self.keyPressEvent = self.showSearchDialog
        for button in (self.myAccountButton, self.mostRecentButton, self.mostViewedButton, self.searchButton):
            button.setStyleSheet("border: 2px solid rgb(120, 120, 120); border-radius: 6px;") 
            self.connect(button,  SIGNAL("pressed()"), self.changeButtonStyleSheet)
            self.connect(button,  SIGNAL("released()"), self.changeButtonStyleSheet)
        self.searchDialogRaised = False
        taskQueueLength = len(YouTubeVideoDownloader.taskQueue)
        if taskQueueLength > 0:
            self.actionDownloads.setVisible(True)
            self.actionDownloads.setText("Downloads (%s videos)" % unicode(taskQueueLength))
        self.show()
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, True)
        self.restoreSettings()
        self.youtubeLogin()
        
    def changeButtonStyleSheet(self):
        """
        Setting the QToolButton style sheets in the conventional 
        way caused trouble with the Maemo specific widgets, so 
        I'm using a signal instead.
        """
        button = self.sender()
        if button.isDown():
            button.setStyleSheet("border: 2px solid rgb(120, 120, 120); border-radius: 6px; background-color: rgb(120, 120, 120)")
        else:
            button.setStyleSheet("border: 2px solid rgb(120, 120, 120); border-radius: 6px;") 
            
        
    def youtubeLogin(self):
        """
        Attempt to sign in using the default account details.
        """
        try:
            if YouTubeAPISettings.userDict == {}:
                if YouTubeAPISettings.raiseNoAccountFoundDialog:
                    noAccountFoundDialog = cutetubedialogs.NoAccountFoundDialog(self)
            else:
                username = YouTubeAPISettings.defaultUser
                YouTubeDataService.youtubeLogin(username)
                QMaemo5InformationBox.information(self, "You are signed in to YouTube as \'%s\'" % username)
        except:
            try:
                username = YouTubeAPISettings.userDict.keys()[0]
                YouTubeDataService.youtubeLogin(username)
                QMaemo5InformationBox.information(self, "You are signed in to YouTube as \'%s\'" % username)
            except:
                QMaemo5InformationBox.information(self, "Unable to sign in to your YouTube account")
        self.setAttribute(Qt.WA_Maemo5ShowProgressIndicator, False)
                
    def showSearchDialog(self,  event):
        """
        Raises a SearchDialog when the user presses a key.
        """
        if event.key() not in (Qt.Key_Left, Qt.Key_Right, Qt.Key_Up, Qt.Key_Down, Qt.Key_Backspace, Qt.Key_Return, Qt.Key_Control, Qt.Key_Shift, Qt.Key_Space) and not self.searchDialogRaised:
            self.setSearchDialogRaised(True)
            searchDialog = cutetubedialogs.SearchDialog(self)
            searchDialog.lineEdit.insert(event.text())
            QTimer.singleShot(1000,  lambda isRaised = False: self.setSearchDialogRaised(isRaised))
        else:
            pass
            
    def setSearchDialogRaised(self,  isRaised):
        self.searchDialogRaised = isRaised
                
    def restoreSettings(self):
        """
        Restores the application settings.
        """
        settings = QSettings()
        YouTubeAPISettings.userDict = settings.value("userDict",  {}).toPyObject()
        YouTubeAPISettings.defaultUser = settings.value("defaultUser",  "").toString()
        YouTubeAPISettings.raiseNoAccountFoundDialog = settings.value("noAccountDialog",  True).toBool()
        VideoPlaybackSettings.playbackFormat = settings.value("playbackFormat",  "18").toPyObject()
        VideoDownloadSettings.downloadFolder = settings.value("downloadFolder",  "/home/user/MyDocs/.videos").toString()
        VideoDownloadSettings.defaultTaskStatus = settings.value("downloadStatus",  "paused").toString()
        if settings.value("downloadFormat",  "18").toPyObject() in ("17", "18"):
            VideoDownloadSettings.downloadFormat = settings.value("downloadFormat",  "18").toPyObject()
        SearchSettings.searchOrder = settings.value("searchOrder",  0).toInt()[0]
        SearchSettings.recentSearches = settings.value("recentSearches", QStringList()).toStringList()
        
    def closeEvent(self,  event):
        """
        Saves the application settings.
        """
        settings = QSettings()
        userDict = QVariant(YouTubeAPISettings.userDict)
        defaultUser = QVariant(YouTubeAPISettings.defaultUser)
        noAccountDialog = QVariant(YouTubeAPISettings.raiseNoAccountFoundDialog)
        playbackFormat = QVariant(VideoPlaybackSettings.playbackFormat)        
        downloadFolder = QVariant(VideoDownloadSettings.downloadFolder)
        downloadStatus = QVariant(VideoDownloadSettings.defaultTaskStatus)
        downloadFormat = QVariant(VideoDownloadSettings.downloadFormat)
        searchOrder = QVariant(SearchSettings.searchOrder)
        recentSearches = QVariant(SearchSettings.recentSearches)
        settings.setValue("userDict", userDict)
        settings.setValue("defaultUser", defaultUser)
        settings.setValue("noAccountDialog", noAccountDialog)
        settings.setValue("downloadFolder", downloadFolder)
        settings.setValue("downloadStatus", downloadStatus)
        settings.setValue("playbackFormat", playbackFormat)
        settings.setValue("downloadFormat", downloadFormat)
        settings.setValue("searchOrder", searchOrder)
        settings.setValue("recentSearches", recentSearches)

    def orientationChanged(self):
        """
        Re-orders the window's widgets.
        """
        screenGeometry = QApplication.desktop().screenGeometry()
        if screenGeometry.height() > screenGeometry.width():
            self.resize(480, 800)
            self.myAccountButton.setGeometry(QRect(80, 150, 120, 120))
            self.mostRecentButton.setGeometry(QRect(280, 150, 120, 120))
            self.mostViewedButton.setGeometry(QRect(80, 400, 120, 120))
            self.searchButton.setGeometry(QRect(280, 400, 120, 120))
            self.myAccountLabel.setGeometry(QRect(76, 278, 127, 26))
            self.mostRecentLabel.setGeometry(QRect(274, 278, 132, 26))
            self.mostViewedLabel.setGeometry(QRect(72, 527, 135, 26))
            self.searchLabel.setGeometry(QRect(301, 527, 75, 26))
        else:
            self.resize(800, 480)
            self.myAccountButton.setGeometry(QRect(57, 123, 120, 120))
            self.mostRecentButton.setGeometry(QRect(243, 123, 120, 120))
            self.mostViewedButton.setGeometry(QRect(434, 123, 120, 120))
            self.searchButton.setGeometry(QRect(628, 123, 120, 120))
            self.myAccountLabel.setGeometry(QRect(53, 250, 127, 26))
            self.mostRecentLabel.setGeometry(QRect(239, 250, 132, 26))
            self.mostViewedLabel.setGeometry(QRect(426, 250, 135, 26))
            self.searchLabel.setGeometry(QRect(651, 250, 75, 26))

    def queueChanged(self, queueLength):
        """
        Makes actionGoToQueue visible if there are 
        videos in the user's playback queue.
        """
        if queueLength > 0:
            self.actionGoToQueue.setVisible(True)
            self.actionGoToQueue.setText("Queue (%s videos)" % unicode(queueLength))
        else:
            self.actionGoToQueue.setVisible(False)
            
    def downloadsChanged(self,  taskQueueLength):
        """
        Makes actionDownloads visible if there are videos 
        in the user's download queue.
        """
        if taskQueueLength > 0:
            self.actionDownloads.setVisible(True)
            self.actionDownloads.setText("Downloads (%s videos)" % unicode(taskQueueLength))
        else:
            self.actionDownloads.setVisible(False)

    @pyqtSignature("")
    def on_actionGoToQueue_triggered(self):
        """
        Creates a PlayBackQueueWindow.
        """
        playbackQueueWindow = cutetubewindows.PlaybackQueueWindow(self)
        self.connect(playbackQueueWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(playbackQueueWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
    
    @pyqtSignature("")
    def on_myAccountButton_clicked(self):
        """
        Creates a MyAccounrtWindow
        """
        if YouTubeDataService.currentUser == "":
            QMaemo5InformationBox.information(self, "You are not signed in to YouTube")
        else:
            myAccountWindow = myaccountwindow.MyAccountWindow(self)
            self.connect(myAccountWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
            self.connect(myAccountWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
    
    @pyqtSignature("")
    def on_mostRecentButton_clicked(self):
        """
        Creates a MostRecentWindow.
        """
        mostRecentWindow = cutetubewindows.MostRecentWindow(self)
        self.connect(mostRecentWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(mostRecentWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
    
    @pyqtSignature("")
    def on_mostViewedButton_clicked(self):
        """
        Creates a MostViewedWindow
        """
        mostViewedWindow = cutetubewindows.MostViewedWindow(self)
        self.connect(mostViewedWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(mostViewedWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)
    
    @pyqtSignature("")
    def on_searchButton_clicked(self):
        """
        Raises a SearchDialog.
        """
        searchDialog = cutetubedialogs.SearchDialog(self)
    
    @pyqtSignature("")
    def on_actionSettings_triggered(self):
        """
        Raises a SettingsDialog.
        """
        settingsDialog = cutetubedialogs.SettingsDialog(self)
    
    @pyqtSignature("")
    def on_actionDownloads_triggered(self):
        """
        Creates a DownloadsWindow
        """
        downloadsWindow = cutetubewindows.DownloadsWindow(self)
        self.connect(downloadsWindow, SIGNAL("queueChanged(int)"), self.queueChanged)
        self.connect(downloadsWindow, SIGNAL("downloadsChanged(int)"), self.downloadsChanged)

    @pyqtSignature("")
    def on_actionManageAccounts_triggered(self):
        """
        Raises an ManageAccountDialog
        """
        manageAccountsDialog = cutetubedialogs.ManageAccountsDialog(self)
    
    @pyqtSignature("")
    def on_actionAbout_triggered(self):
        """
        Raises an AboutDialog
        """
        aboutDialog = cutetubedialogs.AboutDialog(self)

        
        
