/*
*  currency exchange widget for the maemo desktop.
*  Copyright (C) 2010 binop consulting group
*  
*  This program is free software; you can redistribute it and/or modify
*  it under the terms of the GNU General Public License as published by
*  the Free Software Foundation; either version 2 of the License, or
*  (at your option) any later version.
*  
*  This program is distributed in the hope that it will be useful,
*  but WITHOUT ANY WARRANTY; without even the implied warranty of
*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*  GNU General Public License for more details.
*  
*  You should have received a copy of the GNU General Public License
*  along with this program; if not, write to the Free Software
*  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <gtk/gtk.h>

#include <pthread.h>
#include <hildon/hildon.h>
#include <libintl.h> 
#include <libhildondesktop/libhildondesktop.h>
#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include "lib-currency-home-widget.h"
#include <string.h>

HD_DEFINE_PLUGIN_MODULE (CurrencyWidget, currency_widget_plugin, HD_TYPE_HOME_PLUGIN_ITEM)

static void currency_widget_realize(GtkWidget *widget);
static gboolean currency_widget_expose_event(GtkWidget *widget, GdkEventExpose *event);
static void currency_widget_finalize(GObject *object);
void create_icon_factory();
static DBusHandlerResult
change_ref_currency_filter (DBusConnection *connection, DBusMessage *message, void *user_data);


extern char *get_icon_string(char *);
extern void build_ui(CurrencyWidget **);
extern gboolean callback( GtkWidget *, GdkEventButton *, CurrencyWidget *);
extern void save_config(CurrencyWidget **);
extern gboolean as_update_values(CurrencyWidget *);
extern gboolean as_update_values_once(CurrencyWidget *);


char *currencies[] = { "EUR", "USD", "GBP", "AUD", "BRL", "CAD", "CHF", "CNY", "DKK", "HKD", "INR", "JPY", "KRW", "LKR", "MYR", "NZD", "SGD", "TWD", "ZAR", "VEB", "THB", "SEK", "NOK", "MXN", "BGN", 
        "CZK", "EEK", "HUF", "LTL", "LVL", "PLN", "RON", "SKK", "ISK","HRK", "RUB", "TRY", "PHP", "COP", "ARS", "CLP", "SVC", "TND" ,"PYG", "MAD", "JMD", "SAR", "QAR", "HNL" };

char *countries[] = { "European Union", "United States", "United Kingdom", "Australia", "Brazil", "Canada", "Switzerland", "China", "Denmark", "Hong Kong", "India", "Japan", "Korea",
    "Sri Lanka", "Malasia", "New Zealand", "Singapore", "Taiwan", "South Africa", "Venezuela", "Thailand", "Sweden", "Norway", "Mexico", "Bulgaria", "Czech Republic", "Estonia", 
    "Hungary", "Lithuania", "Latvia", "Poland", "Romania", "Slovakia", "Iceland", "Croatia", "Russia", "Turkey", "Philippines", "Colombia", "Argentina", "Chile", "El Salvador",
    "Tunisia", "Paraguay", "Morocco", "Jamaica", "Saudi Arabia", "Qatar", "Honduras" };

int n_currencies = 49;


static void
currency_widget_plugin_init (CurrencyWidget *desktop_plugin)
{

    desktop_plugin->id = malloc(sizeof(char)*100);
    desktop_plugin->curset = NULL;

    if ( pthread_mutex_init(&desktop_plugin->mutex, NULL) != 0 ) {
        fprintf(stderr, "Could not init mutex!");
    }

    if ( !desktop_plugin->id ) {
        fprintf(stderr, "Could not alloc memory for id!\n");
        exit(EXIT_FAILURE);
    }
   
    if ( pthread_mutex_lock(&desktop_plugin->mutex) != 0 ) {
        fprintf(stderr, "Could not lock mutex");
    }

    init_currency_widget(&desktop_plugin);
    gtk_widget_set_size_request(GTK_WIDGET(desktop_plugin), 260, 50);
    build_ui (desktop_plugin);
    

    g_signal_connect (GTK_CONTAINER(desktop_plugin->main_event_box), "button-release-event",
           G_CALLBACK (callback), desktop_plugin );



    gtk_container_add (GTK_CONTAINER (desktop_plugin), GTK_WIDGET(desktop_plugin->contents));
    gtk_window_set_opacity(GTK_WINDOW(desktop_plugin), 0.4);
    
    if ( pthread_mutex_unlock(&desktop_plugin->mutex) != 0 ) {
        fprintf(stderr, "could not unlock mutex");
    }

    // Initialize dbus
    DBusError error;

    dbus_error_init (&error);
    desktop_plugin->bus = dbus_bus_get (DBUS_BUS_SESSION, &error);

    if ( !desktop_plugin->bus ) {
        fprintf(stderr, "Failed to connect to dbus!");
        dbus_error_free( &error);
        exit(EXIT_FAILURE);
    }

    dbus_bus_add_match (desktop_plugin->bus, "type='signal', interface='com.binop.currencyexchangewidget.refcursignal.Signal'", &error);
    dbus_connection_add_filter(desktop_plugin->bus, change_ref_currency_filter, desktop_plugin, NULL);
} 

static DBusHandlerResult
change_ref_currency_filter (DBusConnection *connection, DBusMessage *message, void *user_data)
{
    int *type;
    
    CurrencyWidget *plug = (CurrencyWidget *)user_data;


    if ( dbus_message_is_signal(message, "com.binop.currencyexchangewidget.refcursignal.Signal", "Changed")) {
        DBusError error;
        
        dbus_error_init (&error);

        if ( dbus_message_get_args ( message, &error, DBUS_TYPE_INT32, type, DBUS_TYPE_INVALID)) {
            if ( plug->curset->type != (*type)) {
                pthread_mutex_lock(&plug->mutex);
                plug->curset->type = (*type);
                plug->curset->currency_changed = 1;
                pthread_mutex_unlock(&plug->mutex);
                g_timeout_add(100, as_update_values_once, plug);
            } 
               
        }
        else {
            fprintf(stderr, "Error receiving message");
            dbus_error_free(&error);
            exit(EXIT_FAILURE);
        }
        return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    }

    return DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
}




static void
currency_widget_plugin_class_init (CurrencyWidgetClass *class) {


    GObjectClass *object_class = GTK_OBJECT_CLASS(class);

    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(class);

    object_class->finalize = currency_widget_finalize;

    widget_class->realize = currency_widget_realize;
    widget_class->expose_event = currency_widget_expose_event;

} 

static void 
currency_widget_realize(GtkWidget *widget) {
 
    int error;

    
    CurrencyWidget *desktop_plugin = CURRENCY_WIDGET(widget);
   
    desktop_plugin->id = hd_home_plugin_item_get_applet_id (HD_HOME_PLUGIN_ITEM (widget));
   
    GdkScreen *screen = gtk_widget_get_screen(widget);
    gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
    gtk_widget_set_app_paintable(widget, TRUE);

    GTK_WIDGET_CLASS (currency_widget_plugin_parent_class)->realize (widget);

    load_config(&desktop_plugin);

    as_update_values_once(desktop_plugin);
    if ( desktop_plugin->curset->update_interval != -1 ) {
        desktop_plugin->timeout =  g_timeout_add(TO_HOUR(desktop_plugin->curset->update_interval), as_update_values, desktop_plugin);
    }

}

static gboolean
currency_widget_expose_event(GtkWidget *widget, GdkEventExpose *event) {


    CurrencyWidget *desktop_plugin = CURRENCY_WIDGET(widget);

    if (  pthread_mutex_lock(&desktop_plugin->mutex) != 0 ) {
        fprintf(stderr, "could not lock mutex");
    } 
   
    cairo_t *cr;
    cr = gdk_cairo_create(GDK_DRAWABLE(widget->window));
    gdk_cairo_region(cr, event->region);
    cairo_clip(cr); 
    GtkImage *bg; 
    bg = gtk_image_new_from_file(BACKGROUND_ICON);
    cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE); 
    gdk_cairo_set_source_pixbuf(cr, gtk_image_get_pixbuf(GTK_IMAGE(bg)),0,0);

    cairo_paint(cr);
    cairo_destroy(cr);
    g_object_ref_sink(G_OBJECT(bg));
    g_object_unref(G_OBJECT(bg));

    save_config(&desktop_plugin);

    if (  pthread_mutex_unlock(&desktop_plugin->mutex) != 0 ) {
        fprintf(stderr, "could not unlock mutex");
    } 


    return GTK_WIDGET_CLASS (currency_widget_plugin_parent_class)->expose_event (widget, event);
}

static void
currency_widget_finalize (GObject *object) {

    CurrencyWidget *desktop_plugin = CURRENCY_WIDGET(object);

    if ( desktop_plugin->timeout != - 1 ) {
        g_source_remove(desktop_plugin->timeout);
    }

    if (   pthread_mutex_lock(&desktop_plugin->mutex) != 0 ) {
        fprintf(stderr, "could not lock mutex");
    }

    dbus_connection_remove_filter(desktop_plugin->bus, change_ref_currency_filter,  desktop_plugin);
    dbus_bus_remove_match (desktop_plugin->bus, "type='signal', interface='com.binop.currencyexchangewidget.refcursignal.Signal'", NULL);
  //  dbus_connection_close(desktop_plugin->bus);
  //  dbus_connection_unref(desktop_plugin->bus);

    save_config(&desktop_plugin);
    dealloc_curset(&desktop_plugin->curset);
    free(desktop_plugin->id);
    
    if ( pthread_mutex_unlock(&desktop_plugin->mutex) != 0) {
        fprintf(stderr, "could not unlock mutex");
    }
    pthread_mutex_destroy(&desktop_plugin->mutex);
}

static void
currency_widget_plugin_class_finalize (CurrencyWidgetClass *class) {} 


void create_icon_factory() {
    int i;
    GtkIconFactory *factory = gtk_icon_factory_new();

    for ( i = 0; i < n_currencies; i++) { 
        GtkIconSource *source = gtk_icon_source_new();
        GtkIconSet *set = gtk_icon_set_new();
        char *tmp = get_icon_string(currencies[i]);

        gtk_icon_source_set_filename (source, get_icon_string(currencies[i]));



        gtk_icon_set_add_source(set, source);
        gtk_icon_factory_add (factory, currencies[i], set);

        free(tmp);
    }

    gtk_icon_factory_add_default(factory);
}

